import express from 'express';
import cors from 'cors';
import jwt from 'jsonwebtoken';
import bcrypt from 'bcryptjs';
import { CompanyModel } from '../src/models/company.js';
import { UserModel } from '../src/models/user.js';
import { ClientModel } from '../src/models/client.js';
import { EmployeeModel } from '../src/models/employee.js';
import { ServiceModel } from '../src/models/service.js';
import { OrderModel } from '../src/models/order.js';
import { EvidenceModel } from '../src/models/evidence.js';
import { ModuleModel } from '../src/models/module.js';
import { SystemConfigModel } from '../src/models/systemConfig.js';
import multer from 'multer';

const app = express();
const PORT = process.env.PORT || 8081;
const JWT_SECRET = process.env.JWT_SECRET || 'supersecretkey';

const allowedOrigins = [
    'https://keservicios.kemarketing.mx',
    'http://keservicios.kemarketing.mx',
    'http://localhost',
    'http://localhost:3000'
];

app.use(cors({
    origin: function (origin, callback) {
        // Allow requests with no origin (like mobile apps or curl)
        if (!origin) return callback(null, true);
        if (allowedOrigins.indexOf(origin) !== -1) {
            callback(null, true);
        } else {
            console.log('Origin not allowed:', origin);
            callback(new Error('Not allowed by CORS'));
        }
    },
    credentials: true
}));
app.use(express.json());

// Sirve archivos estáticos desde la carpeta 'uploads'
app.use('/uploads', express.static('uploads'));

// Configuración de Multer para la subida de archivos
const storage = multer.diskStorage({
    destination: (req, file, cb) => {
        cb(null, 'uploads/'); // Carpeta donde se guardarán los archivos subidos
    },
    filename: (req, file, cb) => {
        const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
        cb(null, file.fieldname + '-' + uniqueSuffix + '.' + file.originalname.split('.').pop());
    }
});

const upload = multer({
    storage: storage,
    limits: {
        fileSize: 5 * 1024 * 1024 // Límite de tamaño de archivo a 5MB
    },
    fileFilter: (req, file, cb) => {
        if (file.mimetype.startsWith('image/')) {
            cb(null, true);
        } else {
            cb(new Error('Solo se permiten archivos de imagen'), false);
        }
    }
});

// Middleware para el registro básico de solicitudes (logging)
app.use((req, res, next) => {
    console.log(`${new Date().toISOString()} - ${req.method} ${req.url}`);
    next();
});

// Middleware para verificar el token de autenticación
const verifyToken = (req, res, next) => {
    // Permitir rutas de inicio de sesión y rutas públicas
    if (req.path === '/api/auth/login' || req.path.startsWith('/public')) {
        return next();
    }

    const authHeader = req.headers['authorization'];
    const token = authHeader && authHeader.split(' ')[1];

    if (!token) {
        return res.status(401).json({ error: 'Token missing' });
    }

    jwt.verify(token, JWT_SECRET, (err, user) => {
        if (err) {
            if (err.name === 'TokenExpiredError') {
                return res.status(401).json({ error: 'Token expired' });
            }
            return res.status(403).json({ error: 'Invalid token' });
        }
        req.user = user;
        next();
    });
};

// Aplicar el middleware de autenticación a todas las rutas /api excepto el inicio de sesión (manejado dentro de verifyToken o por orden)
// En realidad, es mejor definir la ruta de inicio de sesión ANTES del middleware si usamos app.use('/api', verifyToken)
// Pero aquí simplemente añadiré la ruta de inicio de sesión primero, y luego usaré el middleware para el resto.


// Ruta de autenticación
app.post('/api/auth/login', async (req, res) => {
    const { email, password } = req.body;
    if (!email || !password) {
        return res.status(400).json({ error: 'Email and password required' });
    }
    try {
        const user = await UserModel.getByEmail(email);
        if (!user) {
            return res.status(401).json({ error: 'Invalid credentials' });
        }

        const match = await bcrypt.compare(password, user.password_hash);
        if (!match) {
            return res.status(401).json({ error: 'Invalid credentials' });
        }
        // Comprobar si la compañía está activa si el usuario tiene un company_id
        if (user.company_id) {
            const company = await CompanyModel.getById(user.company_id);
            if (company && !company.is_active) {
                return res.status(403).json({ error: 'Company is inactive' });
            }
        }

        const token = jwt.sign({
            id: user.id,
            email: user.email,
            role: user.role,
            company_id: user.company_id
        }, JWT_SECRET, { expiresIn: '1h' });
        res.json({ token, user: { id: user.id, email: user.email, role: user.role, company_id: user.company_id } });
    } catch (error) {
        console.error('Login error:', error);
        res.status(500).json({ error: 'Internal server error' });
    }
});// Ruta de autenticación de clientes
app.post('/api/auth/client-login', async (req, res) => {
    const { clientIdentifier, serviceFolio } = req.body;
    if (!clientIdentifier || !serviceFolio) {
        return res.status(400).json({ error: 'Client identifier and service folio required' });
    }
    try {
        // Buscar cliente por número de cliente o correo electrónico
        // Necesitamos consultar todos los clientes o añadir un método específico. 
        // Para eficiencia, asumamos que podemos filtrar en memoria o añadir un método.
        // Como ClientModel.getAll() existe, podemos usarlo por ahora (no es eficiente para una base de datos grande, pero funciona a esta escala).
        const clients = await ClientModel.getAll();
        const client = clients.find(c => c.client_number === clientIdentifier || c.email === clientIdentifier);

        if (!client) {
            return res.status(401).json({ error: 'Invalid credentials' });
        }

        // Comprobar si la compañía está activa
        const company = await CompanyModel.getById(client.company_id);
        if (company && !company.is_active) {
            return res.status(403).json({ error: 'Company is inactive' });
        }

        // Buscar orden
        const orders = await OrderModel.getAll();
        const order = orders.find(o => o.service_number === serviceFolio && o.client_id === client.user_id);

        if (!order) {
            return res.status(401).json({ error: 'Invalid credentials' });
        }

        // Generar token para el cliente
        const token = jwt.sign(
            { id: client.user_id, role: 'CLIENT', company_id: client.company_id },
            JWT_SECRET,
            { expiresIn: '8h' }
        );

        res.json({
            token,
            user: {
                id: client.user_id,
                role: 'CLIENT',
                company_id: client.company_id,
                name: client.name
            }
        });

    } catch (error) {
        console.error(error);
        res.status(500).json({ error: 'Server error' });
    }
});

// Proteger todas las demás rutas de la API
app.use('/api', verifyToken);

// Rutas de la API

// Compañías
app.get('/api/companies', async (req, res) => {
    try {
        const companies = await CompanyModel.getAll();
        res.json(companies);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

app.get('/api/companies/:id', async (req, res) => {
    try {
        const company = await CompanyModel.getById(req.params.id);
        if (!company) return res.status(404).json({ error: 'Company not found' });
        res.json(company);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

app.post('/api/companies', async (req, res) => {
    try {
        const company = await CompanyModel.create(req.body);
        res.status(201).json(company);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

app.put('/api/companies/:id', async (req, res) => {
    try {
        const company = await CompanyModel.update(req.params.id, req.body);
        res.json(company);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

app.delete('/api/companies/:id', async (req, res) => {
    try {
        await CompanyModel.delete(req.params.id);
        res.json({ message: 'Company deleted' });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

// Usuarios
app.get('/api/users', async (req, res) => {
    try {
        const users = await UserModel.getAll();
        res.json(users);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

app.post('/api/users', async (req, res) => {
    try {
        const user = await UserModel.create(req.body);
        res.status(201).json(user);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

app.put('/api/users/:id', async (req, res) => {
    try {
        const data = { ...req.body };
        if (data.password) {
            const salt = await bcrypt.genSalt(10);
            data.password_hash = await bcrypt.hash(data.password, salt);
        }
        delete data.password;
        const user = await UserModel.update(req.params.id, data);
        res.json(user);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

app.delete('/api/users/:id', async (req, res) => {
    try {
        await UserModel.delete(req.params.id);
        res.json({ message: 'User deleted' });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

// Clientes
app.get('/api/clients', async (req, res) => {
    try {
        let clients;
        if (req.user.role === 'ADMIN' || req.user.role === 'ROOT') {
            clients = await ClientModel.getAll();
        } else if (req.user.company_id) {
            clients = await ClientModel.getByCompanyId(req.user.company_id);
        } else {
            return res.status(403).json({ error: 'Acceso no autorizado o empresa no especificada.' });
        }
        res.json(clients);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

app.post('/api/clients', async (req, res) => {
    try {
        const { email, password, company_id, ...clientInfo } = req.body;
        // Create user for client
        const salt = await bcrypt.genSalt(10);
        const password_hash = await bcrypt.hash(password || 'demo123', salt);
        const userData = {
            id: `client-${Date.now()}`,
            email,
            password_hash,
            role: 'CLIENT',
            company_id
        };
        await UserModel.create(userData);
        // Create client record linked to user
        const clientData = { ...clientInfo, user_id: userData.id };
        const client = await ClientModel.create(clientData);
        res.status(201).json(client);
    } catch (err) {
        console.error(err);
        res.status(500).json({ error: err.message });
    }
});
app.put('/api/clients/:id', async (req, res) => {
    try {
        const data = { ...req.body };
        const userUpdates = {};

        if (data.password) {
            const salt = await bcrypt.genSalt(10);
            userUpdates.password_hash = await bcrypt.hash(data.password, salt);
        }

        if (data.email) {
            userUpdates.email = data.email;
        }

        if (typeof data.is_active === 'boolean') {
            userUpdates.is_active = data.is_active;
        }

        // Clean up data for ClientModel update
        delete data.password;
        delete data.email;
        delete data.is_active;

        // Apply user updates if any
        if (Object.keys(userUpdates).length > 0) {
            await UserModel.update(req.params.id, userUpdates);
        }

        const client = await ClientModel.update(req.params.id, data);
        res.json(client);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

app.delete('/api/clients/:id', async (req, res) => {
    try {
        await ClientModel.delete(req.params.id);
        res.json({ message: 'Client deleted' });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

// Employees
app.get('/api/employees', async (req, res) => {
    try {
        const employees = await EmployeeModel.getAll();
        res.json(employees);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

app.post('/api/employees', async (req, res) => {
    try {
        // Create user first
        const userData = {
            id: req.body.user_id || `emp-${Date.now()}`,
            email: req.body.email,
            password: req.body.password || 'demo123',
            role: 'EMPLOYEE',
            company_id: req.body.company_id,
            is_active: req.body.is_active
        };

        // Hash password
        const salt = await bcrypt.genSalt(10);
        userData.password_hash = await bcrypt.hash(userData.password, salt);
        delete userData.password;

        await UserModel.create(userData);

        // Create employee
        const employeeData = { ...req.body, user_id: userData.id };
        delete employeeData.password; // Remove password from employee data

        const employee = await EmployeeModel.create(employeeData);
        res.status(201).json(employee);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

app.put('/api/employees/:id', async (req, res) => {
    try {
        const data = { ...req.body };
        const userUpdates = {};

        // Update user password if provided
        if (data.password) {
            const salt = await bcrypt.genSalt(10);
            userUpdates.password_hash = await bcrypt.hash(data.password, salt);
        }
        delete data.password;

        // Actualizar correo electrónico del usuario si se proporciona

        // Apply user updates if any
        if (Object.keys(userUpdates).length > 0) {
            await UserModel.update(req.params.id, userUpdates);
        }

        const employee = await EmployeeModel.update(req.params.id, data);
        res.json(employee);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

app.get('/api/employees/:employeeId/services', async (req, res) => {
    try {
        const services = await OrderModel.getByEmployeeId(req.params.employeeId);
        res.json(services);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

app.delete('/api/employees/:id', async (req, res) => {
    try {
        await EmployeeModel.delete(req.params.id);
        // ¿Eliminar usuario opcionalmente? Por ahora, mantengámoslo simple.
        res.json({ message: 'Employee deleted' });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

// Servicios
app.get('/api/services', async (req, res) => {
    try {
        let services;
        if (req.user.role === 'ADMIN' || req.user.role === 'ROOT') {
            services = await ServiceModel.getAll();
        } else if (req.user.company_id) {
            services = await ServiceModel.getByCompanyId(req.user.company_id);
        } else {
            return res.status(403).json({ error: 'Acceso no autorizado o empresa no especificada.' });
        }
        res.json(services);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

app.get('/api/companies/:companyId/services', async (req, res) => {
    try {
        const services = await ServiceModel.getByCompanyId(req.params.companyId);
        res.json(services);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

app.post('/api/services', async (req, res) => {
    try {
        const service = await ServiceModel.create(req.body);
        res.status(201).json(service);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

app.put('/api/services/:id', async (req, res) => {
    try {
        const service = await ServiceModel.update(req.params.id, req.body);
        res.json(service);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

app.delete('/api/services/:id', async (req, res) => {
    try {
        await ServiceModel.delete(req.params.id);
        res.json({ message: 'Service deleted' });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

// Órdenes
app.get('/api/orders', async (req, res) => {
    try {
        let orders;
        if (req.user.role === 'ADMIN' || req.user.role === 'ROOT') {
            orders = await OrderModel.getAll();
        } else if (req.user.company_id) {
            orders = await OrderModel.getByCompanyId(req.user.company_id);
        } else {
            return res.status(403).json({ error: 'Acceso no autorizado o empresa no especificada.' });
        }
        res.json(orders);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

app.post('/api/orders', async (req, res) => {
    try {
        const order = await OrderModel.create(req.body);
        res.status(201).json(order);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

app.put('/api/orders/:id', async (req, res) => {
    try {
        // Obtener el orden actual para validar su estado
        const currentOrder = await OrderModel.getById(req.params.id);

        // Estados finales en los que no se permite hacer cambios
        const finalStates = ['CERRADO_ADMIN', 'CANCELLED'];

        // Si el servicio está en estado final y se intenta cambiar el estado, rechazar
        if (finalStates.includes(currentOrder.status) && req.body.status && req.body.status !== currentOrder.status) {
            return res.status(400).json({
                error: 'Este servicio está finalizado y cerrado. No se pueden realizar cambios en su estado.'
            });
        }

        // Validar que solo administradores puedan cambiar a CANCELLED
        if (req.body.status === 'CANCELLED' && currentOrder.status !== 'CANCELLED') {
            // Extraer el rol del token JWT
            const token = req.headers['authorization']?.split(' ')[1];
            if (token) {
                try {
                    const decoded = jwt.verify(token, JWT_SECRET);
                    if (decoded.role !== 'admin') {
                        return res.status(403).json({
                            error: 'Solo los administradores pueden cancelar servicios.'
                        });
                    }
                } catch (err) {
                    return res.status(401).json({ error: 'Token inválido' });
                }
            } else {
                return res.status(401).json({ error: 'Token no proporcionado' });
            }
        }

        const order = await OrderModel.update(req.params.id, req.body);
        res.json(order);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

// Configuración del sistema
app.get('/api/config', async (req, res) => {
    try {
        const config = await SystemConfigModel.getAll();
        res.json(config);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

app.get('/api/config/:key', async (req, res) => {
    try {
        const config = await SystemConfigModel.getByKey(req.params.key);
        res.json(config);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

app.put('/api/config/:key', async (req, res) => {
    try {
        const { config_value, description } = req.body;
        const config = await SystemConfigModel.set(req.params.key, config_value, description);
        res.json(config);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

// Evidencias
app.get('/api/services/:serviceId/evidences', async (req, res) => {
    try {
        const evidences = await EvidenceModel.getByServiceId(req.params.serviceId);
        res.json(evidences);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

app.post('/api/evidences', upload.array('evidences', 4), async (req, res) => {
    try {
        const { contracted_service_id, evidence_type, descriptions } = req.body;

        if (!contracted_service_id) {
            return res.status(400).json({ error: "contracted_service_id is required" });
        }

        const evidences = [];
        if (req.files && req.files.length > 0) {
            for (let i = 0; i < req.files.length; i++) {
                const file = req.files[i];
                const evidenceData = {
                    contracted_service_id: contracted_service_id,
                    evidence_type: evidence_type || 'PHOTO',
                    file_url: `/uploads/${file.filename}`,
                    description: descriptions ? descriptions[i] : `Evidencia ${i + 1}`
                };
                const evidence = await EvidenceModel.create(evidenceData);
                evidences.push(evidence);
            }
        }
        res.status(201).json(evidences);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

app.delete('/api/evidences/:id', async (req, res) => {
    try {
        await EvidenceModel.delete(req.params.id);
        res.json({ message: 'Evidence deleted' });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

// Módulos
app.get('/api/modules', async (req, res) => {
    try {
        const modules = await ModuleModel.getAll();
        res.json(modules);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

app.get('/api/companies/:companyId/modules', async (req, res) => {
    try {
        const modules = await ModuleModel.getByCompanyId(req.params.companyId);
        res.json(modules);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

app.put('/api/companies/:companyId/modules/:moduleId', async (req, res) => {
    try {
        const { is_enabled } = req.body;
        const result = await ModuleModel.toggleModule(req.params.companyId, req.params.moduleId, is_enabled);
        res.json(result);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

// Iniciar servidor
app.listen(PORT, '0.0.0.0', () => {
    console.log(`Server running on http://0.0.0.0:${PORT}`);
});
