import { Box, Flex, HStack, Heading, Text, Button, Icon } from '@chakra-ui/react';
import { useState, useEffect } from 'react';
import { useLocation } from 'react-router-dom';
import { useBreakpointValue } from '@chakra-ui/react';
import { FiMenu, FiBell } from 'react-icons/fi';
import Sidebar from './Sidebar';

const Layout = ({ links, userType, userName, userPhoto, children }) => {
  const isMobile = useBreakpointValue({ base: true, md: false });
  const [isSidebarOpen, setIsSidebarOpen] = useState(true);

  useEffect(() => {
    if (isMobile) {
      setIsSidebarOpen(false);
    } else {
      setIsSidebarOpen(true);
    }
  }, [isMobile]);

  const toggleSidebar = () => {
    setIsSidebarOpen(!isSidebarOpen);
  };

  const handleSidebarClose = () => {
    if (isMobile) {
      setIsSidebarOpen(false);
    }
  };

  // Determinar título de la página basado en la ruta
  const getPageTitle = () => {
    const currentLink = links.find(link => link.path === location.pathname);
    return currentLink?.nombre || 'Dashboard';
  };

  return (
    <Flex>
      <Sidebar
        links={links}
        userType={userType}
        userName={userName}
        userPhoto={userPhoto}
        isOpen={isSidebarOpen}
        onClose={handleSidebarClose}
      />
      <Box
        ml={isSidebarOpen && !isMobile ? "260px" : "0"}
        w={isSidebarOpen && !isMobile ? "calc(100% - 260px)" : "100%"}
        minH="100vh"
        bg="#F8FAFC"
        transition="all 0.3s ease-in-out"
      >
        {/* Topbar */}
        <Box
          borderBottom="1px solid"
          borderColor="gray.200"
          px={8}
          py={6}
          position="sticky"
          top="0"
          zIndex={100}
          backdropFilter="blur(10px)"
          bg="rgba(255, 255, 255, 0.95)"
          boxShadow="0 1px 3px 0 rgba(0, 0, 0, 0.1), 0 1px 2px 0 rgba(0, 0, 0, 0.06)"
          transition="all 0.2s ease-in-out"
        >
          <Flex justify="space-between" align="center">
            <HStack spacing={4}>
              <Button
                onClick={toggleSidebar}
                variant="ghost"
                size="md"
                fontSize="24px"
                aria-label="Toggle Sidebar"
              >
                <Icon as={FiMenu} />
              </Button>
              <Heading
                size="xl"
                color="gray.900"
                fontWeight="700"
                letterSpacing="-0.5px"
                fontSize="24px"
              >
                {getPageTitle()}
              </Heading>
            </HStack>
            <HStack spacing={3}>
              <Button
                variant="ghost"
                size="md"
                w="44px"
                h="44px"
                borderRadius="10px"
                fontSize="20px"
                _hover={{
                  bg: 'gray.100',
                  transform: 'scale(1.05)',
                }}
                position="relative"
                transition="all 0.2s ease-in-out"
              >
                <Icon as={FiBell} />
                <Box
                  position="absolute"
                  top="10px"
                  right="10px"
                  w="8px"
                  h="8px"
                  bg="red.500"
                  borderRadius="full"
                  border="2px solid white"
                  boxShadow="0 0 0 2px rgba(239, 68, 68, 0.2)"
                  animation="pulse 2s cubic-bezier(0.4, 0, 0.6, 1) infinite"
                />
              </Button>
            </HStack>
          </Flex>
        </Box>

        {/* Content */}
        <Box p={8} pb={0} minH="calc(100vh - 80px)" display="flex" flexDirection="column">
          <Box flex="1">
            {children}
          </Box>

          {/* Footer */}
          <Box
            mt={8}
            py={4}
            borderTop="1px solid"
            borderColor="gray.200"
            textAlign="center"
          >
            <Text fontSize="xs" color="gray.500">
              © {new Date().getFullYear()} Desarrollado por{' '}
              <Text as="span" fontWeight="600" color="brand.600">
                KeSoftware
              </Text>
              {' '}• Todos los derechos reservados
            </Text>
          </Box>
        </Box>
      </Box>
    </Flex>
  );
};

export default Layout;

