import { useState } from 'react';
import { useNavigate } from 'react-router-dom';
import {
  Box,
  Flex,
  Heading,
  FormControl,
  FormLabel,
  Input,
  Button,
  VStack,
  Text,
  Image,
  useToast,
  InputGroup,
  InputLeftElement,
  Icon,
  Link,
  HStack
} from '@chakra-ui/react';
import { FiUser, FiFileText, FiArrowRight } from 'react-icons/fi';
import { clientLogin } from '../services/api';

const LoginPage = () => {
  const [clientIdentifier, setClientIdentifier] = useState('');
  const [serviceFolio, setServiceFolio] = useState('');
  const [isLoading, setIsLoading] = useState(false);
  const navigate = useNavigate();
  const toast = useToast();

  const handleLogin = async (e) => {
    e.preventDefault();

    if (!clientIdentifier || !serviceFolio) {
      toast({
        title: 'Error',
        description: 'Por favor completa todos los campos',
        status: 'error',
        duration: 3000,
        isClosable: true,
      });
      return;
    }

    setIsLoading(true);

    try {
      const { token, user } = await clientLogin(clientIdentifier, serviceFolio);

      // Login exitoso
      toast({
        title: 'Bienvenido',
        description: `Acceso concedido. Bienvenido ${user.name}`,
        status: 'success',
        duration: 2000,
        isClosable: true,
      });

      // Guardar sesión
      sessionStorage.setItem('token', token);
      sessionStorage.setItem('user', JSON.stringify(user));

      navigate('/client');
    } catch (error) {
      console.error(error);
      toast({
        title: 'Acceso denegado',
        description: error.message || 'Credenciales incorrectas',
        status: 'error',
        duration: 3000,
        isClosable: true,
      });
    } finally {
      setIsLoading(false);
    }
  };

  return (
    <Flex minH="100vh" overflow="hidden">
      {/* Left Side - Construction/Maintenance Image & Branding */}
      <Box
        w={{ base: '0%', lg: '50%' }}
        bg="gray.900"
        position="relative"
        display={{ base: 'none', lg: 'block' }}
      >
        <Image
          src="/construction_maintenance_bg_1765306718080.png"
          alt="Servicios de Mantenimiento y Construcción"
          w="100%"
          h="100%"
          objectFit="cover"
          opacity={0.7}
        />
        <Box
          position="absolute"
          top="0"
          left="0"
          w="100%"
          h="100%"
          bgGradient="linear(to-b, rgba(0,0,0,0.4), rgba(0,0,0,0.8))"
          display="flex"
          flexDirection="column"
          justifyContent="flex-end"
          p={16}
        >
          <VStack align="flex-start" spacing={6} maxW="lg">
            <Box
              bg="white"
              p={2}
              borderRadius="xl"
              boxShadow="lg"
              w="80px"
              h="80px"
              display="flex"
              alignItems="center"
              justifyContent="center"
              overflow="hidden"
            >
              <Image src="/logo.jpeg" alt="Logo" objectFit="cover" w="100%" h="100%" />
            </Box>
            <Heading color="white" fontSize="5xl" fontWeight="bold" lineHeight="1.1">
              Seguimiento de <br /> tus servicios
            </Heading>
            <Text color="gray.300" fontSize="xl" maxW="md">
              Consulta el estado de tus servicios en tiempo real, revisa evidencias y califica la atención recibida.
            </Text>
            <HStack spacing={4} pt={4}>
              <Text color="whiteAlpha.800" fontSize="sm" fontWeight="600">
                © 2025 KeSoftware
              </Text>
              <Box w="1px" h="16px" bg="whiteAlpha.400" />
              <Link color="whiteAlpha.800" fontSize="sm" _hover={{ color: 'white' }}>
                Privacidad
              </Link>
              <Link color="whiteAlpha.800" fontSize="sm" _hover={{ color: 'white' }}>
                Términos
              </Link>
            </HStack>
          </VStack>
        </Box>
      </Box>

      {/* Right Side - Login Form */}
      <Flex
        w={{ base: '100%', lg: '50%' }}
        bg="white"
        align="center"
        justify="center"
        p={8}
      >
        <Box w="100%" maxW="md" mt={{ base: 0, md: -16 }}>
          <VStack spacing={8} align="stretch">
            <Box textAlign="center" mb={8}>
              <Image
                src="/logop.jpg"
                alt="Logo"
                mx="auto"
                mb={8}
                h="160px"
                objectFit="contain"
              />
              <Heading fontSize="3xl" fontWeight="bold" color="gray.900">
                Portal de Clientes
              </Heading>
              <Text color="gray.500" mt={2}>
                Ingresa tu número de cliente y folio de servicio
              </Text>
            </Box>

            <Box as="form" onSubmit={handleLogin}>
              <VStack spacing={5}>
                <FormControl>
                  <FormLabel fontWeight="600" color="gray.700">Número de Cliente o Email</FormLabel>
                  <InputGroup>
                    <InputLeftElement pointerEvents="none">
                      <Icon as={FiUser} color="gray.400" />
                    </InputLeftElement>
                    <Input
                      type="text"
                      value={clientIdentifier}
                      onChange={(e) => setClientIdentifier(e.target.value)}
                      placeholder="CLI-2024-001 o email@ejemplo.com"
                      size="lg"
                      borderRadius="lg"
                      bg="gray.50"
                      border="1px solid"
                      borderColor="gray.200"
                      _focus={{ borderColor: 'brand.500', bg: 'white', boxShadow: 'none' }}
                    />
                  </InputGroup>
                </FormControl>

                <FormControl>
                  <FormLabel fontWeight="600" color="gray.700">Folio de Servicio</FormLabel>
                  <InputGroup>
                    <InputLeftElement pointerEvents="none">
                      <Icon as={FiFileText} color="gray.400" />
                    </InputLeftElement>
                    <Input
                      type="text"
                      value={serviceFolio}
                      onChange={(e) => setServiceFolio(e.target.value)}
                      placeholder="SRV-2024-001"
                      size="lg"
                      borderRadius="lg"
                      bg="gray.50"
                      border="1px solid"
                      borderColor="gray.200"
                      _focus={{ borderColor: 'brand.500', bg: 'white', boxShadow: 'none' }}
                    />
                  </InputGroup>
                </FormControl>

                <Button
                  type="submit"
                  colorScheme="brand"
                  size="lg"
                  w="100%"
                  h="56px"
                  fontSize="md"
                  borderRadius="lg"
                  rightIcon={<FiArrowRight />}
                  _hover={{ transform: 'translateY(-1px)', boxShadow: 'lg' }}
                  transition="all 0.2s"
                  isLoading={isLoading}
                  loadingText="Verificando..."
                >
                  Consultar Servicio
                </Button>
              </VStack>
            </Box>

            <Box textAlign="center">
              <Text fontSize="sm" color="gray.600" mb={2}>
                ¿Eres personal de la empresa?
              </Text>
              <Link
                href="/kepanel"
                color="brand.500"
                fontSize="sm"
                fontWeight="600"
                _hover={{ textDecoration: 'underline' }}
              >
                Acceder al panel administrativo →
              </Link>
            </Box>

            <Box
              p={5}
              bg="blue.50"
              borderRadius="xl"
              border="1px dashed"
              borderColor="blue.200"
            >
              <Text fontSize="sm" color="blue.800" fontWeight="600" mb={2}>
                💡 Credenciales de Demo:
              </Text>
              <VStack align="stretch" spacing={2}>
                <HStack>
                  <Text fontSize="xs" color="blue.700" fontWeight="500">Cliente:</Text>
                  <Text fontSize="xs" color="blue.600">CLI-2024-001</Text>
                </HStack>
                <HStack>
                  <Text fontSize="xs" color="blue.700" fontWeight="500">Folio:</Text>
                  <Text fontSize="xs" color="blue.600">SRV-2024-001</Text>
                </HStack>
              </VStack>
            </Box>
          </VStack>
        </Box>
      </Flex>

      {/* Footer */}
      <Box
        position="absolute"
        bottom="0"
        left="0"
        right="0"
        py={4}
        bg="rgba(255, 255, 255, 0.8)"
        backdropFilter="blur(10px)"
        borderTop="1px solid"
        borderColor="gray.200"
      >
        <Text textAlign="center" fontSize="xs" color="gray.600">
          © {new Date().getFullYear()} Desarrollado por{' '}
          <Text as="span" fontWeight="700" color="brand.600">
            KeSoftware
          </Text>
          {' '}• Todos los derechos reservados
        </Text>
      </Box>
    </Flex>
  );
};

export default LoginPage;

