import { useState } from 'react';
import { useNavigate } from 'react-router-dom';
import {
    Box,
    Flex,
    Heading,
    FormControl,
    FormLabel,
    Input,
    Button,
    VStack,
    Text,
    Image,
    useToast,
    InputGroup,
    InputLeftElement,
    Icon,
    Checkbox,
    Link,
    HStack
} from '@chakra-ui/react';
import { FiMail, FiLock, FiArrowRight } from 'react-icons/fi';
import { login } from '../services/api';

const LoginPanel = () => {
    const [email, setEmail] = useState('');
    const [password, setPassword] = useState('');
    const [rememberMe, setRememberMe] = useState(true);
    const [isLoading, setIsLoading] = useState(false);
    const navigate = useNavigate();
    const toast = useToast();

    const handleLogin = async (e) => {
        e.preventDefault();

        if (!email || !password) {
            toast({
                title: 'Error',
                description: 'Por favor completa todos los campos',
                status: 'error',
                duration: 3000,
                isClosable: true,
            });
            return;
        }

        setIsLoading(true);

        try {
            const { token, user } = await login(email, password);

            // Verificar que sea personal (ROOT, ADMIN, EMPLOYEE)
            if (user.role === 'CLIENT') {
                toast({
                    title: 'Acceso denegado',
                    description: 'Esta área es solo para personal. Los clientes deben acceder desde la página principal.',
                    status: 'warning',
                    duration: 4000,
                    isClosable: true,
                });
                setIsLoading(false);
                return;
            }

            // Login exitoso
            let userName = user.name || user.role;

            toast({
                title: 'Bienvenido',
                description: `Acceso concedido. Bienvenido ${userName}`,
                status: 'success',
                duration: 2000,
                isClosable: true,
            });

            // Guardar sesión
            sessionStorage.setItem('token', token);
            sessionStorage.setItem('user', JSON.stringify(user));

            // Redirigir según rol
            if (user.role === 'ROOT') {
                navigate('/root');
            } else if (user.role === 'ADMIN') {
                navigate('/admin');
            } else if (user.role === 'EMPLOYEE') {
                navigate('/employee');
            }
        } catch (error) {
            console.error(error);
            toast({
                title: 'Acceso denegado',
                description: error.message || 'Credenciales incorrectas',
                status: 'error',
                duration: 3000,
                isClosable: true,
            });
        } finally {
            setIsLoading(false);
        }
    };

    return (
        <Flex minH="100vh" overflow="hidden">
            {/* Left Side - Professional Background */}
            <Box
                w={{ base: '0%', lg: '50%' }}
                bg="gray.900"
                position="relative"
                display={{ base: 'none', lg: 'block' }}
            >
                <Image
                    src="/construction_maintenance_bg_1765306718080.png"
                    alt="Panel Administrativo"
                    w="100%"
                    h="100%"
                    objectFit="cover"
                    opacity={0.5}
                />
                <Box
                    position="absolute"
                    top="0"
                    left="0"
                    w="100%"
                    h="100%"
                    bgGradient="linear(to-b, rgba(0,0,0,0.6), rgba(0,0,0,0.9))"
                    display="flex"
                    flexDirection="column"
                    justifyContent="flex-end"
                    p={16}
                >
                    <VStack align="flex-start" spacing={6} maxW="lg">
                        <Box
                            bg="white"
                            p={2}
                            borderRadius="xl"
                            boxShadow="lg"
                            w="80px"
                            h="80px"
                            display="flex"
                            alignItems="center"
                            justifyContent="center"
                            overflow="hidden"
                        >
                            <Image src="/logo.jpeg" alt="Logo" objectFit="cover" w="100%" h="100%" />
                        </Box>
                        <Heading color="white" fontSize="5xl" fontWeight="bold" lineHeight="1.1">
                            Panel de <br /> Administración
                        </Heading>
                        <Text color="gray.300" fontSize="xl" maxW="md">
                            Sistema de gestión integral para administradores, empleados y super usuarios.
                        </Text>
                        <HStack spacing={4} pt={4}>
                            <Text color="whiteAlpha.800" fontSize="sm" fontWeight="600">
                                © 2025 KeSoftware
                            </Text>
                            <Box w="1px" h="16px" bg="whiteAlpha.400" />
                            <Link color="whiteAlpha.800" fontSize="sm" _hover={{ color: 'white' }}>
                                Privacidad
                            </Link>
                            <Link color="whiteAlpha.800" fontSize="sm" _hover={{ color: 'white' }}>
                                Términos
                            </Link>
                        </HStack>
                    </VStack>
                </Box>
            </Box>

            {/* Right Side - Login Form */}
            <Flex
                w={{ base: '100%', lg: '50%' }}
                bg="white"
                align="center"
                justify="center"
                p={8}
            >
                <Box w="100%" maxW="md" mt={{ base: 0, md: -16 }}>
                    <VStack spacing={8} align="stretch">
                        <Box textAlign="center" mb={8}>
                            <Image
                                src="/logop.jpg"
                                alt="Logo"
                                mx="auto"
                                mb={8}
                                h="160px"
                                objectFit="contain"
                            />
                            <Heading fontSize="3xl" fontWeight="bold" color="gray.900">
                                Acceso al Sistema
                            </Heading>
                            <Text color="gray.500" mt={2}>
                                Ingresa tus credenciales de personal
                            </Text>
                        </Box>

                        <Box as="form" onSubmit={handleLogin}>
                            <VStack spacing={5}>
                                <FormControl>
                                    <FormLabel fontWeight="600" color="gray.700">Correo Electrónico</FormLabel>
                                    <InputGroup>
                                        <InputLeftElement pointerEvents="none">
                                            <Icon as={FiMail} color="gray.400" />
                                        </InputLeftElement>
                                        <Input
                                            type="email"
                                            value={email}
                                            onChange={(e) => setEmail(e.target.value)}
                                            placeholder="tu@email.com"
                                            size="lg"
                                            borderRadius="lg"
                                            bg="gray.50"
                                            border="1px solid"
                                            borderColor="gray.200"
                                            _focus={{ borderColor: 'brand.500', bg: 'white', boxShadow: 'none' }}
                                        />
                                    </InputGroup>
                                </FormControl>

                                <FormControl>
                                    <FormLabel fontWeight="600" color="gray.700">Contraseña</FormLabel>
                                    <InputGroup>
                                        <InputLeftElement pointerEvents="none">
                                            <Icon as={FiLock} color="gray.400" />
                                        </InputLeftElement>
                                        <Input
                                            type="password"
                                            value={password}
                                            onChange={(e) => setPassword(e.target.value)}
                                            placeholder="••••••••"
                                            size="lg"
                                            borderRadius="lg"
                                            bg="gray.50"
                                            border="1px solid"
                                            borderColor="gray.200"
                                            _focus={{ borderColor: 'brand.500', bg: 'white', boxShadow: 'none' }}
                                        />
                                    </InputGroup>
                                </FormControl>

                                <HStack justify="space-between" w="100%">
                                    <Checkbox
                                        colorScheme="brand"
                                        isChecked={rememberMe}
                                        onChange={(e) => setRememberMe(e.target.checked)}
                                    >
                                        Recordarme
                                    </Checkbox>
                                    <Link color="brand.500" fontSize="sm" fontWeight="600">
                                        ¿Olvidaste tu contraseña?
                                    </Link>
                                </HStack>

                                <Button
                                    type="submit"
                                    colorScheme="brand"
                                    size="lg"
                                    w="100%"
                                    h="56px"
                                    fontSize="md"
                                    borderRadius="lg"
                                    rightIcon={<FiArrowRight />}
                                    _hover={{ transform: 'translateY(-1px)', boxShadow: 'lg' }}
                                    transition="all 0.2s"
                                    isLoading={isLoading}
                                    loadingText="Iniciando..."
                                >
                                    Iniciar Sesión
                                </Button>
                            </VStack>
                        </Box>

                        <Box textAlign="center">
                            <Text fontSize="sm" color="gray.600" mb={2}>
                                ¿Eres cliente?
                            </Text>
                            <Link
                                href="/"
                                color="brand.500"
                                fontSize="sm"
                                fontWeight="600"
                                _hover={{ textDecoration: 'underline' }}
                            >
                                ← Volver al portal de clientes
                            </Link>
                        </Box>

                        <Box
                            p={5}
                            bg="blue.50"
                            borderRadius="xl"
                            border="1px dashed"
                            borderColor="blue.200"
                        >
                            <Text fontSize="sm" color="blue.800" fontWeight="600" mb={3}>
                                💡 Credenciales de Demo:
                            </Text>
                            <VStack align="stretch" spacing={2}>
                                <Box>
                                    <Text fontSize="xs" color="blue.700" fontWeight="600" mb={1}>ROOT:</Text>
                                    <Text fontSize="xs" color="blue.600">root@kesoftware.com / demo123</Text>
                                </Box>
                                <Box>
                                    <Text fontSize="xs" color="blue.700" fontWeight="600" mb={1}>ADMIN:</Text>
                                    <Text fontSize="xs" color="blue.600">admin@serviciosnorte.com / demo123</Text>
                                </Box>
                                <Box>
                                    <Text fontSize="xs" color="blue.700" fontWeight="600" mb={1}>EMPLEADO:</Text>
                                    <Text fontSize="xs" color="blue.600">juan.perez@serviciosnorte.com / demo123</Text>
                                </Box>
                            </VStack>
                        </Box>
                    </VStack>
                </Box>
            </Flex>

            {/* Footer */}
            <Box
                position="absolute"
                bottom="0"
                left="0"
                right="0"
                py={4}
                bg="rgba(255, 255, 255, 0.8)"
                backdropFilter="blur(10px)"
                borderTop="1px solid"
                borderColor="gray.200"
            >
                <Text textAlign="center" fontSize="xs" color="gray.600">
                    © {new Date().getFullYear()} Desarrollado por{' '}
                    <Text as="span" fontWeight="700" color="brand.600">
                        KeSoftware
                    </Text>
                    {' '}• Todos los derechos reservados
                </Text>
            </Box>
        </Flex>
    );
};

export default LoginPanel;
