import { useState, useEffect } from 'react';
import {
  Box,
  SimpleGrid,
  Stat,
  StatLabel,
  StatNumber,
  StatHelpText,
  StatArrow,
  Card,
  CardHeader,
  CardBody,
  Heading,
  Text,
  Icon,
  Flex,
  Table,
  Thead,
  Tbody,
  Tr,
  Th,
  Td,
  Badge
} from '@chakra-ui/react';
import { FiUsers, FiClipboard, FiCheckCircle, FiClock, FiAlertCircle } from 'react-icons/fi';
import { getOrders, getClients } from '../../services/api';

// Función para traducir estados a leyendas amigables
const getStatusLabel = (status) => {
  const statusMap = {
    'PENDING': 'Pendiente',
    'IN_PROGRESS': 'En Progreso',
    'FINISHED': 'Finalizado',
    'CANCELLED': 'Cancelado',
    'CERRADO_ADMIN': 'Cerrado',
    'IN_REVIEW': 'Reapertura'
  };
  return statusMap[status] || status;
};

const AdminDashboard = () => {
  const [orders, setOrders] = useState([]);
  const [clients, setClients] = useState([]);

  useEffect(() => {
    const fetchDashboardData = async () => {
      try {
        const user = JSON.parse(sessionStorage.getItem('user'));
        const companyId = user ? user.company_id : null;

        if (companyId) {
          const [allOrders, allClients] = await Promise.all([
            getOrders(),
            getClients()
          ]);

          // Filter by company
          setOrders(allOrders.filter(o => o.company_id === companyId));
          setClients(allClients.filter(c => c.company_id === companyId));
        }
      } catch (error) {
        console.error("Error fetching dashboard data:", error);
      }
    };

    fetchDashboardData();
  }, []);

  // Estadísticas reales
  const currentYear = new Date().getFullYear();
  const yearlyOrders = orders.filter(o => new Date(o.request_date).getFullYear() === currentYear);

  const activeServices = yearlyOrders.filter(s => s.status === 'IN_PROGRESS' || s.status === 'IN_REVIEW').length;
  const pendingServices = yearlyOrders.filter(s => s.status === 'PENDING').length;
  const finishedServices = yearlyOrders.filter(s => s.status === 'FINISHED' || s.status === 'CERRADO_ADMIN').length;
  const totalClients = clients.length;

  const statusOrder = { 'IN_REVIEW': 1, 'PENDING': 2, 'IN_PROGRESS': 3, 'FINISHED': 4, 'CERRADO_ADMIN': 5, 'CANCELLED': 6 };

  const sortedOrders = [...yearlyOrders].sort((a, b) => {
    const orderA = statusOrder[a.status] || 99;
    const orderB = statusOrder[b.status] || 99;
    if (orderA !== orderB) {
      return orderA - orderB;
    }
    return new Date(b.request_date) - new Date(a.request_date);
  });

  return (
    <Box>
      <Heading mb={6}>Panel Operativo</Heading>

      {/* Stats Grid */}
      <SimpleGrid columns={{ base: 1, md: 2, lg: 4 }} spacing={6} mb={8}>
        <Card>
          <CardBody>
            <Stat>
              <Flex alignItems="center" mb={2}>
                <Box p={2} bg="blue.100" borderRadius="md" mr={3}>
                  <Icon as={FiClipboard} color="blue.500" boxSize={6} />
                </Box>
                <StatLabel fontSize="lg">Servicios Activos</StatLabel>
              </Flex>
              <StatNumber>{activeServices}</StatNumber>
              <StatHelpText>
                En progreso actualmente
              </StatHelpText>
            </Stat>
          </CardBody>
        </Card>

        <Card>
          <CardBody>
            <Stat>
              <Flex alignItems="center" mb={2}>
                <Box p={2} bg="orange.100" borderRadius="md" mr={3}>
                  <Icon as={FiClock} color="orange.500" boxSize={6} />
                </Box>
                <StatLabel fontSize="lg">Pendientes</StatLabel>
              </Flex>
              <StatNumber>{pendingServices}</StatNumber>
              <StatHelpText>
                Por asignar o iniciar
              </StatHelpText>
            </Stat>
          </CardBody>
        </Card>

        <Card>
          <CardBody>
            <Stat>
              <Flex alignItems="center" mb={2}>
                <Box p={2} bg="green.100" borderRadius="md" mr={3}>
                  <Icon as={FiCheckCircle} color="green.500" boxSize={6} />
                </Box>
                <StatLabel fontSize="lg">Finalizados</StatLabel>
              </Flex>
              <StatNumber>{finishedServices}</StatNumber>
              <StatHelpText>
                Este mes
              </StatHelpText>
            </Stat>
          </CardBody>
        </Card>

        <Card>
          <CardBody>
            <Stat>
              <Flex alignItems="center" mb={2}>
                <Box p={2} bg="purple.100" borderRadius="md" mr={3}>
                  <Icon as={FiUsers} color="purple.500" boxSize={6} />
                </Box>
                <StatLabel fontSize="lg">Clientes</StatLabel>
              </Flex>
              <StatNumber>{totalClients}</StatNumber>
              <StatHelpText>
                <StatArrow type="increase" />
                2 nuevos esta semana
              </StatHelpText>
            </Stat>
          </CardBody>
        </Card>
      </SimpleGrid>

      {/* Recent Orders */}
      <Card mb={8}>
        <CardHeader>
          <Heading size="md">Actividad Reciente</Heading>
        </CardHeader>
        <CardBody>
          <Table variant="simple" size="sm">
            <Thead>
              <Tr>
                <Th>Folio</Th>
                <Th>Cliente</Th>
                <Th>Estado</Th>
                <Th>Fecha Solicitud</Th>
              </Tr>
            </Thead>
            <Tbody>
              {sortedOrders.slice(0, 7).map((service) => {
                const client = clients.find(c => c.user_id === service.client_id);
                return (
                  <Tr key={service.id} bg={service.status === 'IN_REVIEW' ? 'purple.50' : 'transparent'}>
                    <Td fontWeight="bold">{service.service_number}</Td>
                    <Td>{client?.name || 'Desconocido'}</Td>
                    <Td>
                      <Badge
                        colorScheme={
                            service.status === 'FINISHED' ? 'green' :
                            service.status === 'IN_PROGRESS' ? 'blue' :
                            service.status === 'PENDING' ? 'orange' :
                            service.status === 'IN_REVIEW' ? 'purple' : 'gray'
                        }
                      >
                        {getStatusLabel(service.status)}
                      </Badge>
                    </Td>
                    <Td>{new Date(service.request_date).toLocaleDateString()}</Td>
                  </Tr>
                );
              })}
            </Tbody>
          </Table>
        </CardBody>
      </Card>
    </Box>
  );
};

export default AdminDashboard;
