import { useState, useEffect } from 'react';
import { Outlet } from 'react-router-dom';
import Layout from '../../components/Layout';
import { adminLinks } from './adminLinks';
import { getModulesByCompany } from '../../services/api';

const AdminLayout = () => {
  const user = JSON.parse(sessionStorage.getItem('user') || '{}');
  const [filteredLinks, setFilteredLinks] = useState([]);

  useEffect(() => {
    const fetchModules = async () => {
      if (!user.company_id) {
        setFilteredLinks(adminLinks); // Fallback si no hay company_id
        return;
      }

      try {
        const modules = await getModulesByCompany(user.company_id);

        // Mapeo de nombres de módulos a nombres de enlaces
        // Asegúrate de que las claves coincidan con los nombres en la BD (ModuleModel)
        // y los valores con los nombres en adminLinks.js
        const moduleMap = {
          'Gestión de Clientes': 'Clientes',
          'Catálogo de Servicios': 'Catálogo de Servicios',
          'Gestión de Empleados': 'Empleados',
          'Órdenes de Servicio': 'Órdenes de Servicio',
          'Reportes': 'Reportes'
        };

        const enabledModuleNames = modules
          .filter(m => m.is_enabled)
          .map(m => moduleMap[m.name]);

        const newLinks = adminLinks.filter(link => {
          // Dashboard y Configuración siempre visibles
          if (link.nombre === 'Dashboard' || link.nombre === 'Configuración') return true;
          // Los demás dependen de si su módulo está habilitado
          return enabledModuleNames.includes(link.nombre);
        });

        setFilteredLinks(newLinks);
      } catch (error) {
        console.error('Error fetching modules:', error);
        setFilteredLinks(adminLinks); // Fallback en caso de error
      }
    };

    fetchModules();
  }, [user.company_id]);

  return (
    <Layout
      links={filteredLinks.length > 0 ? filteredLinks : adminLinks}
      userType="ADMIN"
      userName={user.email || 'Administrador'}
      userPhoto="https://ui-avatars.com/api/?name=Admin+User&background=0D8ABC&color=fff"
    >
      <Outlet />
    </Layout>
  );
};

export default AdminLayout;
