import { useState, useEffect } from 'react';
import {
    Box,
    Button,
    Heading,
    Table,
    Thead,
    Tbody,
    Tr,
    Th,
    Td,
    Badge,
    IconButton,
    useDisclosure,
    Modal,
    ModalOverlay,
    ModalContent,
    ModalHeader,
    ModalFooter,
    ModalBody,
    ModalCloseButton,
    FormControl,
    FormLabel,
    Input,
    Select,
    useToast,
    HStack,
    Text,
    Switch,
    NumberInput,
    NumberInputField,
    NumberInputStepper,
    NumberIncrementStepper,
    NumberDecrementStepper,
    Textarea
} from '@chakra-ui/react';
import { FiEdit2, FiTrash2, FiPlus } from 'react-icons/fi';
import { getServicesByCompany, createService, updateService } from '../../services/api';

const CatalogoServicios = () => {
    const [services, setServices] = useState([]);
    const [currentService, setCurrentService] = useState(null);
    const { isOpen, onOpen, onClose } = useDisclosure();
    const toast = useToast();

    const fetchServices = async () => {
        try {
            const user = JSON.parse(sessionStorage.getItem('user'));
            const companyId = user ? user.company_id : null;

            if (companyId) {
                const data = await getServicesByCompany(companyId);
                setServices(data);
            }
        } catch (error) {
            console.error(error);
            toast({
                title: 'Error al cargar servicios',
                status: 'error',
                duration: 3000,
            });
        }
    };

    useEffect(() => {
        fetchServices();
    }, []);

    const handleEdit = (service) => {
        setCurrentService(service);
        onOpen();
    };

    const handleCreate = () => {
        const user = JSON.parse(sessionStorage.getItem('user'));
        setCurrentService({
            name: '',
            description: '',
            category: '',
            max_time_hours: 1,
            has_warranty: true,
            warranty_months: 3,
            warranty_exceptions: '',
            company_id: user ? user.company_id : null
        });
        onOpen();
    };

    const handleSave = async () => {
        try {
            if (currentService.id) {
                await updateService(currentService.id, currentService);
                toast({
                    title: 'Servicio actualizado',
                    status: 'success',
                    duration: 2000,
                });
            } else {
                await createService(currentService);
                toast({
                    title: 'Servicio creado',
                    status: 'success',
                    duration: 2000,
                });
            }
            fetchServices();
            onClose();
        } catch (error) {
            toast({
                title: 'Error al guardar',
                description: error.message,
                status: 'error',
                duration: 3000,
            });
        }
    };

    return (
        <Box>
            <HStack justify="space-between" mb={6}>
                <Heading>Catálogo de Servicios</Heading>
                <Button leftIcon={<FiPlus />} colorScheme="brand" onClick={handleCreate}>
                    Nuevo Servicio
                </Button>
            </HStack>

            <Box bg="white" borderRadius="lg" boxShadow="sm" overflowX="auto">
                <Table variant="simple">
                    <Thead bg="gray.50">
                        <Tr>
                            <Th>Servicio</Th>
                            <Th>Categoría</Th>
                            <Th>Duración Est.</Th>
                            <Th>Garantía</Th>
                            <Th>Acciones</Th>
                        </Tr>
                    </Thead>
                    <Tbody>
                        {services.map((service) => (
                            <Tr key={service.id}>
                                <Td>
                                    <Text fontWeight="bold">{service.name}</Text>
                                    <Text fontSize="xs" color="gray.500" noOfLines={1}>{service.description}</Text>
                                </Td>
                                <Td>
                                    <Badge>{service.category}</Badge>
                                </Td>
                                <Td>{service.max_time_hours} Horas</Td>
                                <Td>
                                    {service.has_warranty ? (
                                        <Badge colorScheme="green">{service.warranty_months} Meses</Badge>
                                    ) : (
                                        <Badge colorScheme="gray">Sin Garantía</Badge>
                                    )}
                                </Td>
                                <Td>
                                    <HStack spacing={2}>
                                        <IconButton
                                            icon={<FiEdit2 />}
                                            size="sm"
                                            aria-label="Editar"
                                            onClick={() => handleEdit(service)}
                                        />
                                    </HStack>
                                </Td>
                            </Tr>
                        ))}
                    </Tbody>
                </Table>
            </Box>

            {/* Modal */}
            <Modal isOpen={isOpen} onClose={onClose} size="lg">
                <ModalOverlay />
                <ModalContent>
                    <ModalHeader>{currentService?.id ? 'Editar Servicio' : 'Nuevo Servicio'}</ModalHeader>
                    <ModalCloseButton />
                    <ModalBody>
                        <Box display="flex" flexDirection="column" gap={4}>
                            <FormControl>
                                <FormLabel>Nombre del Servicio</FormLabel>
                                <Input
                                    value={currentService?.name || ''}
                                    onChange={(e) => setCurrentService({ ...currentService, name: e.target.value })}
                                />
                            </FormControl>

                            <FormControl>
                                <FormLabel>Descripción</FormLabel>
                                <Textarea
                                    value={currentService?.description || ''}
                                    onChange={(e) => setCurrentService({ ...currentService, description: e.target.value })}
                                />
                            </FormControl>

                            <HStack>
                                <FormControl>
                                    <FormLabel>Categoría</FormLabel>
                                    <Select
                                        value={currentService?.category || ''}
                                        onChange={(e) => setCurrentService({ ...currentService, category: e.target.value })}
                                    >
                                        <option value="">Seleccionar...</option>
                                        <option value="Electricidad">Electricidad</option>
                                        <option value="Plomería">Plomería</option>
                                        <option value="Pintura">Pintura</option>
                                        <option value="Climatización">Climatización</option>
                                        <option value="Limpieza">Limpieza</option>
                                        <option value="Mantenimiento">Mantenimiento</option>
                                    </Select>
                                </FormControl>

                                <FormControl>
                                    <FormLabel>Duración (Horas)</FormLabel>
                                    <NumberInput
                                        min={1}
                                        value={currentService?.max_time_hours || 1}
                                        onChange={(val) => setCurrentService({ ...currentService, max_time_hours: parseInt(val) })}
                                    >
                                        <NumberInputField />
                                        <NumberInputStepper>
                                            <NumberIncrementStepper />
                                            <NumberDecrementStepper />
                                        </NumberInputStepper>
                                    </NumberInput>
                                </FormControl>
                            </HStack>

                            <FormControl display="flex" alignItems="center">
                                <FormLabel mb="0">
                                    Ofrece Garantía
                                </FormLabel>
                                <Switch
                                    isChecked={currentService?.has_warranty || false}
                                    onChange={(e) => setCurrentService({ ...currentService, has_warranty: e.target.checked })}
                                />
                            </FormControl>

                            {currentService?.has_warranty && (
                                <>
                                    <FormControl>
                                        <FormLabel>Meses de Garantía</FormLabel>
                                        <NumberInput
                                            min={1}
                                            value={currentService?.warranty_months || 1}
                                            onChange={(val) => setCurrentService({ ...currentService, warranty_months: parseInt(val) })}
                                        >
                                            <NumberInputField />
                                            <NumberInputStepper>
                                                <NumberIncrementStepper />
                                                <NumberDecrementStepper />
                                            </NumberInputStepper>
                                        </NumberInput>
                                    </FormControl>

                                    <FormControl>
                                        <FormLabel>Excepciones de Garantía</FormLabel>
                                        <Textarea
                                            value={currentService?.warranty_exceptions || ''}
                                            onChange={(e) => setCurrentService({ ...currentService, warranty_exceptions: e.target.value })}
                                            placeholder="Ej. Daños causados por mal uso..."
                                        />
                                    </FormControl>
                                </>
                            )}
                        </Box>
                    </ModalBody>

                    <ModalFooter>
                        <Button variant="ghost" mr={3} onClick={onClose}>
                            Cancelar
                        </Button>
                        <Button colorScheme="brand" onClick={handleSave}>
                            Guardar
                        </Button>
                    </ModalFooter>
                </ModalContent>
            </Modal>
        </Box>
    );
};

export default CatalogoServicios;
