import { useState, useEffect } from 'react';
import {
    Box,
    Heading,
    FormControl,
    FormLabel,
    Input,
    Button,
    VStack,
    useToast,
    Text,
    Card,
    CardBody,
    Divider,
    Textarea,
    Image,
    HStack
} from '@chakra-ui/react';
import { getCompanyById, updateCompany } from '../../services/api';

const ConfiguracionEmpresa = () => {
    const [company, setCompany] = useState(null);
    const toast = useToast();

    useEffect(() => {
        const fetchCompany = async () => {
            try {
                const user = JSON.parse(sessionStorage.getItem('user'));
                const companyId = user ? user.company_id : null;

                if (companyId) {
                    const data = await getCompanyById(companyId);
                    setCompany(data);
                }
            } catch (error) {
                console.error(error);
                toast({
                    title: 'Error al cargar configuración',
                    status: 'error',
                    duration: 3000,
                });
            }
        };
        fetchCompany();
    }, []);

    const handleSave = async () => {
        try {
            if (company && company.id) {
                await updateCompany(company.id, company);
                toast({
                    title: 'Configuración actualizada',
                    status: 'success',
                    duration: 3000,
                });
            }
        } catch (error) {
            toast({
                title: 'Error al guardar',
                description: error.message,
                status: 'error',
                duration: 3000,
            });
        }
    };

    if (!company) return <Text>Cargando...</Text>;

    return (
        <Box maxW="container.md">
            <Heading mb={6}>Configuración de Empresa</Heading>

            <Card>
                <CardBody>
                    <VStack spacing={6} align="stretch">
                        <Box>
                            <Heading size="md" mb={2}>Identidad Corporativa</Heading>
                            <Text color="gray.500" fontSize="sm" mb={4}>
                                Esta información será visible para tus clientes y empleados.
                            </Text>
                        </Box>

                        <Divider />

                        <HStack align="start" spacing={6}>
                            <Box>
                                <FormLabel>Logo Actual</FormLabel>
                                <Image
                                    src={company.logo_url}
                                    alt="Logo"
                                    boxSize="100px"
                                    objectFit="cover"
                                    borderRadius="md"
                                    border="1px solid"
                                    borderColor="gray.200"
                                />
                            </Box>
                            <FormControl>
                                <FormLabel>URL del Logo</FormLabel>
                                <Input
                                    value={company.logo_url}
                                    onChange={(e) => setCompany({ ...company, logo_url: e.target.value })}
                                />
                            </FormControl>
                        </HStack>

                        <FormControl>
                            <FormLabel>Nombre Comercial</FormLabel>
                            <Input
                                value={company.name}
                                onChange={(e) => setCompany({ ...company, name: e.target.value })}
                            />
                        </FormControl>

                        <FormControl>
                            <FormLabel>Eslogan</FormLabel>
                            <Input
                                value={company.slogan || ''}
                                onChange={(e) => setCompany({ ...company, slogan: e.target.value })}
                                placeholder="Ej. Soluciones integrales para tu hogar"
                            />
                        </FormControl>

                        <Box mt={4}>
                            <Heading size="md" mb={2}>Mensajes Automáticos</Heading>
                            <Divider mb={4} />
                        </Box>

                        <FormControl>
                            <FormLabel>Mensaje de Finalización de Servicio</FormLabel>
                            <Textarea
                                value={company.service_completion_message}
                                onChange={(e) => setCompany({ ...company, service_completion_message: e.target.value })}
                                rows={3}
                            />
                            <Text fontSize="xs" color="gray.500" mt={1}>
                                Este mensaje se enviará al cliente cuando un servicio sea marcado como finalizado.
                            </Text>
                        </FormControl>

                        <Box pt={4}>
                            <Button colorScheme="brand" onClick={handleSave}>
                                Guardar Cambios
                            </Button>
                        </Box>
                    </VStack>
                </CardBody>
            </Card>
        </Box>
    );
};

export default ConfiguracionEmpresa;
