import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import {
  Box,
  Heading,
  Button,
  Table,
  Thead,
  Tbody,
  Tr,
  Th,
  Td,
  TableContainer,
  Modal,
  ModalOverlay,
  ModalContent,
  ModalHeader,
  ModalBody,
  ModalFooter,
  ModalCloseButton,
  useDisclosure,
  FormControl,
  FormLabel,
  Input,
  Avatar,
  Select,
  useToast,
  Flex,
  HStack,
  IconButton,
  Icon,
  Text,
  Grid,
  GridItem,
  VStack,
  Tooltip,
  AlertDialog,
  AlertDialogBody,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogContent,
  AlertDialogOverlay,
} from '@chakra-ui/react';
import { BiPencil, BiTrash, BiShow, BiPlus, BiUserMinus } from 'react-icons/bi';
import { FilePond, registerPlugin } from 'react-filepond';
import 'filepond/dist/filepond.min.css';
import FilePondPluginImagePreview from 'filepond-plugin-image-preview';
import 'filepond-plugin-image-preview/dist/filepond-plugin-image-preview.css';
import Layout from '../../components/Layout';
import { adminLinks } from './adminLinks';
import { mockAlumnos, mockPadres } from '../../data/mockData';
import { estadosMexico } from '../../data/states';

registerPlugin(FilePondPluginImagePreview);

const GestionAlumnos = () => {
  const [alumnos, setAlumnos] = useState(mockAlumnos);
  const [alumnoEditando, setAlumnoEditando] = useState(null);
  const navigate = useNavigate();
  const { isOpen, onOpen, onClose } = useDisclosure();
  const { isOpen: isEditOpen, onOpen: onEditOpen, onClose: onEditClose } = useDisclosure();
  const { isOpen: isBajaOpen, onOpen: onBajaOpen, onClose: onBajaClose } = useDisclosure();
  const [alumnoParaBaja, setAlumnoParaBaja] = useState(null);
  const cancelRef = React.useRef();
  const toast = useToast();

  // Form States
  const [formData, setFormData] = useState({
    nombre: '',
    email: '',
    fecha_nacimiento: '',
    id_padre: '',
    telefono: '',
    curp: '',
    direccion: {
      estado: '',
      municipio: '',
      cp: '',
      calle: '',
      numero: ''
    }
  });
  const [files, setFiles] = useState([]);
  const [municipiosDisponibles, setMunicipiosDisponibles] = useState([]);

  useEffect(() => {
    if (formData.direccion.estado) {
      const estadoObj = estadosMexico.find(e => e.nombre === formData.direccion.estado);
      setMunicipiosDisponibles(estadoObj ? estadoObj.municipios : []);
    } else {
      setMunicipiosDisponibles([]);
    }
  }, [formData.direccion.estado]);

  // Calcular edad basado en fecha de nacimiento
  const calcularEdad = (fechaNacimiento) => {
    if (!fechaNacimiento) return '';
    const hoy = new Date();
    const nacimiento = new Date(fechaNacimiento);
    let edad = hoy.getFullYear() - nacimiento.getFullYear();
    const mes = hoy.getMonth() - nacimiento.getMonth();
    if (mes < 0 || (mes === 0 && hoy.getDate() < nacimiento.getDate())) {
      edad--;
    }
    return edad;
  };

  const handleCrear = () => {
    setAlumnoEditando(null);
    setFormData({
      nombre: '',
      email: '',
      fecha_nacimiento: '',
      id_padre: '',
      telefono: '',
      curp: '',
      direccion: {
        estado: '',
        municipio: '',
        cp: '',
        calle: '',
        numero: ''
      }
    });
    setFiles([]);
    onOpen();
  };

  const handleEditar = (alumno) => {
    setAlumnoEditando(alumno);
    setFormData({
      nombre: alumno.nombre,
      email: alumno.email,
      fecha_nacimiento: alumno.fecha_nacimiento || '',
      id_padre: alumno.id_padre,
      telefono: alumno.telefono || '',
      curp: alumno.curp || '',
      direccion: alumno.direccion || {
        estado: '',
        municipio: '',
        cp: '',
        calle: '',
        numero: ''
      }
    });
    setFiles(alumno.foto_url ? [{
      source: alumno.foto_url,
      options: {
        type: 'local'
      }
    }] : []);
    onEditOpen();
  };

  const handleInputChange = (e) => {
    const { name, value } = e.target;
    if (name.includes('.')) {
      const [parent, child] = name.split('.');
      setFormData(prev => ({
        ...prev,
        [parent]: {
          ...prev[parent],
          [child]: value
        }
      }));
    } else {
      setFormData(prev => ({
        ...prev,
        [name]: value
      }));
    }
  };

  const handleVerDetalle = (id) => {
    navigate(`/admin/alumnos/${id}`);
  };

  const handleClickBaja = (alumno) => {
    setAlumnoParaBaja(alumno);
    onBajaOpen();
  };

  const handleConfirmarBaja = () => {
    // Aquí iría la lógica para actualizar el estatus en el backend
    // Por ahora, simulamos la acción y mostramos el mensaje requerido

    toast({
      title: 'Alumno dado de baja',
      description: `El alumno ${alumnoParaBaja?.nombre} ha sido dado de baja. Las clases de esos días han sido liberadas en el calendario.`,
      status: 'warning',
      duration: 5000,
      isClosable: true,
    });

    // Opcional: Actualizar estado local para reflejar cambio visual si fuera necesario
    // setAlumnos(alumnos.filter(a => a.id !== alumnoParaBaja.id)); 

    onBajaClose();
    setAlumnoParaBaja(null);
  };

  const handleGuardar = (e) => {
    e.preventDefault();

    const nuevoAlumno = {
      id: alumnoEditando ? alumnoEditando.id : alumnos.length + 1,
      ...formData,
      foto_url: files.length > 0 ? URL.createObjectURL(files[0].file) : (alumnoEditando?.foto_url || 'https://i.pravatar.cc/150?img=1')
    };

    if (alumnoEditando) {
      setAlumnos(alumnos.map(a => a.id === alumnoEditando.id ? nuevoAlumno : a));
    } else {
      setAlumnos([...alumnos, nuevoAlumno]);
    }

    toast({
      title: 'Alumno guardado',
      description: alumnoEditando ? 'Alumno actualizado' : 'Alumno creado',
      status: 'success',
      duration: 2000,
      isClosable: true,
    });
    onClose();
    onEditClose();
  };

  const getNombrePadre = (id) => {
    const padre = mockPadres.find(p => p.id === parseInt(id));
    return padre ? padre.nombre : 'No asignado';
  };

  return (
    <Layout links={adminLinks} userType="Administrador" userName="Admin User" userPhoto="https://i.pravatar.cc/150?img=68">
      <Box>

        {/* Instrucciones minimalistas */}
        <Text fontSize="sm" color="gray.600" mb={6}>
          Administra los alumnos inscritos en la escuela. Puedes ver su información, editarla o agregar nuevos alumnos.
        </Text>

        {/* ¡LA TARJETA! Aquí está la magia */}
        <Box
          bg="rgba(255, 255, 255, 0.6)"
          backdropFilter="blur(20px) saturate(180%)"
          WebkitBackdropFilter="blur(20px) saturate(180%)"
          borderRadius="xl"
          boxShadow="0 8px 32px 0 rgba(0, 0, 0, 0.1)"
          border="1px solid"
          borderColor="rgba(255, 255, 255, 0.3)"
          overflow="hidden"
        >

          {/* Header de la Tarjeta (como en "Facit") */}
          <Flex justify="space-between" align="center" p={5} borderBottom="1px" borderColor="rgba(255, 255, 255, 0.2)">
            <Heading as="h3" size="md">Lista de Alumnos</Heading>
            {/* Botón "Píldora" */}
            <Button
              colorScheme="blue"
              borderRadius="full"
              leftIcon={<Icon as={BiPlus} />}
              onClick={handleCrear}
            >
              Crear Nuevo
            </Button>
          </Flex>

          {/* La tabla va DENTRO del cuerpo de la tarjeta */}
          <TableContainer>
            {/* ¡Quitar 'variant="striped"'! */}
            <Table variant="simple">
              <Thead>
                {/* Estilo de cabecera moderno */}
                <Tr>
                  <Th textTransform="uppercase" fontSize="xs" color="gray.500">Alumno</Th>
                  <Th textTransform="uppercase" fontSize="xs" color="gray.500">Padre/Tutor</Th>
                  <Th textTransform="uppercase" fontSize="xs" color="gray.500">Edad</Th>
                  <Th textTransform="uppercase" fontSize="xs" color="gray.500">Teléfono</Th>
                  <Th textTransform="uppercase" fontSize="xs" color="gray.500">Acciones</Th>
                </Tr>
              </Thead>
              <Tbody>
                {alumnos.map((alumno) => (
                  <Tr key={alumno.id} _hover={{ bg: "gray.50" }}>
                    <Td>
                      <HStack spacing={3}>
                        <Avatar size="sm" name={alumno.nombre} src={alumno.foto_url} />
                        <Box>
                          <Text fontWeight="bold">{alumno.nombre}</Text>
                          <Text fontSize="sm" color="gray.500">{alumno.email}</Text>
                        </Box>
                      </HStack>
                    </Td>
                    <Td>{getNombrePadre(alumno.id_padre)}</Td>
                    <Td>{calcularEdad(alumno.fecha_nacimiento)} años</Td>
                    <Td>{alumno.telefono || 'N/A'}</Td>
                    <Td>
                      <HStack spacing={2}>
                        <Tooltip label="Ver Detalle">
                          <IconButton
                            icon={<Icon as={BiShow} />}
                            variant="ghost"
                            colorScheme="blue"
                            aria-label="Ver Detalle"
                            onClick={() => handleVerDetalle(alumno.id)}
                          />
                        </Tooltip>
                        <Tooltip label="Editar">
                          <IconButton
                            icon={<BiPencil />}
                            aria-label="Editar"
                            size="sm"
                            variant="ghost"
                            colorScheme="blue"
                            onClick={() => handleEditar(alumno)}
                          />
                        </Tooltip>
                        <Tooltip label="Dar de baja">
                          <IconButton
                            icon={<BiUserMinus />}
                            aria-label="Dar de baja"
                            size="sm"
                            variant="ghost"
                            colorScheme="orange"
                            onClick={() => handleClickBaja(alumno)}
                          />
                        </Tooltip>
                        <Tooltip label="Eliminar">
                          <IconButton
                            icon={<BiTrash />}
                            aria-label="Eliminar"
                            size="sm"
                            variant="ghost"
                            colorScheme="red"
                          />
                        </Tooltip>
                      </HStack>
                    </Td>
                  </Tr>
                ))}
              </Tbody>
            </Table>
          </TableContainer>
        </Box>
      </Box>

      <Modal isOpen={isOpen || isEditOpen} onClose={() => { onClose(); onEditClose(); }} isCentered motionPreset="scale" size="6xl">
        <ModalOverlay bg="blackAlpha.600" backdropFilter="blur(4px)" />
        <ModalContent
          borderRadius="16px"
          bg="rgba(255, 255, 255, 0.95)"
          backdropFilter="blur(20px) saturate(180%)"
          boxShadow="0 8px 32px 0 rgba(0, 0, 0, 0.2)"
        >
          <ModalHeader borderBottom="1px solid" borderColor="rgba(255, 255, 255, 0.2)" pb={4}>
            {alumnoEditando ? 'Editar Alumno' : 'Crear Nuevo Alumno'}
          </ModalHeader>
          <ModalCloseButton />
          <form onSubmit={handleGuardar}>
            <ModalBody p={6}>
              <Flex gap={8} flexDirection={{ base: 'column', lg: 'row' }}>
                {/* Columna Izquierda: Foto */}
                <Box w={{ base: '100%', lg: '250px' }} flexShrink={0}>
                  <FormLabel textAlign="center" fontWeight="bold" mb={4} color="gray.700">Fotografía</FormLabel>
                  <Box
                    w="220px"
                    h="220px"
                    mx="auto"
                    borderRadius="full"
                    overflow="hidden"
                    boxShadow="0 4px 12px rgba(0,0,0,0.1)"
                    bg="gray.50"
                    border="2px solid"
                    borderColor="gray.100"
                  >
                    <FilePond
                      files={files}
                      onupdatefiles={setFiles}
                      allowMultiple={false}
                      maxFiles={1}
                      name="files"
                      labelIdle='<span class="filepond--label-action">Subir Foto</span>'
                      acceptedFileTypes={['image/*']}
                      stylePanelLayout="compact circle"
                      styleLoadIndicatorPosition="center bottom"
                      styleProgressIndicatorPosition="right bottom"
                      styleButtonRemoveItemPosition="left bottom"
                      styleButtonProcessItemPosition="right bottom"
                    />
                  </Box>
                  <Text fontSize="xs" color="gray.500" textAlign="center" mt={3}>
                    Arrastra una imagen o haz clic para seleccionar
                  </Text>
                </Box>

                {/* Columna Derecha: Datos */}
                <Box flex={1}>
                  <VStack spacing={6} align="stretch">

                    {/* Sección Personal */}
                    <Box bg="white" p={4} borderRadius="lg" border="1px solid" borderColor="gray.100" shadow="sm">
                      <Heading size="sm" mb={4} color="blue.600" borderBottom="1px solid" borderColor="gray.100" pb={2} display="flex" alignItems="center">
                        <Icon as={BiPencil} mr={2} /> Información Personal
                      </Heading>
                      <Grid templateColumns={{ base: '1fr', md: 'repeat(2, 1fr)' }} gap={4}>
                        <GridItem colSpan={{ base: 1, md: 2 }}>
                          <FormControl isRequired>
                            <FormLabel fontSize="sm" fontWeight="600" color="gray.700">Nombre Completo</FormLabel>
                            <Input name="nombre" value={formData.nombre} onChange={handleInputChange} placeholder="Ej. Sofía López" borderRadius="md" />
                          </FormControl>
                        </GridItem>
                        <GridItem>
                          <FormControl isRequired>
                            <FormLabel fontSize="sm" fontWeight="600" color="gray.700">Email</FormLabel>
                            <Input name="email" type="email" value={formData.email} onChange={handleInputChange} placeholder="correo@ejemplo.com" borderRadius="md" />
                          </FormControl>
                        </GridItem>
                        <GridItem>
                          <FormControl isRequired>
                            <FormLabel fontSize="sm" fontWeight="600" color="gray.700">Fecha de Nacimiento</FormLabel>
                            <Input name="fecha_nacimiento" type="date" value={formData.fecha_nacimiento} onChange={handleInputChange} borderRadius="md" />
                          </FormControl>
                        </GridItem>
                        <GridItem>
                          <FormControl>
                            <FormLabel fontSize="sm" fontWeight="600" color="gray.700">Edad Calculada</FormLabel>
                            <Input value={formData.fecha_nacimiento ? `${calcularEdad(formData.fecha_nacimiento)} años` : ''} isReadOnly bg="gray.50" borderRadius="md" />
                          </FormControl>
                        </GridItem>
                        <GridItem>
                          <FormControl isRequired>
                            <FormLabel fontSize="sm" fontWeight="600" color="gray.700">Padre/Tutor</FormLabel>
                            <Select name="id_padre" value={formData.id_padre} onChange={handleInputChange} borderRadius="md">
                              <option value="">Seleccionar...</option>
                              {mockPadres.map(padre => (
                                <option key={padre.id} value={padre.id}>{padre.nombre}</option>
                              ))}
                            </Select>
                          </FormControl>
                        </GridItem>
                        <GridItem>
                          <FormControl>
                            <FormLabel fontSize="sm" fontWeight="600" color="gray.700">Teléfono</FormLabel>
                            <Input name="telefono" value={formData.telefono} onChange={handleInputChange} placeholder="555-555-5555" borderRadius="md" />
                          </FormControl>
                        </GridItem>
                        <GridItem colSpan={{ base: 1, md: 2 }}>
                          <FormControl>
                            <FormLabel fontSize="sm" fontWeight="600" color="gray.700">CURP</FormLabel>
                            <Input name="curp" value={formData.curp} onChange={handleInputChange} placeholder="ABCD123456..." borderRadius="md" />
                          </FormControl>
                        </GridItem>
                      </Grid>
                    </Box>

                    {/* Sección Dirección */}
                    <Box bg="white" p={4} borderRadius="lg" border="1px solid" borderColor="gray.100" shadow="sm">
                      <Heading size="sm" mb={4} color="blue.600" borderBottom="1px solid" borderColor="gray.100" pb={2} display="flex" alignItems="center">
                        <Icon as={BiPencil} mr={2} /> Dirección
                      </Heading>
                      <Grid templateColumns={{ base: '1fr', md: 'repeat(2, 1fr)' }} gap={4}>
                        <GridItem>
                          <FormControl>
                            <FormLabel fontSize="sm" fontWeight="600" color="gray.700">Estado</FormLabel>
                            <Select name="direccion.estado" value={formData.direccion.estado} onChange={handleInputChange} placeholder="Selecciona estado" borderRadius="md">
                              {estadosMexico.map(e => (
                                <option key={e.nombre} value={e.nombre}>{e.nombre}</option>
                              ))}
                            </Select>
                          </FormControl>
                        </GridItem>
                        <GridItem>
                          <FormControl>
                            <FormLabel fontSize="sm" fontWeight="600" color="gray.700">Municipio</FormLabel>
                            <Select name="direccion.municipio" value={formData.direccion.municipio} onChange={handleInputChange} placeholder="Selecciona municipio" borderRadius="md">
                              {municipiosDisponibles.map(m => (
                                <option key={m} value={m}>{m}</option>
                              ))}
                            </Select>
                          </FormControl>
                        </GridItem>
                        <GridItem colSpan={{ base: 1, md: 2 }}>
                          <FormControl>
                            <FormLabel fontSize="sm" fontWeight="600" color="gray.700">Calle y Número</FormLabel>
                            <HStack>
                              <Input name="direccion.calle" value={formData.direccion.calle} onChange={handleInputChange} placeholder="Calle" flex={2} borderRadius="md" />
                              <Input name="direccion.numero" value={formData.direccion.numero} onChange={handleInputChange} placeholder="No." flex={1} borderRadius="md" />
                            </HStack>
                          </FormControl>
                        </GridItem>
                        <GridItem>
                          <FormControl>
                            <FormLabel fontSize="sm" fontWeight="600" color="gray.700">Código Postal</FormLabel>
                            <Input name="direccion.cp" value={formData.direccion.cp} onChange={handleInputChange} placeholder="00000" borderRadius="md" />
                          </FormControl>
                        </GridItem>
                      </Grid>
                    </Box>

                  </VStack>
                </Box>
              </Flex>
            </ModalBody>
            <ModalFooter borderTop="1px solid" borderColor="rgba(255, 255, 255, 0.2)" pt={4}>
              <Button variant="ghost" mr={3} onClick={() => { onClose(); onEditClose(); }} borderRadius="8px">Cancelar</Button>
              <Button colorScheme="blue" type="submit" borderRadius="8px" px={6}>Guardar</Button>
            </ModalFooter>
          </form>
        </ModalContent>
      </Modal>

      {/* Diálogo de Confirmación de Baja */}
      <AlertDialog
        isOpen={isBajaOpen}
        leastDestructiveRef={cancelRef}
        onClose={onBajaClose}
      >
        <AlertDialogOverlay>
          <AlertDialogContent>
            <AlertDialogHeader fontSize="lg" fontWeight="bold">
              Dar de baja alumno
            </AlertDialogHeader>

            <AlertDialogBody>
              ¿Estás seguro que deseas dar de baja a <strong>{alumnoParaBaja?.nombre}</strong>?
              <br /><br />
              <Text color="orange.600" fontWeight="medium">
                Esta acción liberará todas las clases asignadas a este alumno en el calendario.
              </Text>
            </AlertDialogBody>

            <AlertDialogFooter>
              <Button ref={cancelRef} onClick={onBajaClose}>
                Cancelar
              </Button>
              <Button colorScheme="orange" onClick={handleConfirmarBaja} ml={3}>
                Dar de Baja
              </Button>
            </AlertDialogFooter>
          </AlertDialogContent>
        </AlertDialogOverlay>
      </AlertDialog>
    </Layout>
  );
};

export default GestionAlumnos;

