import { useState, useEffect } from 'react';
import {
    Box,
    Button,
    Heading,
    Table,
    Thead,
    Tbody,
    Tr,
    Th,
    Td,
    Badge,
    IconButton,
    useDisclosure,
    Modal,
    ModalOverlay,
    ModalContent,
    ModalHeader,
    ModalFooter,
    ModalBody,
    ModalCloseButton,
    FormControl,
    FormLabel,
    Input,
    useToast,
    HStack,
    Text,
    Avatar,
    Switch
} from '@chakra-ui/react';
import { FiEdit2, FiTrash2, FiPlus, FiEye } from 'react-icons/fi';
import { getClients as fetchClients, createClient, updateClient } from '../../services/api';

const GestionClientes = () => {
    const [clients, setClients] = useState([]);
    const [currentClient, setCurrentClient] = useState(null);
    const [currentUser, setCurrentUser] = useState(null);
    const { isOpen, onOpen, onClose } = useDisclosure();
    const toast = useToast();

    const loadClients = async () => {
        try {
            const data = await fetchClients();
            setClients(data);
        } catch (error) {
            toast({
                title: 'Error al cargar clientes',
                description: error.message,
                status: 'error',
                duration: 3000,
            });
        }
    };

    const handleToggleStatus = async (client) => {
        try {
            const newStatus = !client.is_active;
            await updateClient(client.user_id, { ...client, is_active: newStatus });

            // Actualizar el estado local inmediatamente
            setClients(clients.map(c =>
                c.user_id === client.user_id ? { ...c, is_active: newStatus } : c
            ));

            toast({
                title: `Cliente ${newStatus ? 'activado' : 'desactivado'}`,
                status: 'success',
                duration: 2000,
            });
        } catch (error) {
            toast({
                title: 'Error al cambiar estado',
                description: error.message,
                status: 'error',
                duration: 3000,
            });
        }
    };

    useEffect(() => {
        const user = JSON.parse(sessionStorage.getItem('user'));
        setCurrentUser(user);
        loadClients();
    }, []);

    const handleEdit = (client) => {
        setCurrentClient(client);
        onOpen();
    };

    const handleCreate = () => {
        setCurrentClient({
            name: '',
            email: '',
            phone: '',
            client_number: `CLI-${new Date().getFullYear()}-${clients.length + 1}`,
            is_active: true,
            allow_access_without_service: false,
            address: '',
            rfc: '',
            company_id: currentUser ? currentUser.company_id : null
        });
        onOpen();
    };

    const handleSave = async () => {
        try {
            if (currentClient.user_id) {
                const dataToUpdate = { ...currentClient };
                if (!dataToUpdate.password) {
                    delete dataToUpdate.password;
                }
                const updatedClient = await updateClient(currentClient.user_id, dataToUpdate);

                // Actualizar el cliente en el estado local inmediatamente
                setClients(clients.map(c =>
                    c.user_id === currentClient.user_id ? { ...c, ...updatedClient } : c
                ));
                 // Forzar la actualización visual recargando los clientes
                loadClients();

                toast({
                    title: 'Cliente actualizado',
                    status: 'success',
                    duration: 2000,
                });
            } else {
                const newClientData = {
                    ...currentClient,
                    user_id: `client-${Date.now()}`,
                    company_id: currentUser.company_id
                };
                const newClient = await createClient(newClientData);

                // Agregar el nuevo cliente al estado local y refrescar
                setClients(prevClients => [...prevClients, newClient]);
                loadClients();


                toast({
                    title: 'Cliente creado',
                    status: 'success',
                    duration: 2000,
                });
            }

            onClose();
        } catch (error) {
            toast({
                title: 'Error al guardar',
                description: error.message,
                status: 'error',
                duration: 3000,
            });
        }
    };

    return (
        <Box>
            <HStack justify="space-between" mb={6}>
                <Heading>Gestión de Clientes</Heading>
                <Button leftIcon={<FiPlus />} colorScheme="brand" onClick={handleCreate}>
                    Nuevo Cliente
                </Button>
            </HStack>

            <Box bg="white" borderRadius="lg" boxShadow="sm" overflowX="auto">
                <Table variant="simple">
                    <Thead bg="gray.50">
                        <Tr>
                            <Th>Cliente</Th>
                            <Th>Contacto</Th>
                            <Th>RFC</Th>
                            <Th>Acceso Especial</Th>
                            <Th>Estado</Th>
                            <Th>Acciones</Th>
                        </Tr>
                    </Thead>
                    <Tbody>
                        {clients.map((client) => (
                            <Tr key={client.user_id}>
                                <Td>
                                    <HStack>
                                        <Avatar size="sm" name={client.name} />
                                        <Box>
                                            <Text fontWeight="bold">{client.name}</Text>
                                            <Text fontSize="xs" color="gray.500">{client.client_number}</Text>
                                        </Box>
                                    </HStack>
                                </Td>
                                <Td>
                                    <Text fontSize="sm">{client.email}</Text>
                                    <Text fontSize="xs" color="gray.500">{client.phone}</Text>
                                </Td>
                                <Td>{client.rfc || 'N/A'}</Td>
                                <Td>
                                    <Badge colorScheme={client.allow_access_without_service ? 'purple' : 'gray'}>
                                        {client.allow_access_without_service ? 'PERMITIDO' : 'NORMAL'}
                                    </Badge>
                                </Td>
                                <Td>
                                    <Badge colorScheme={client.is_active ? 'green' : 'red'}>
                                        {client.is_active ? 'ACTIVO' : 'INACTIVO'}
                                    </Badge>
                                </Td>
                                <Td>
                                    <HStack spacing={2}>
                                        <Switch
                                            size="sm"
                                            isChecked={client.is_active}
                                            onChange={() => handleToggleStatus(client)}
                                            colorScheme="green"
                                        />
                                        <IconButton
                                            icon={<FiEdit2 />}
                                            size="sm"
                                            aria-label="Editar"
                                            onClick={() => handleEdit(client)}
                                        />
                                    </HStack>
                                </Td>
                            </Tr>
                        ))}
                    </Tbody>
                </Table>
            </Box>

            {/* Modal */}
            <Modal isOpen={isOpen} onClose={onClose} size="lg">
                <ModalOverlay />
                <ModalContent>
                    <ModalHeader>{currentClient?.user_id ? 'Editar Cliente' : 'Nuevo Cliente'}</ModalHeader>
                    <ModalCloseButton />
                    <ModalBody>
                        <Box display="grid" gridTemplateColumns="1fr 1fr" gap={4}>
                            <FormControl gridColumn="span 2">
                                <FormLabel>Nombre Completo</FormLabel>
                                <Input
                                    value={currentClient?.name || ''}
                                    onChange={(e) => setCurrentClient({ ...currentClient, name: e.target.value })}
                                />
                            </FormControl>

                            <FormControl>
                                <FormLabel>Email</FormLabel>
                                <Input
                                    value={currentClient?.email || ''}
                                    onChange={(e) => setCurrentClient({ ...currentClient, email: e.target.value })}
                                />
                            </FormControl>

                            <FormControl>
                                <FormLabel>Teléfono</FormLabel>
                                <Input
                                    value={currentClient?.phone || ''}
                                    onChange={(e) => setCurrentClient({ ...currentClient, phone: e.target.value })}
                                />
                            </FormControl>

                            <FormControl>
                                <FormLabel>RFC</FormLabel>
                                <Input
                                    value={currentClient?.rfc || ''}
                                    onChange={(e) => setCurrentClient({ ...currentClient, rfc: e.target.value })}
                                />
                            </FormControl>

                            <FormControl display="flex" alignItems="center">
                                <FormLabel mb="0">
                                    Acceso sin servicio activo
                                </FormLabel>
                                <Switch
                                    isChecked={currentClient?.allow_access_without_service || false}
                                    onChange={(e) => setCurrentClient({ ...currentClient, allow_access_without_service: e.target.checked })}
                                />
                            </FormControl>

                            <FormControl display="flex" alignItems="center">
                                <FormLabel mb="0">
                                    Cliente Activo
                                </FormLabel>
                                <Switch
                                    isChecked={currentClient?.is_active || false}
                                    onChange={(e) => setCurrentClient({ ...currentClient, is_active: e.target.checked })}
                                />
                            </FormControl>

                            <FormControl gridColumn="span 2">
                                <FormLabel>Dirección</FormLabel>
                                <Input
                                    value={currentClient?.address || ''}
                                    onChange={(e) => setCurrentClient({ ...currentClient, address: e.target.value })}
                                />
                            </FormControl>

                            <FormControl gridColumn="span 2">
                                <FormLabel>Contraseña {currentClient?.user_id && '(Dejar en blanco para no cambiar)'}</FormLabel>
                                <Input
                                    type="password"
                                    placeholder={currentClient?.user_id ? '••••••••' : 'Ingresar contraseña'}
                                    value={currentClient?.password || ''}
                                    onChange={(e) => setCurrentClient({ ...currentClient, password: e.target.value })}
                                />
                            </FormControl>
                        </Box>
                    </ModalBody>

                    <ModalFooter>
                        <Button variant="ghost" mr={3} onClick={onClose}>
                            Cancelar
                        </Button>
                        <Button colorScheme="brand" onClick={handleSave}>
                            Guardar
                        </Button>
                    </ModalFooter>
                </ModalContent>
            </Modal>
        </Box>
    );
};

export default GestionClientes;
