import { useState, useEffect } from 'react';
import {
    Box,
    Button,
    Heading,
    Table,
    Thead,
    Tbody,
    Tr,
    Th,
    Td,
    Badge,
    IconButton,
    useDisclosure,
    Modal,
    ModalOverlay,
    ModalContent,
    ModalHeader,
    ModalFooter,
    ModalBody,
    ModalCloseButton,
    FormControl,
    FormLabel,
    Input,
    Select,
    useToast,
    HStack,
    Text,
    Avatar
} from '@chakra-ui/react';
import { FiEdit2, FiTrash2, FiPlus, FiPower } from 'react-icons/fi';
import { getEmployees, createEmployee, updateEmployee } from '../../services/api';

const GestionEmpleados = () => {
    const [employees, setEmployees] = useState([]);
    const [currentEmployee, setCurrentEmployee] = useState(null);
    const { isOpen, onOpen, onClose } = useDisclosure();
    const toast = useToast();

    const fetchEmployees = async () => {
        try {
            const user = JSON.parse(sessionStorage.getItem('user'));
            const companyId = user ? user.company_id : null;

            if (companyId) {
                const allEmployees = await getEmployees();
                setEmployees(allEmployees.filter(e => e.company_id === companyId));
            }
        } catch (error) {
            console.error(error);
            toast({
                title: 'Error al cargar empleados',
                status: 'error',
                duration: 3000,
            });
        }
    };

    useEffect(() => {
        fetchEmployees();
    }, []);

    const handleEdit = (employee) => {
        setCurrentEmployee(employee);
        onOpen();
    };

    const handleCreate = () => {
        const user = JSON.parse(sessionStorage.getItem('user'));
        setCurrentEmployee({
            name: '',
            email: '',
            specialty: '',
            phone: '',
            is_active: true,
            company_id: user ? user.company_id : null
        });
        onOpen();
    };

    const handleSave = async () => {
        try {
            if (currentEmployee.user_id) {
                const dataToUpdate = { ...currentEmployee };
                if (!dataToUpdate.password) {
                    delete dataToUpdate.password;
                }
                await updateEmployee(currentEmployee.user_id, dataToUpdate);
                toast({
                    title: 'Empleado actualizado',
                    status: 'success',
                    duration: 2000,
                });
            } else {
                if (!currentEmployee.password) {
                    toast({
                        title: 'Error',
                        description: 'La contraseña es requerida para nuevos empleados',
                        status: 'error',
                        duration: 3000,
                    });
                    return;
                }
                await createEmployee({
                    ...currentEmployee,
                    user_id: `emp-${Date.now()}`, // ID temporal
                    // password will be hashed by backend
                });
                toast({
                    title: 'Empleado creado',
                    status: 'success',
                    duration: 2000,
                });
            }
            fetchEmployees();
            onClose();
        } catch (error) {
            toast({
                title: 'Error al guardar',
                description: error.message,
                status: 'error',
                duration: 3000,
            });
        }
    };

    const toggleStatus = async (id) => {
        const employee = employees.find(e => e.user_id === id);
        if (!employee) return;

        try {
            const newStatus = !employee.is_active;
            await updateEmployee(id, { ...employee, is_active: newStatus });

            toast({
                title: `Empleado ${newStatus ? 'activado' : 'desactivado'}`,
                status: newStatus ? 'success' : 'warning',
                duration: 2000,
            });
            fetchEmployees();
        } catch (error) {
            toast({
                title: 'Error al actualizar estado',
                description: error.message,
                status: 'error',
                duration: 3000,
            });
        }
    };

    return (
        <Box>
            <HStack justify="space-between" mb={6}>
                <Heading>Gestión de Empleados</Heading>
                <Button leftIcon={<FiPlus />} colorScheme="brand" onClick={handleCreate}>
                    Nuevo Empleado
                </Button>
            </HStack>

            <Box bg="white" borderRadius="lg" boxShadow="sm" overflowX="auto">
                <Table variant="simple">
                    <Thead bg="gray.50">
                        <Tr>
                            <Th>Empleado</Th>
                            <Th>Especialidad</Th>
                            <Th>Contacto</Th>
                            <Th>Estado</Th>
                            <Th>Acciones</Th>
                        </Tr>
                    </Thead>
                    <Tbody>
                        {employees.map((employee) => (
                            <Tr key={employee.user_id}>
                                <Td>
                                    <HStack>
                                        <Avatar size="sm" name={employee.name} />
                                        <Box>
                                            <Text fontWeight="bold">{employee.name}</Text>
                                            <Text fontSize="xs" color="gray.500">ID: {employee.user_id}</Text>
                                        </Box>
                                    </HStack>
                                </Td>
                                <Td>
                                    <Badge colorScheme="blue">{employee.specialty}</Badge>
                                </Td>
                                <Td>
                                    <Text fontSize="sm">{employee.email}</Text>
                                    <Text fontSize="xs" color="gray.500">{employee.phone}</Text>
                                </Td>
                                <Td>
                                    <Badge colorScheme={employee.is_active ? 'green' : 'red'}>
                                        {employee.is_active ? 'ACTIVO' : 'INACTIVO'}
                                    </Badge>
                                </Td>
                                <Td>
                                    <HStack spacing={2}>
                                        <IconButton
                                            icon={<FiEdit2 />}
                                            size="sm"
                                            aria-label="Editar"
                                            onClick={() => handleEdit(employee)}
                                        />
                                        <IconButton
                                            icon={<FiPower />}
                                            size="sm"
                                            colorScheme={employee.is_active ? 'red' : 'green'}
                                            aria-label="Cambiar estado"
                                            onClick={() => toggleStatus(employee.user_id)}
                                        />
                                    </HStack>
                                </Td>
                            </Tr>
                        ))}
                    </Tbody>
                </Table>
            </Box>

            {/* Modal */}
            <Modal isOpen={isOpen} onClose={onClose}>
                <ModalOverlay />
                <ModalContent>
                    <ModalHeader>{currentEmployee?.user_id ? 'Editar Empleado' : 'Nuevo Empleado'}</ModalHeader>
                    <ModalCloseButton />
                    <ModalBody>
                        <Box display="flex" flexDirection="column" gap={4}>
                            <FormControl>
                                <FormLabel>Nombre Completo</FormLabel>
                                <Input
                                    value={currentEmployee?.name || ''}
                                    onChange={(e) => setCurrentEmployee({ ...currentEmployee, name: e.target.value })}
                                />
                            </FormControl>

                            <FormControl>
                                <FormLabel>Email</FormLabel>
                                <Input
                                    value={currentEmployee?.email || ''}
                                    onChange={(e) => setCurrentEmployee({ ...currentEmployee, email: e.target.value })}
                                />
                            </FormControl>

                            <FormControl>
                                <FormLabel>Teléfono</FormLabel>
                                <Input
                                    value={currentEmployee?.phone || ''}
                                    onChange={(e) => setCurrentEmployee({ ...currentEmployee, phone: e.target.value })}
                                />
                            </FormControl>

                            <FormControl>
                                <FormLabel>Especialidad</FormLabel>
                                <Select
                                    value={currentEmployee?.specialty || ''}
                                    onChange={(e) => setCurrentEmployee({ ...currentEmployee, specialty: e.target.value })}
                                >
                                    <option value="">Seleccionar...</option>
                                    <option value="Mantenimiento General">Mantenimiento General</option>
                                    <option value="Electricidad">Electricidad</option>
                                    <option value="Plomería">Plomería</option>
                                    <option value="Pintura">Pintura</option>
                                    <option value="Climatización">Climatización</option>
                                </Select>
                            </FormControl>

                            <FormControl>
                                <FormLabel>Contraseña {currentEmployee?.user_id && '(Dejar en blanco para no cambiar)'}</FormLabel>
                                <Input
                                    type="password"
                                    placeholder={currentEmployee?.user_id ? '••••••••' : 'Ingresar contraseña'}
                                    value={currentEmployee?.password || ''}
                                    onChange={(e) => setCurrentEmployee({ ...currentEmployee, password: e.target.value })}
                                />
                            </FormControl>
                        </Box>
                    </ModalBody>

                    <ModalFooter>
                        <Button variant="ghost" mr={3} onClick={onClose}>
                            Cancelar
                        </Button>
                        <Button colorScheme="brand" onClick={handleSave}>
                            Guardar
                        </Button>
                    </ModalFooter>
                </ModalContent>
            </Modal>
        </Box>
    );
};

export default GestionEmpleados;
