import { useState, useEffect } from 'react';
import {
  Box,
  Heading,
  Button,
  Table,
  Thead,
  Tbody,
  Tr,
  Th,
  Td,
  TableContainer,
  Modal,
  ModalOverlay,
  ModalContent,
  ModalHeader,
  ModalBody,
  ModalFooter,
  ModalCloseButton,
  useDisclosure,
  FormControl,
  FormLabel,
  Input,
  Avatar,
  useToast,
  Flex,
  HStack,
  IconButton,
  Icon,
  Text,
  Select,
  Grid,
  GridItem,
  VStack
} from '@chakra-ui/react';
import { BiPencil, BiTrash } from 'react-icons/bi';
import { FilePond, registerPlugin } from 'react-filepond';
import 'filepond/dist/filepond.min.css';
import FilePondPluginImagePreview from 'filepond-plugin-image-preview';
import 'filepond-plugin-image-preview/dist/filepond-plugin-image-preview.css';
import Layout from '../../components/Layout';
import { adminLinks } from './adminLinks';
import { mockProfesores } from '../../data/mockData';
import { estadosMexico } from '../../data/states';

registerPlugin(FilePondPluginImagePreview);

const GestionProfesores = () => {
  const [profesores, setProfesores] = useState(mockProfesores);
  const [profesorEditando, setProfesorEditando] = useState(null);
  const { isOpen, onOpen, onClose } = useDisclosure();
  const { isOpen: isEditOpen, onOpen: onEditOpen, onClose: onEditClose } = useDisclosure();
  const toast = useToast();

  // Form States
  const [formData, setFormData] = useState({
    nombre: '',
    email: '',
    especialidad: '',
    telefono: '',
    curp: '',
    direccion: {
      estado: '',
      municipio: '',
      cp: '',
      calle: '',
      numero: ''
    }
  });
  const [files, setFiles] = useState([]);
  const [municipiosDisponibles, setMunicipiosDisponibles] = useState([]);

  useEffect(() => {
    if (formData.direccion.estado) {
      const estadoObj = estadosMexico.find(e => e.nombre === formData.direccion.estado);
      setMunicipiosDisponibles(estadoObj ? estadoObj.municipios : []);
    } else {
      setMunicipiosDisponibles([]);
    }
  }, [formData.direccion.estado]);

  const handleCrear = () => {
    setProfesorEditando(null);
    setFormData({
      nombre: '',
      email: '',
      especialidad: '',
      telefono: '',
      curp: '',
      direccion: {
        estado: '',
        municipio: '',
        cp: '',
        calle: '',
        numero: ''
      }
    });
    setFiles([]);
    onOpen();
  };

  const handleEditar = (profesor) => {
    setProfesorEditando(profesor);
    setFormData({
      nombre: profesor.nombre,
      email: profesor.email,
      especialidad: profesor.especialidad,
      telefono: profesor.telefono || '',
      curp: profesor.curp || '',
      direccion: profesor.direccion || {
        estado: '',
        municipio: '',
        cp: '',
        calle: '',
        numero: ''
      }
    });
    setFiles(profesor.foto_url ? [{
      source: profesor.foto_url,
      options: {
        type: 'local'
      }
    }] : []);
    onEditOpen();
  };

  const handleInputChange = (e) => {
    const { name, value } = e.target;
    if (name.includes('.')) {
      const [parent, child] = name.split('.');
      setFormData(prev => ({
        ...prev,
        [parent]: {
          ...prev[parent],
          [child]: value
        }
      }));
    } else {
      setFormData(prev => ({
        ...prev,
        [name]: value
      }));
    }
  };

  const handleDesactivar = (id) => {
    toast({
      title: 'Profesor desactivado',
      description: 'El profesor ha sido marcado como inactivo',
      status: 'info',
      duration: 2000,
      isClosable: true,
    });
  };

  const handleGuardar = (e) => {
    e.preventDefault();

    const nuevoProfesor = {
      id: profesorEditando ? profesorEditando.id : profesores.length + 1,
      ...formData,
      foto_url: files.length > 0 ? URL.createObjectURL(files[0].file) : (profesorEditando?.foto_url || 'https://i.pravatar.cc/150?img=1')
    };

    if (profesorEditando) {
      setProfesores(profesores.map(p => p.id === profesorEditando.id ? nuevoProfesor : p));
    } else {
      setProfesores([...profesores, nuevoProfesor]);
    }

    toast({
      title: 'Profesor guardado',
      description: profesorEditando ? 'Profesor actualizado' : 'Profesor creado',
      status: 'success',
      duration: 2000,
      isClosable: true,
    });
    onClose();
    onEditClose();
  };

  return (
    <Layout links={adminLinks} userType="Administrador" userName="Admin User" userPhoto="https://i.pravatar.cc/150?img=68">
      <Box>

        {/* Instrucciones minimalistas */}
        <Text fontSize="sm" color="gray.600" mb={6}>
          Administra la información de los profesores. <strong>Los profesores desactivados no podrán recibir nuevas asignaciones</strong> de clases.
        </Text>

        {/* Tarjeta con estilo Facit */}
        <Box
          bg="rgba(255, 255, 255, 0.6)"
          backdropFilter="blur(20px) saturate(180%)"
          WebkitBackdropFilter="blur(20px) saturate(180%)"
          borderRadius="xl"
          boxShadow="0 8px 32px 0 rgba(0, 0, 0, 0.1)"
          border="1px solid"
          borderColor="rgba(255, 255, 255, 0.3)"
          overflow="hidden"
        >
          {/* Header de la Tarjeta */}
          <Flex justify="space-between" align="center" p={5} borderBottom="1px" borderColor="rgba(255, 255, 255, 0.2)">
            <Heading as="h3" size="md">Lista de Profesores</Heading>
            <Button colorScheme="blue" borderRadius="full" onClick={handleCrear}>Crear Nuevo</Button>
          </Flex>

          {/* Cuerpo de la Tabla */}
          <TableContainer>
            <Table variant="simple">
              <Thead>
                <Tr>
                  <Th textTransform="uppercase" fontSize="xs" color="gray.500" fontWeight="bold">Foto</Th>
                  <Th textTransform="uppercase" fontSize="xs" color="gray.500" fontWeight="bold">Nombre</Th>
                  <Th textTransform="uppercase" fontSize="xs" color="gray.500" fontWeight="bold">Especialidad</Th>
                  <Th textTransform="uppercase" fontSize="xs" color="gray.500" fontWeight="bold">Email</Th>
                  <Th textTransform="uppercase" fontSize="xs" color="gray.500" fontWeight="bold">Teléfono</Th>
                  <Th textTransform="uppercase" fontSize="xs" color="gray.500" fontWeight="bold">Acciones</Th>
                </Tr>
              </Thead>
              <Tbody>
                {profesores.map((profesor) => (
                  <Tr key={profesor.id} _hover={{ bg: "gray.50" }}>
                    <Td verticalAlign="middle"><Avatar size="sm" src={profesor.foto_url} name={profesor.nombre} /></Td>
                    <Td verticalAlign="middle">{profesor.nombre}</Td>
                    <Td verticalAlign="middle">{profesor.especialidad}</Td>
                    <Td verticalAlign="middle">{profesor.email}</Td>
                    <Td verticalAlign="middle">{profesor.telefono || 'N/A'}</Td>
                    <Td verticalAlign="middle">
                      <HStack spacing={2}>
                        <IconButton
                          icon={<Icon as={BiPencil} />}
                          variant="ghost"
                          colorScheme="green"
                          aria-label="Editar"
                          onClick={() => handleEditar(profesor)}
                        />
                        <IconButton
                          icon={<Icon as={BiTrash} />}
                          variant="ghost"
                          colorScheme="red"
                          aria-label="Desactivar"
                          onClick={() => handleDesactivar(profesor.id)}
                        />
                      </HStack>
                    </Td>
                  </Tr>
                ))}
              </Tbody>
            </Table>
          </TableContainer>
        </Box>

        <Modal isOpen={isOpen || isEditOpen} onClose={() => { onClose(); onEditClose(); }} isCentered size="6xl">
          <ModalOverlay bg="blackAlpha.600" backdropFilter="blur(4px)" />
          <ModalContent
            borderRadius="xl"
            bg="rgba(255, 255, 255, 0.95)"
            backdropFilter="blur(20px) saturate(180%)"
            boxShadow="0 8px 32px 0 rgba(0, 0, 0, 0.2)"
          >
            <ModalHeader>{profesorEditando ? 'Editar Profesor' : 'Crear Nuevo Profesor'}</ModalHeader>
            <ModalCloseButton />
            <form onSubmit={handleGuardar}>
              <ModalBody p={6}>
                <Flex gap={8} flexDirection={{ base: 'column', lg: 'row' }}>
                  {/* Columna Izquierda: Foto */}
                  <Box w={{ base: '100%', lg: '250px' }} flexShrink={0}>
                    <FormLabel textAlign="center" fontWeight="bold" mb={4} color="gray.700">Fotografía</FormLabel>
                    <Box
                      w="220px"
                      h="220px"
                      mx="auto"
                      borderRadius="full"
                      overflow="hidden"
                      boxShadow="0 4px 12px rgba(0,0,0,0.1)"
                      bg="gray.50"
                      border="2px solid"
                      borderColor="gray.100"
                    >
                      <FilePond
                        files={files}
                        onupdatefiles={setFiles}
                        allowMultiple={false}
                        maxFiles={1}
                        name="files"
                        labelIdle='<span class="filepond--label-action">Subir Foto</span>'
                        acceptedFileTypes={['image/*']}
                        stylePanelLayout="compact circle"
                        styleLoadIndicatorPosition="center bottom"
                        styleProgressIndicatorPosition="right bottom"
                        styleButtonRemoveItemPosition="left bottom"
                        styleButtonProcessItemPosition="right bottom"
                      />
                    </Box>
                    <Text fontSize="xs" color="gray.500" textAlign="center" mt={3}>
                      Arrastra una imagen o haz clic para seleccionar
                    </Text>
                  </Box>

                  {/* Columna Derecha: Datos */}
                  <Box flex={1}>
                    <VStack spacing={6} align="stretch">

                      {/* Sección Personal */}
                      <Box bg="white" p={4} borderRadius="lg" border="1px solid" borderColor="gray.100" shadow="sm">
                        <Heading size="sm" mb={4} color="blue.600" borderBottom="1px solid" borderColor="gray.100" pb={2} display="flex" alignItems="center">
                          <Icon as={BiPencil} mr={2} /> Información Personal
                        </Heading>
                        <Grid templateColumns={{ base: '1fr', md: 'repeat(2, 1fr)' }} gap={4}>
                          <GridItem colSpan={{ base: 1, md: 2 }}>
                            <FormControl isRequired>
                              <FormLabel fontSize="sm" fontWeight="600" color="gray.700">Nombre Completo</FormLabel>
                              <Input name="nombre" value={formData.nombre} onChange={handleInputChange} placeholder="Ej. Juan Pérez" borderRadius="md" />
                            </FormControl>
                          </GridItem>
                          <GridItem>
                            <FormControl isRequired>
                              <FormLabel fontSize="sm" fontWeight="600" color="gray.700">Email</FormLabel>
                              <Input name="email" type="email" value={formData.email} onChange={handleInputChange} placeholder="correo@ejemplo.com" borderRadius="md" />
                            </FormControl>
                          </GridItem>
                          <GridItem>
                            <FormControl isRequired>
                              <FormLabel fontSize="sm" fontWeight="600" color="gray.700">Especialidad</FormLabel>
                              <Input name="especialidad" value={formData.especialidad} onChange={handleInputChange} placeholder="Ej. Piano" borderRadius="md" />
                            </FormControl>
                          </GridItem>
                          <GridItem>
                            <FormControl>
                              <FormLabel fontSize="sm" fontWeight="600" color="gray.700">Teléfono</FormLabel>
                              <Input name="telefono" value={formData.telefono} onChange={handleInputChange} placeholder="555-555-5555" borderRadius="md" />
                            </FormControl>
                          </GridItem>
                          <GridItem>
                            <FormControl>
                              <FormLabel fontSize="sm" fontWeight="600" color="gray.700">CURP</FormLabel>
                              <Input name="curp" value={formData.curp} onChange={handleInputChange} placeholder="ABCD123456..." borderRadius="md" />
                            </FormControl>
                          </GridItem>
                        </Grid>
                      </Box>

                      {/* Sección Dirección */}
                      <Box bg="white" p={4} borderRadius="lg" border="1px solid" borderColor="gray.100" shadow="sm">
                        <Heading size="sm" mb={4} color="blue.600" borderBottom="1px solid" borderColor="gray.100" pb={2} display="flex" alignItems="center">
                          <Icon as={BiPencil} mr={2} /> Dirección
                        </Heading>
                        <Grid templateColumns={{ base: '1fr', md: 'repeat(2, 1fr)' }} gap={4}>
                          <GridItem>
                            <FormControl>
                              <FormLabel fontSize="sm" fontWeight="600" color="gray.700">Estado</FormLabel>
                              <Select name="direccion.estado" value={formData.direccion.estado} onChange={handleInputChange} placeholder="Selecciona estado" borderRadius="md">
                                {estadosMexico.map(e => (
                                  <option key={e.nombre} value={e.nombre}>{e.nombre}</option>
                                ))}
                              </Select>
                            </FormControl>
                          </GridItem>
                          <GridItem>
                            <FormControl>
                              <FormLabel fontSize="sm" fontWeight="600" color="gray.700">Municipio</FormLabel>
                              <Select name="direccion.municipio" value={formData.direccion.municipio} onChange={handleInputChange} placeholder="Selecciona municipio" borderRadius="md">
                                {municipiosDisponibles.map(m => (
                                  <option key={m} value={m}>{m}</option>
                                ))}
                              </Select>
                            </FormControl>
                          </GridItem>
                          <GridItem colSpan={{ base: 1, md: 2 }}>
                            <FormControl>
                              <FormLabel fontSize="sm" fontWeight="600" color="gray.700">Calle y Número</FormLabel>
                              <HStack>
                                <Input name="direccion.calle" value={formData.direccion.calle} onChange={handleInputChange} placeholder="Calle" flex={2} borderRadius="md" />
                                <Input name="direccion.numero" value={formData.direccion.numero} onChange={handleInputChange} placeholder="No." flex={1} borderRadius="md" />
                              </HStack>
                            </FormControl>
                          </GridItem>
                          <GridItem>
                            <FormControl>
                              <FormLabel fontSize="sm" fontWeight="600" color="gray.700">Código Postal</FormLabel>
                              <Input name="direccion.cp" value={formData.direccion.cp} onChange={handleInputChange} placeholder="00000" borderRadius="md" />
                            </FormControl>
                          </GridItem>
                        </Grid>
                      </Box>

                    </VStack>
                  </Box>
                </Flex>
              </ModalBody>
              <ModalFooter>
                <Button variant="ghost" mr={3} onClick={() => { onClose(); onEditClose(); }}>Cancelar</Button>
                <Button colorScheme="blue" type="submit">Guardar</Button>
              </ModalFooter>
            </form>
          </ModalContent>
        </Modal>
      </Box>
    </Layout>
  );
};

export default GestionProfesores;

