
import { useState, useEffect } from 'react';
import {
  Box,
  Heading,
  Button,
  Table,
  Thead,
  Tbody,
  Tr,
  Th,
  Td,
  TableContainer,
  Modal,
  ModalOverlay,
  ModalContent,
  ModalHeader,
  ModalBody,
  ModalFooter,
  ModalCloseButton,
  useDisclosure,
  FormControl,
  FormLabel,
  Input,
  Select,
  useToast,
  Flex,
  HStack,
  IconButton,
  Icon,
  Text,
  Avatar,
  Grid,
  GridItem,
  VStack
} from '@chakra-ui/react';
import { BiPencil, BiTrash, BiPlus } from 'react-icons/bi';
import { FilePond, registerPlugin } from 'react-filepond';
import 'filepond/dist/filepond.min.css';
import FilePondPluginImagePreview from 'filepond-plugin-image-preview';
import 'filepond-plugin-image-preview/dist/filepond-plugin-image-preview.css';
import Layout from '../../components/Layout';
import { estadosMexico } from '../../data/states';

registerPlugin(FilePondPluginImagePreview);

const adminLinks = [
  { nombre: 'Dashboard', path: '/admin' },
  { nombre: 'Usuarios', path: '/admin/usuarios' },
  { nombre: 'Profesores', path: '/admin/profesores' },
  { nombre: 'Alumnos', path: '/admin/alumnos' },
  { nombre: 'Materias', path: '/admin/materias' },
  { nombre: 'Asignar Clases', path: '/admin/asignar-clases' },
  { nombre: 'Calendario Global', path: '/admin/calendario' },
  { nombre: 'Consultar Bitácoras', path: '/admin/bitacoras' },
  { nombre: 'Calendario Pagos', path: '/admin/calendario-pagos' },
  { nombre: 'Validar Pagos', path: '/admin/validar-pagos' },
  { nombre: 'Notificaciones', path: '/admin/notificaciones' }
];

const mockUsuarios = [
  { id: 1, nombre: 'Admin User', email: 'admin@arteschool.com', tipo: 'admin', activo: true, telefono: '555-000-0000', foto_url: 'https://i.pravatar.cc/150?img=68' },
  { id: 2, nombre: 'María González', email: 'maria@arteschool.com', tipo: 'profesor', activo: true, telefono: '555-123-4567', foto_url: 'https://i.pravatar.cc/150?img=1' },
  { id: 3, nombre: 'Roberto López', email: 'roberto@email.com', tipo: 'padre', activo: true, telefono: '555-111-2222', foto_url: 'https://i.pravatar.cc/150?img=12' },
  { id: 4, nombre: 'Sofía López', email: 'sofia@arteschool.com', tipo: 'alumno', activo: true, telefono: '555-333-4444', foto_url: 'https://i.pravatar.cc/150?img=9' }
];

const GestionUsuarios = () => {
  const [usuarios, setUsuarios] = useState(mockUsuarios);
  const [usuarioEditando, setUsuarioEditando] = useState(null);
  const { isOpen, onOpen, onClose } = useDisclosure();
  const { isOpen: isEditOpen, onOpen: onEditOpen, onClose: onEditClose } = useDisclosure();
  const toast = useToast();

  // Form States
  const [formData, setFormData] = useState({
    nombre: '',
    email: '',
    tipo: '',
    telefono: '',
    curp: '',
    direccion: {
      estado: '',
      municipio: '',
      cp: '',
      calle: '',
      numero: ''
    }
  });
  const [files, setFiles] = useState([]);
  const [municipiosDisponibles, setMunicipiosDisponibles] = useState([]);

  useEffect(() => {
    if (formData.direccion.estado) {
      const estadoObj = estadosMexico.find(e => e.nombre === formData.direccion.estado);
      setMunicipiosDisponibles(estadoObj ? estadoObj.municipios : []);
    } else {
      setMunicipiosDisponibles([]);
    }
  }, [formData.direccion.estado]);

  const handleCrear = () => {
    setUsuarioEditando(null);
    setFormData({
      nombre: '',
      email: '',
      tipo: '',
      telefono: '',
      curp: '',
      direccion: {
        estado: '',
        municipio: '',
        cp: '',
        calle: '',
        numero: ''
      }
    });
    setFiles([]);
    onOpen();
  };

  const handleEditar = (usuario) => {
    setUsuarioEditando(usuario);
    setFormData({
      nombre: usuario.nombre,
      email: usuario.email,
      tipo: usuario.tipo,
      telefono: usuario.telefono || '',
      curp: usuario.curp || '',
      direccion: usuario.direccion || {
        estado: '',
        municipio: '',
        cp: '',
        calle: '',
        numero: ''
      }
    });
    setFiles(usuario.foto_url ? [{
      source: usuario.foto_url,
      options: {
        type: 'local'
      }
    }] : []);
    onEditOpen();
  };

  const handleInputChange = (e) => {
    const { name, value } = e.target;
    if (name.includes('.')) {
      const [parent, child] = name.split('.');
      setFormData(prev => ({
        ...prev,
        [parent]: {
          ...prev[parent],
          [child]: value
        }
      }));
    } else {
      setFormData(prev => ({
        ...prev,
        [name]: value
      }));
    }
  };

  const handleDesactivar = (id) => {
    setUsuarios(usuarios.map(u => u.id === id ? { ...u, activo: !u.activo } : u));
    toast({
      title: 'Usuario actualizado',
      description: 'El estado del usuario ha sido modificado',
      status: 'success',
      duration: 2000,
      isClosable: true,
    });
  };

  const handleGuardar = (e) => {
    e.preventDefault();

    const nuevoUsuario = {
      id: usuarioEditando ? usuarioEditando.id : usuarios.length + 1,
      ...formData,
      activo: usuarioEditando ? usuarioEditando.activo : true,
      foto_url: files.length > 0 ? URL.createObjectURL(files[0].file) : (usuarioEditando?.foto_url || 'https://i.pravatar.cc/150?img=1')
    };

    if (usuarioEditando) {
      setUsuarios(usuarios.map(u => u.id === usuarioEditando.id ? nuevoUsuario : u));
    } else {
      setUsuarios([...usuarios, nuevoUsuario]);
    }

    toast({
      title: 'Usuario guardado',
      description: usuarioEditando ? 'Usuario actualizado' : 'Usuario creado',
      status: 'success',
      duration: 2000,
      isClosable: true,
    });
    onClose();
    onEditClose();
  };

  return (
    <Layout links={adminLinks} userType="Administrador" userName="Admin User" userPhoto="https://i.pravatar.cc/150?img=68">
      <Box>

        {/* Instrucciones minimalistas */}
        <Text fontSize="sm" color="gray.600" mb={6}>
          Administra todos los usuarios del sistema. <strong>Los usuarios inactivos perderán acceso</strong> a la plataforma.
        </Text>

        {/* ¡LA TARJETA! Aquí está la magia */}
        <Box
          bg="rgba(255, 255, 255, 0.6)"
          backdropFilter="blur(20px) saturate(180%)"
          WebkitBackdropFilter="blur(20px) saturate(180%)"
          borderRadius="xl"
          boxShadow="0 8px 32px 0 rgba(0, 0, 0, 0.1)"
          border="1px solid"
          borderColor="rgba(255, 255, 255, 0.3)"
          overflow="hidden"
        >

          {/* Header de la Tarjeta (como en "Facit") */}
          <Flex justify="space-between" align="center" p={5} borderBottom="1px" borderColor="rgba(255, 255, 255, 0.2)">
            <Heading as="h3" size="md">Lista de Usuarios</Heading>
            {/* Botón "Píldora" */}
            <Button
              colorScheme="blue"
              borderRadius="full"
              leftIcon={<Icon as={BiPlus} />}
              onClick={handleCrear}
            >
              Crear Nuevo
            </Button>
          </Flex>

          {/* La tabla va DENTRO del cuerpo de la tarjeta */}
          <TableContainer>
            {/* ¡Quitar 'variant="striped"'! */}
            <Table variant="simple">
              <Thead>
                {/* Estilo de cabecera moderno */}
                <Tr>
                  <Th textTransform="uppercase" fontSize="xs" color="gray.500">Foto</Th>
                  <Th textTransform="uppercase" fontSize="xs" color="gray.500">ID</Th>
                  <Th textTransform="uppercase" fontSize="xs" color="gray.500">Nombre</Th>
                  <Th textTransform="uppercase" fontSize="xs" color="gray.500">Email</Th>
                  <Th textTransform="uppercase" fontSize="xs" color="gray.500">Teléfono</Th>
                  <Th textTransform="uppercase" fontSize="xs" color="gray.500">Tipo</Th>
                  <Th textTransform="uppercase" fontSize="xs" color="gray.500">Estado</Th>
                  <Th textTransform="uppercase" fontSize="xs" color="gray.500">Acciones</Th>
                </Tr>
              </Thead>
              <Tbody>
                {usuarios.map((usuario) => (
                  <Tr key={usuario.id} _hover={{ bg: "gray.50" }}>
                    <Td verticalAlign="middle"><Avatar size="sm" src={usuario.foto_url} name={usuario.nombre} /></Td>
                    <Td verticalAlign="middle">{usuario.id}</Td>
                    <Td verticalAlign="middle">{usuario.nombre}</Td>
                    <Td verticalAlign="middle">{usuario.email}</Td>
                    <Td verticalAlign="middle">{usuario.telefono || 'N/A'}</Td>
                    <Td verticalAlign="middle">{usuario.tipo}</Td>
                    <Td verticalAlign="middle">{usuario.activo ? 'Activo' : 'Inactivo'}</Td>
                    <Td>
                      {/* ¡Usar IconButtons! */}
                      <HStack spacing={2}>
                        <IconButton
                          icon={<Icon as={BiPencil} />}
                          variant="ghost"
                          colorScheme="green"
                          aria-label="Editar"
                          onClick={() => handleEditar(usuario)}
                        />
                        <IconButton
                          icon={<Icon as={BiTrash} />}
                          variant="ghost"
                          colorScheme="red"
                          aria-label={usuario.activo ? "Desactivar" : "Activar"}
                          onClick={() => handleDesactivar(usuario.id)}
                        />
                      </HStack>
                    </Td>
                  </Tr>
                ))}
              </Tbody>
            </Table>
          </TableContainer>
        </Box>
      </Box>

      <Modal isOpen={isOpen || isEditOpen} onClose={() => { onClose(); onEditClose(); }} isCentered size="6xl">
        <ModalOverlay bg="blackAlpha.600" backdropFilter="blur(4px)" />
        <ModalContent
          borderRadius="xl"
          bg="rgba(255, 255, 255, 0.95)"
          backdropFilter="blur(20px) saturate(180%)"
          boxShadow="0 8px 32px 0 rgba(0, 0, 0, 0.2)"
        >
          <ModalHeader>{usuarioEditando ? 'Editar Usuario' : 'Crear Nuevo Usuario'}</ModalHeader>
          <ModalCloseButton />
          <form onSubmit={handleGuardar}>
            <ModalBody p={6}>
              <Flex gap={8} flexDirection={{ base: 'column', lg: 'row' }}>
                {/* Columna Izquierda: Foto */}
                <Box w={{ base: '100%', lg: '250px' }} flexShrink={0}>
                  <FormLabel textAlign="center" fontWeight="bold" mb={4} color="gray.700">Fotografía</FormLabel>
                  <Box
                    w="220px"
                    h="220px"
                    mx="auto"
                    borderRadius="full"
                    overflow="hidden"
                    boxShadow="0 4px 12px rgba(0,0,0,0.1)"
                    bg="gray.50"
                    border="2px solid"
                    borderColor="gray.100"
                  >
                    <FilePond
                      files={files}
                      onupdatefiles={setFiles}
                      allowMultiple={false}
                      maxFiles={1}
                      name="files"
                      labelIdle='<span class="filepond--label-action">Subir Foto</span>'
                      acceptedFileTypes={['image/*']}
                      stylePanelLayout="compact circle"
                      styleLoadIndicatorPosition="center bottom"
                      styleProgressIndicatorPosition="right bottom"
                      styleButtonRemoveItemPosition="left bottom"
                      styleButtonProcessItemPosition="right bottom"
                    />
                  </Box>
                  <Text fontSize="xs" color="gray.500" textAlign="center" mt={3}>
                    Arrastra una imagen o haz clic para seleccionar
                  </Text>
                </Box>

                {/* Columna Derecha: Datos */}
                <Box flex={1}>
                  <VStack spacing={6} align="stretch">

                    {/* Sección Personal */}
                    <Box bg="white" p={4} borderRadius="lg" border="1px solid" borderColor="gray.100" shadow="sm">
                      <Heading size="sm" mb={4} color="blue.600" borderBottom="1px solid" borderColor="gray.100" pb={2} display="flex" alignItems="center">
                        <Icon as={BiPencil} mr={2} /> Información Personal
                      </Heading>
                      <Grid templateColumns={{ base: '1fr', md: 'repeat(2, 1fr)' }} gap={4}>
                        <GridItem colSpan={{ base: 1, md: 2 }}>
                          <FormControl isRequired>
                            <FormLabel fontSize="sm" fontWeight="600" color="gray.700">Nombre Completo</FormLabel>
                            <Input name="nombre" value={formData.nombre} onChange={handleInputChange} placeholder="Ej. Juan Pérez" borderRadius="md" />
                          </FormControl>
                        </GridItem>
                        <GridItem>
                          <FormControl isRequired>
                            <FormLabel fontSize="sm" fontWeight="600" color="gray.700">Email</FormLabel>
                            <Input name="email" type="email" value={formData.email} onChange={handleInputChange} placeholder="correo@ejemplo.com" borderRadius="md" />
                          </FormControl>
                        </GridItem>
                        <GridItem>
                          <FormControl isRequired>
                            <FormLabel fontSize="sm" fontWeight="600" color="gray.700">Tipo de Usuario</FormLabel>
                            <Select name="tipo" value={formData.tipo} onChange={handleInputChange} placeholder="Selecciona tipo" borderRadius="md">
                              <option value="admin">Administrador</option>
                              <option value="profesor">Profesor</option>
                              <option value="padre">Padre/Tutor</option>
                              <option value="alumno">Alumno</option>
                            </Select>
                          </FormControl>
                        </GridItem>
                        <GridItem>
                          <FormControl>
                            <FormLabel fontSize="sm" fontWeight="600" color="gray.700">Teléfono</FormLabel>
                            <Input name="telefono" value={formData.telefono} onChange={handleInputChange} placeholder="555-555-5555" borderRadius="md" />
                          </FormControl>
                        </GridItem>
                        <GridItem colSpan={{ base: 1, md: 2 }}>
                          <FormControl>
                            <FormLabel fontSize="sm" fontWeight="600" color="gray.700">CURP</FormLabel>
                            <Input name="curp" value={formData.curp} onChange={handleInputChange} placeholder="ABCD123456..." borderRadius="md" />
                          </FormControl>
                        </GridItem>
                      </Grid>
                    </Box>

                    {/* Sección Dirección */}
                    <Box bg="white" p={4} borderRadius="lg" border="1px solid" borderColor="gray.100" shadow="sm">
                      <Heading size="sm" mb={4} color="blue.600" borderBottom="1px solid" borderColor="gray.100" pb={2} display="flex" alignItems="center">
                        <Icon as={BiPencil} mr={2} /> Dirección
                      </Heading>
                      <Grid templateColumns={{ base: '1fr', md: 'repeat(2, 1fr)' }} gap={4}>
                        <GridItem>
                          <FormControl>
                            <FormLabel fontSize="sm" fontWeight="600" color="gray.700">Estado</FormLabel>
                            <Select name="direccion.estado" value={formData.direccion.estado} onChange={handleInputChange} placeholder="Selecciona estado" borderRadius="md">
                              {estadosMexico.map(e => (
                                <option key={e.nombre} value={e.nombre}>{e.nombre}</option>
                              ))}
                            </Select>
                          </FormControl>
                        </GridItem>
                        <GridItem>
                          <FormControl>
                            <FormLabel fontSize="sm" fontWeight="600" color="gray.700">Municipio</FormLabel>
                            <Select name="direccion.municipio" value={formData.direccion.municipio} onChange={handleInputChange} placeholder="Selecciona municipio" borderRadius="md">
                              {municipiosDisponibles.map(m => (
                                <option key={m} value={m}>{m}</option>
                              ))}
                            </Select>
                          </FormControl>
                        </GridItem>
                        <GridItem colSpan={{ base: 1, md: 2 }}>
                          <FormControl>
                            <FormLabel fontSize="sm" fontWeight="600" color="gray.700">Calle y Número</FormLabel>
                            <HStack>
                              <Input name="direccion.calle" value={formData.direccion.calle} onChange={handleInputChange} placeholder="Calle" flex={2} borderRadius="md" />
                              <Input name="direccion.numero" value={formData.direccion.numero} onChange={handleInputChange} placeholder="No." flex={1} borderRadius="md" />
                            </HStack>
                          </FormControl>
                        </GridItem>
                        <GridItem>
                          <FormControl>
                            <FormLabel fontSize="sm" fontWeight="600" color="gray.700">Código Postal</FormLabel>
                            <Input name="direccion.cp" value={formData.direccion.cp} onChange={handleInputChange} placeholder="00000" borderRadius="md" />
                          </FormControl>
                        </GridItem>
                      </Grid>
                    </Box>

                  </VStack>
                </Box>
              </Flex>
            </ModalBody>
            <ModalFooter>
              <Button variant="ghost" mr={3} onClick={() => { onClose(); onEditClose(); }}>Cancelar</Button>
              <Button colorScheme="blue" type="submit">Guardar</Button>
            </ModalFooter>
          </form>
        </ModalContent>
      </Modal>
    </Layout>
  );
};

export default GestionUsuarios;

