import { useState, useMemo } from 'react';
import { useNavigate } from 'react-router-dom';
import {
    Box,
    VStack,
    HStack,
    Text,
    Heading,
    Tabs,
    TabList,
    TabPanels,
    Tab,
    TabPanel,
    Badge,
    Button,
    Input,
    InputGroup,
    InputLeftElement,
    Select,
    Grid,
    Icon,
    Flex,
    useToast,
    SlideFade,
    IconButton,
    Tooltip
} from '@chakra-ui/react';
import {
    FiSearch,
    FiFilter,
    FiCheckCircle,
    FiMail,
    FiArchive,
    FiX,
    FiInbox,
    FiDollarSign,
    FiUserPlus,
    FiAlertCircle,
    FiSend
} from 'react-icons/fi';
import Layout from '../../components/Layout';
import NotificacionCard from '../../components/NotificacionCard';
import { adminLinks } from './adminLinks';

const initialNotificaciones = [
    {
        id: 1,
        titulo: 'Nuevo pago recibido',
        mensaje: 'El alumno Juan Pérez ha realizado un pago de $1,500.00 por concepto de colegiatura. Requiere validación.',
        fecha: '2024-01-20',
        leida: false,
        tipo: 'pago',
        archivada: false
    },
    {
        id: 2,
        titulo: 'Solicitud de inscripción',
        html: `
      <p>Se ha recibido una nueva solicitud de inscripción.</p>
      <div style="background: #E0F2FE; padding: 12px; border-radius: 8px; margin: 16px 0; border-left: 4px solid #0EA5E9;">
        <p style="margin: 0; color: #0C4A6E;"><strong>Alumno:</strong> Ana García</p>
        <p style="margin: 0; color: #0C4A6E;"><strong>Materia:</strong> Pintura al Óleo</p>
      </div>
      <p>Revisa la documentación adjunta para aprobar el ingreso.</p>
    `,
        fecha: '2024-01-19',
        leida: false,
        tipo: 'sistema',
        archivada: false
    },
    {
        id: 3,
        titulo: 'Reporte de asistencia bajo',
        mensaje: 'El sistema ha detectado una asistencia menor al 80% para el grupo de Dibujo Avanzado del profesor Roberto.',
        fecha: '2024-01-18',
        leida: true,
        tipo: 'alerta',
        archivada: false
    },
    {
        id: 4,
        titulo: 'Mantenimiento del sistema programado',
        mensaje: 'Se realizará un mantenimiento preventivo de la plataforma el día Domingo 25 de Enero a las 02:00 AM.',
        fecha: '2024-01-17',
        leida: false,
        tipo: 'sistema',
        archivada: false
    },
    {
        id: 5,
        titulo: 'Nuevo profesor registrado',
        mensaje: 'Se ha completado el registro del profesor Carlos Ruiz. Ya puede asignarse a clases.',
        fecha: '2024-01-16',
        leida: true,
        tipo: 'usuario',
        archivada: true
    }
];

const NotificacionesAdmin = () => {
    const [notificaciones, setNotificaciones] = useState(initialNotificaciones);
    const [searchTerm, setSearchTerm] = useState('');
    const [filterTipo, setFilterTipo] = useState('todas');
    const toast = useToast();
    const navigate = useNavigate();

    // Estadísticas
    const stats = useMemo(() => {
        const noLeidas = notificaciones.filter(n => !n.leida && !n.archivada).length;
        const archivadas = notificaciones.filter(n => n.archivada).length;
        const total = notificaciones.length;
        return { noLeidas, archivadas, total };
    }, [notificaciones]);

    // Filtrar notificaciones
    const filteredNotificaciones = useMemo(() => {
        return notificaciones.filter(notif => {
            const matchesSearch = notif.titulo.toLowerCase().includes(searchTerm.toLowerCase()) ||
                (notif.mensaje && notif.mensaje.toLowerCase().includes(searchTerm.toLowerCase()));
            const matchesTipo = filterTipo === 'todas' || notif.tipo === filterTipo;
            return matchesSearch && matchesTipo;
        });
    }, [notificaciones, searchTerm, filterTipo]);

    // Separar por tabs
    const noLeidas = filteredNotificaciones.filter(n => !n.leida && !n.archivada);
    const archivadas = filteredNotificaciones.filter(n => n.archivada);
    const todas = filteredNotificaciones.filter(n => !n.archivada);

    // Marcar como leída
    const marcarComoLeida = (id) => {
        setNotificaciones(prev =>
            prev.map(n => n.id === id ? { ...n, leida: true } : n)
        );
    };

    // Marcar todas como leídas
    const marcarTodasLeidas = () => {
        setNotificaciones(prev =>
            prev.map(n => ({ ...n, leida: true }))
        );
        toast({
            title: 'Notificaciones marcadas',
            description: 'Todas las notificaciones han sido marcadas como leídas',
            status: 'success',
            duration: 3000,
            isClosable: true,
        });
    };

    // Archivar notificación
    const archivarNotificacion = (id) => {
        setNotificaciones(prev =>
            prev.map(n => n.id === id ? { ...n, archivada: true, leida: true } : n)
        );
        toast({
            title: 'Notificación archivada',
            status: 'info',
            duration: 2000,
            isClosable: true,
        });
    };

    const limpiarFiltros = () => {
        setSearchTerm('');
        setFilterTipo('todas');
    };

    const EmptyState = ({ icon, title, description }) => (
        <Box
            textAlign="center"
            py={16}
            px={6}
            bg="white"
            borderRadius="2xl"
            boxShadow="sm"
            border="1px dashed"
            borderColor="gray.200"
            display="flex"
            flexDirection="column"
            alignItems="center"
            justifyContent="center"
            minH="300px"
        >
            <Box
                p={4}
                bg="gray.50"
                borderRadius="full"
                mb={4}
                color="gray.400"
            >
                <Icon as={icon} boxSize={10} />
            </Box>
            <Text fontSize="xl" fontWeight="bold" color="gray.700" mb={2}>
                {title}
            </Text>
            <Text color="gray.500" maxW="sm">
                {description}
            </Text>
            {(searchTerm || filterTipo !== 'todas') && (
                <Button
                    mt={6}
                    variant="outline"
                    colorScheme="blue"
                    size="sm"
                    onClick={limpiarFiltros}
                    leftIcon={<FiX />}
                >
                    Limpiar filtros
                </Button>
            )}
        </Box>
    );

    return (
        <Layout links={adminLinks} userType="Administrador" userName="Admin User">
            <Box maxW="1200px" mx="auto">
                {/* Header con estadísticas */}
                <Box mb={8}>
                    <Flex justify="space-between" align="center" mb={6}>
                        <Box>
                            <Heading size="lg" mb={2} bgGradient="linear(to-r, blue.500, teal.400)" bgClip="text">
                                📬 Centro de Notificaciones
                            </Heading>
                            <Text color="gray.600">Gestiona alertas, pagos y avisos del sistema</Text>
                        </Box>
                        <Button
                            leftIcon={<FiSend />}
                            colorScheme="blue"
                            onClick={() => navigate('/admin/notificaciones/enviar')}
                            size="md"
                            boxShadow="md"
                            _hover={{ transform: 'translateY(-2px)', boxShadow: 'lg' }}
                        >
                            Enviar Notificación
                        </Button>
                    </Flex>

                    <Grid templateColumns={{ base: '1fr', md: 'repeat(3, 1fr)' }} gap={4} mb={6}>
                        {/* Stat: No leídas */}
                        <Box
                            bg="linear-gradient(135deg, #667eea 0%, #764ba2 100%)"
                            p={5}
                            borderRadius="2xl"
                            color="white"
                            boxShadow="lg"
                            position="relative"
                            overflow="hidden"
                            transition="transform 0.2s"
                            _hover={{ transform: 'translateY(-2px)' }}
                        >
                            <Box position="absolute" top={0} right={0} p={4} opacity={0.2}>
                                <Icon as={FiMail} boxSize={24} transform="rotate(-15deg)" />
                            </Box>
                            <HStack spacing={4} position="relative" zIndex={1}>
                                <Box p={3} bg="whiteAlpha.200" borderRadius="xl">
                                    <Icon as={FiMail} boxSize={6} />
                                </Box>
                                <Box>
                                    <Text fontSize="3xl" fontWeight="800" lineHeight="1">{stats.noLeidas}</Text>
                                    <Text fontSize="sm" fontWeight="medium" opacity={0.9}>No leídas</Text>
                                </Box>
                            </HStack>
                        </Box>

                        {/* Stat: Total */}
                        <Box
                            bg="linear-gradient(135deg, #f093fb 0%, #f5576c 100%)"
                            p={5}
                            borderRadius="2xl"
                            color="white"
                            boxShadow="lg"
                            position="relative"
                            overflow="hidden"
                            transition="transform 0.2s"
                            _hover={{ transform: 'translateY(-2px)' }}
                        >
                            <Box position="absolute" top={0} right={0} p={4} opacity={0.2}>
                                <Icon as={FiInbox} boxSize={24} transform="rotate(-15deg)" />
                            </Box>
                            <HStack spacing={4} position="relative" zIndex={1}>
                                <Box p={3} bg="whiteAlpha.200" borderRadius="xl">
                                    <Icon as={FiInbox} boxSize={6} />
                                </Box>
                                <Box>
                                    <Text fontSize="3xl" fontWeight="800" lineHeight="1">{stats.total}</Text>
                                    <Text fontSize="sm" fontWeight="medium" opacity={0.9}>Total recibidas</Text>
                                </Box>
                            </HStack>
                        </Box>

                        {/* Stat: Archivadas */}
                        <Box
                            bg="linear-gradient(135deg, #4facfe 0%, #00f2fe 100%)"
                            p={5}
                            borderRadius="2xl"
                            color="white"
                            boxShadow="lg"
                            position="relative"
                            overflow="hidden"
                            transition="transform 0.2s"
                            _hover={{ transform: 'translateY(-2px)' }}
                        >
                            <Box position="absolute" top={0} right={0} p={4} opacity={0.2}>
                                <Icon as={FiArchive} boxSize={24} transform="rotate(-15deg)" />
                            </Box>
                            <HStack spacing={4} position="relative" zIndex={1}>
                                <Box p={3} bg="whiteAlpha.200" borderRadius="xl">
                                    <Icon as={FiArchive} boxSize={6} />
                                </Box>
                                <Box>
                                    <Text fontSize="3xl" fontWeight="800" lineHeight="1">{stats.archivadas}</Text>
                                    <Text fontSize="sm" fontWeight="medium" opacity={0.9}>Archivadas</Text>
                                </Box>
                            </HStack>
                        </Box>
                    </Grid>

                    {/* Filtros y búsqueda */}
                    <Flex
                        direction={{ base: 'column', md: 'row' }}
                        gap={4}
                        mb={6}
                        bg="white"
                        p={4}
                        borderRadius="2xl"
                        boxShadow="sm"
                        border="1px solid"
                        borderColor="gray.100"
                        alignItems="center"
                    >
                        <InputGroup flex={1}>
                            <InputLeftElement pointerEvents="none">
                                <Icon as={FiSearch} color="gray.400" />
                            </InputLeftElement>
                            <Input
                                placeholder="Buscar por título o contenido..."
                                value={searchTerm}
                                onChange={(e) => setSearchTerm(e.target.value)}
                                bg="gray.50"
                                border="none"
                                _focus={{ bg: 'white', boxShadow: 'outline' }}
                                borderRadius="lg"
                            />
                            {searchTerm && (
                                <InputGroup size="sm" width="auto" position="absolute" right={2} top={2} zIndex={2}>
                                    <IconButton
                                        size="xs"
                                        aria-label="Limpiar búsqueda"
                                        icon={<FiX />}
                                        onClick={() => setSearchTerm('')}
                                        variant="ghost"
                                        isRound
                                    />
                                </InputGroup>
                            )}
                        </InputGroup>

                        <HStack spacing={3} w={{ base: 'full', md: 'auto' }}>
                            <Select
                                value={filterTipo}
                                onChange={(e) => setFilterTipo(e.target.value)}
                                w={{ base: 'full', md: '200px' }}
                                bg="gray.50"
                                border="none"
                                borderRadius="lg"
                                icon={<FiFilter />}
                                _focus={{ boxShadow: 'outline' }}
                            >
                                <option value="todas">Todos los tipos</option>
                                <option value="pago">💰 Pagos</option>
                                <option value="sistema">🖥️ Sistema</option>
                                <option value="alerta">⚠️ Alertas</option>
                                <option value="usuario">👤 Usuarios</option>
                            </Select>

                            {stats.noLeidas > 0 && (
                                <Tooltip label="Marcar todas como leídas">
                                    <IconButton
                                        icon={<FiCheckCircle />}
                                        colorScheme="blue"
                                        variant="ghost"
                                        onClick={marcarTodasLeidas}
                                        size="lg"
                                        aria-label="Marcar todas como leídas"
                                        isRound
                                    />
                                </Tooltip>
                            )}
                        </HStack>
                    </Flex>
                </Box>

                {/* Tabs */}
                <Tabs colorScheme="blue" variant="soft-rounded" isLazy>
                    <TabList mb={6} bg="white" p={2} borderRadius="2xl" boxShadow="sm" display="inline-flex">
                        <Tab fontWeight="semibold" _selected={{ bg: 'blue.50', color: 'blue.600' }}>
                            <HStack spacing={2}>
                                <Icon as={FiMail} />
                                <Text>No leídas</Text>
                                {stats.noLeidas > 0 && (
                                    <Badge colorScheme="blue" borderRadius="full" px={2}>{stats.noLeidas}</Badge>
                                )}
                            </HStack>
                        </Tab>
                        <Tab fontWeight="semibold" _selected={{ bg: 'gray.100', color: 'gray.800' }}>
                            <HStack spacing={2}>
                                <Icon as={FiInbox} />
                                <Text>Todas</Text>
                                <Badge colorScheme="gray" borderRadius="full" px={2}>{todas.length}</Badge>
                            </HStack>
                        </Tab>
                        <Tab fontWeight="semibold" _selected={{ bg: 'orange.50', color: 'orange.600' }}>
                            <HStack spacing={2}>
                                <Icon as={FiArchive} />
                                <Text>Archivadas</Text>
                                {stats.archivadas > 0 && (
                                    <Badge colorScheme="orange" borderRadius="full" px={2}>{stats.archivadas}</Badge>
                                )}
                            </HStack>
                        </Tab>
                    </TabList>

                    <TabPanels>
                        {/* Panel: No leídas */}
                        <TabPanel px={0} py={0}>
                            <SlideFade in={true} offsetY="20px">
                                {noLeidas.length === 0 ? (
                                    <EmptyState
                                        icon={FiCheckCircle}
                                        title="¡Todo al día!"
                                        description={searchTerm || filterTipo !== 'todas' ? "No hay notificaciones sin leer con los filtros actuales." : "No tienes notificaciones sin leer. ¡Excelente!"}
                                    />
                                ) : (
                                    <VStack align="stretch" spacing={4}>
                                        {noLeidas.map((notif) => (
                                            <NotificacionCard
                                                key={notif.id}
                                                notificacion={notif}
                                                onMarcarLeida={() => marcarComoLeida(notif.id)}
                                                onArchivar={() => archivarNotificacion(notif.id)}
                                            />
                                        ))}
                                    </VStack>
                                )}
                            </SlideFade>
                        </TabPanel>

                        {/* Panel: Todas */}
                        <TabPanel px={0} py={0}>
                            <SlideFade in={true} offsetY="20px">
                                {todas.length === 0 ? (
                                    <EmptyState
                                        icon={FiInbox}
                                        title="Bandeja vacía"
                                        description={searchTerm || filterTipo !== 'todas' ? "No se encontraron notificaciones con los filtros seleccionados." : "Aún no has recibido ninguna notificación."}
                                    />
                                ) : (
                                    <VStack align="stretch" spacing={4}>
                                        {todas.map((notif) => (
                                            <NotificacionCard
                                                key={notif.id}
                                                notificacion={notif}
                                                onMarcarLeida={() => marcarComoLeida(notif.id)}
                                                onArchivar={() => archivarNotificacion(notif.id)}
                                            />
                                        ))}
                                    </VStack>
                                )}
                            </SlideFade>
                        </TabPanel>

                        {/* Panel: Archivadas */}
                        <TabPanel px={0} py={0}>
                            <SlideFade in={true} offsetY="20px">
                                {archivadas.length === 0 ? (
                                    <EmptyState
                                        icon={FiArchive}
                                        title="Archivo vacío"
                                        description={searchTerm || filterTipo !== 'todas' ? "No hay notificaciones archivadas que coincidan con tu búsqueda." : "No has archivado ninguna notificación todavía."}
                                    />
                                ) : (
                                    <VStack align="stretch" spacing={4}>
                                        {archivadas.map((notif) => (
                                            <NotificacionCard
                                                key={notif.id}
                                                notificacion={notif}
                                                onMarcarLeida={() => marcarComoLeida(notif.id)}
                                                onArchivar={() => archivarNotificacion(notif.id)}
                                            />
                                        ))}
                                    </VStack>
                                )}
                            </SlideFade>
                        </TabPanel>
                    </TabPanels>
                </Tabs>
            </Box>
        </Layout>
    );
};

export default NotificacionesAdmin;
