import React, { useState } from 'react';
import {
    Box,
    SimpleGrid,
    Card,
    CardBody,
    Stat,
    StatLabel,
    StatNumber,
    StatHelpText,
    StatArrow,
    Table,
    Thead,
    Tbody,
    Tr,
    Th,
    Td,
    Badge,
    IconButton,
    Modal,
    ModalOverlay,
    ModalContent,
    ModalHeader,
    ModalBody,
    ModalCloseButton,
    useDisclosure,
    Text,
    HStack,
    VStack,
    Divider,
    Tooltip,
    Tabs,
    TabList,
    TabPanels,
    Tab,
    TabPanel,
    Avatar,
    Input,
    InputGroup,
    InputLeftElement
} from '@chakra-ui/react';
import {
    BarChart,
    Bar,
    XAxis,
    YAxis,
    CartesianGrid,
    Tooltip as RechartsTooltip,
    ResponsiveContainer,
    LineChart,
    Line
} from 'recharts';
import { FaEye, FaTrash, FaUsers, FaSearch } from 'react-icons/fa';
import NotificacionCard from '../../components/NotificacionCard';

// Mock Data
const statsData = [
    { label: 'Total Enviadas', value: '1,234', change: 12.5, type: 'increase' },
    { label: 'Tasa de Apertura', value: '85%', change: 5.2, type: 'increase' },
    { label: 'Clics en Enlaces', value: '450', change: -2.4, type: 'decrease' },
    { label: 'Rebotes', value: '12', change: -10.5, type: 'decrease' },
];

const chartData = [
    { name: 'Lun', enviadas: 40, leidas: 24 },
    { name: 'Mar', enviadas: 30, leidas: 13 },
    { name: 'Mie', enviadas: 20, leidas: 18 },
    { name: 'Jue', enviadas: 27, leidas: 20 },
    { name: 'Vie', enviadas: 18, leidas: 15 },
    { name: 'Sab', enviadas: 23, leidas: 18 },
    { name: 'Dom', enviadas: 34, leidas: 30 },
];

// Mock recipients generator
const generateRecipients = (count, type) => {
    return Array.from({ length: count }).map((_, i) => ({
        id: i,
        nombre: type === 'Padres' ? `Padre ${i + 1}` : type === 'Profesores' ? `Profesor ${i + 1}` : `Alumno ${i + 1}`,
        email: `usuario${i + 1}@artschool.com`,
        leido: Math.random() > 0.3,
        fechaLectura: Math.random() > 0.3 ? '2023-11-20 10:30' : '-'
    }));
};

const historialData = [
    {
        id: 1,
        titulo: '🎉 Promoción de Verano',
        destinatario: 'Padres',
        fecha: '2023-11-20',
        leidas: 145,
        total: 150,
        tipo: 'promocion',
        mensaje: 'Aprovecha nuestros descuentos especiales para el curso de verano.',
        html: '<h3>¡Verano de Arte!</h3><p>Descuentos especiales...</p>',
        recipients: generateRecipients(20, 'Padres')
    },
    {
        id: 2,
        titulo: '⚠️ Suspensión de Clases',
        destinatario: 'General',
        fecha: '2023-11-18',
        leidas: 280,
        total: 300,
        tipo: 'recordatorio',
        mensaje: 'Debido al clima severo, se suspenden las clases mañana.',
        recipients: generateRecipients(20, 'Alumnos')
    },
    {
        id: 3,
        titulo: '🎨 Nuevo Taller de Acuarela',
        destinatario: 'Alumnos',
        fecha: '2023-11-15',
        leidas: 45,
        total: 50,
        tipo: 'curso',
        mensaje: 'Inscríbete al nuevo taller de acuarela avanzada.',
        recipients: generateRecipients(15, 'Alumnos')
    },
    {
        id: 4,
        titulo: '💳 Recordatorio de Pago',
        destinatario: 'Padres',
        fecha: '2023-11-10',
        leidas: 120,
        total: 150,
        tipo: 'pago',
        mensaje: 'Recordatorio amable sobre la colegiatura del mes.',
        recipients: generateRecipients(10, 'Padres')
    },
    {
        id: 5,
        titulo: '📝 Entrega de Bitácoras',
        destinatario: 'Profesores',
        fecha: '2023-11-05',
        leidas: 12,
        total: 12,
        tipo: 'bitacora',
        mensaje: 'Favor de entregar las bitácoras antes del viernes.',
        recipients: generateRecipients(12, 'Profesores')
    },
];

const NotificacionesHistorial = () => {
    const { isOpen, onOpen, onClose } = useDisclosure();
    const [selectedNotif, setSelectedNotif] = useState(null);
    const [searchTerm, setSearchTerm] = useState('');

    const handleViewDetail = (notif) => {
        setSelectedNotif(notif);
        onOpen();
    };

    const filteredRecipients = selectedNotif?.recipients?.filter(r =>
        r.nombre.toLowerCase().includes(searchTerm.toLowerCase()) ||
        r.email.toLowerCase().includes(searchTerm.toLowerCase())
    );

    return (
        <Box>
            {/* Stats Cards */}
            <SimpleGrid columns={{ base: 1, md: 2, lg: 4 }} spacing={6} mb={8}>
                {statsData.map((stat, index) => (
                    <Card key={index} borderRadius="xl" boxShadow="sm">
                        <CardBody>
                            <Stat>
                                <StatLabel color="gray.500">{stat.label}</StatLabel>
                                <StatNumber fontSize="2xl" fontWeight="bold" color="brand.600">
                                    {stat.value}
                                </StatNumber>
                                <StatHelpText>
                                    <StatArrow type={stat.type} />
                                    {Math.abs(stat.change)}%
                                </StatHelpText>
                            </Stat>
                        </CardBody>
                    </Card>
                ))}
            </SimpleGrid>

            {/* Charts */}
            <SimpleGrid columns={{ base: 1, lg: 2 }} spacing={6} mb={8}>
                <Card borderRadius="xl" boxShadow="sm" p={4}>
                    <Text fontSize="lg" fontWeight="bold" mb={4}>Actividad Semanal</Text>
                    <Box h="300px">
                        <ResponsiveContainer width="100%" height="100%">
                            <BarChart data={chartData}>
                                <CartesianGrid strokeDasharray="3 3" />
                                <XAxis dataKey="name" />
                                <YAxis />
                                <RechartsTooltip />
                                <Bar dataKey="enviadas" fill="#8884d8" name="Enviadas" radius={[4, 4, 0, 0]} />
                                <Bar dataKey="leidas" fill="#82ca9d" name="Leídas" radius={[4, 4, 0, 0]} />
                            </BarChart>
                        </ResponsiveContainer>
                    </Box>
                </Card>

                <Card borderRadius="xl" boxShadow="sm" p={4}>
                    <Text fontSize="lg" fontWeight="bold" mb={4}>Tendencia de Apertura</Text>
                    <Box h="300px">
                        <ResponsiveContainer width="100%" height="100%">
                            <LineChart data={chartData}>
                                <CartesianGrid strokeDasharray="3 3" />
                                <XAxis dataKey="name" />
                                <YAxis />
                                <RechartsTooltip />
                                <Line type="monotone" dataKey="leidas" stroke="#82ca9d" strokeWidth={3} dot={{ r: 4 }} />
                            </LineChart>
                        </ResponsiveContainer>
                    </Box>
                </Card>
            </SimpleGrid>

            {/* Table */}
            <Card borderRadius="xl" boxShadow="sm" overflow="hidden">
                <Box p={4} borderBottom="1px solid" borderColor="gray.100">
                    <Text fontSize="lg" fontWeight="bold">Historial de Envíos</Text>
                </Box>
                <Box overflowX="auto">
                    <Table variant="simple">
                        <Thead bg="gray.50">
                            <Tr>
                                <Th>Título</Th>
                                <Th>Destinatario</Th>
                                <Th>Tipo</Th>
                                <Th>Fecha</Th>
                                <Th isNumeric>Alcance</Th>
                                <Th>Acciones</Th>
                            </Tr>
                        </Thead>
                        <Tbody>
                            {historialData.map((notif) => (
                                <Tr key={notif.id} _hover={{ bg: 'gray.50' }}>
                                    <Td fontWeight="500">{notif.titulo}</Td>
                                    <Td>
                                        <Badge colorScheme={
                                            notif.destinatario === 'Profesores' ? 'purple' :
                                                notif.destinatario === 'Padres' ? 'blue' :
                                                    notif.destinatario === 'Alumnos' ? 'green' : 'gray'
                                        }>
                                            {notif.destinatario}
                                        </Badge>
                                    </Td>
                                    <Td>{notif.tipo}</Td>
                                    <Td>{notif.fecha}</Td>
                                    <Td isNumeric>
                                        <Text fontSize="sm">
                                            <Text as="span" fontWeight="bold" color="green.500">{notif.leidas}</Text> / {notif.total}
                                        </Text>
                                        <Text fontSize="xs" color="gray.400">
                                            {Math.round((notif.leidas / notif.total) * 100)}% leídas
                                        </Text>
                                    </Td>
                                    <Td>
                                        <Tooltip label="Ver Detalle y Destinatarios">
                                            <IconButton
                                                icon={<FaEye />}
                                                size="sm"
                                                colorScheme="brand"
                                                variant="ghost"
                                                onClick={() => handleViewDetail(notif)}
                                                mr={2}
                                            />
                                        </Tooltip>
                                    </Td>
                                </Tr>
                            ))}
                        </Tbody>
                    </Table>
                </Box>
            </Card>

            {/* Detail Modal */}
            <Modal isOpen={isOpen} onClose={onClose} size="xl" scrollBehavior="inside">
                <ModalOverlay backdropFilter="blur(5px)" />
                <ModalContent borderRadius="xl" maxH="85vh">
                    <ModalHeader borderBottom="1px solid" borderColor="gray.100" pb={4}>
                        <Text fontSize="lg">Detalle de Notificación</Text>
                        <Text fontSize="sm" fontWeight="normal" color="gray.500" mt={1}>
                            {selectedNotif?.titulo}
                        </Text>
                    </ModalHeader>
                    <ModalCloseButton />
                    <ModalBody p={0}>
                        {selectedNotif && (
                            <Tabs isFitted colorScheme="brand">
                                <TabList bg="gray.50">
                                    <Tab py={4} fontWeight="600">📩 Mensaje y Detalles</Tab>
                                    <Tab py={4} fontWeight="600">👥 Destinatarios ({selectedNotif.recipients?.length})</Tab>
                                </TabList>

                                <TabPanels>
                                    {/* Panel de Mensaje */}
                                    <TabPanel p={6}>
                                        <VStack spacing={6} align="stretch">
                                            <Box>
                                                <Text fontSize="xs" textTransform="uppercase" fontWeight="bold" color="gray.500" mb={3}>
                                                    Información General
                                                </Text>
                                                <SimpleGrid columns={{ base: 1, md: 2 }} spacing={4} bg="gray.50" p={4} borderRadius="md">
                                                    <Box>
                                                        <Text fontWeight="bold" fontSize="sm">Destinatario</Text>
                                                        <Text fontSize="sm">{selectedNotif.destinatario}</Text>
                                                    </Box>
                                                    <Box>
                                                        <Text fontWeight="bold" fontSize="sm">Fecha de Envío</Text>
                                                        <Text fontSize="sm">{selectedNotif.fecha}</Text>
                                                    </Box>
                                                    <Box>
                                                        <Text fontWeight="bold" fontSize="sm">Tipo</Text>
                                                        <Text fontSize="sm" textTransform="capitalize">{selectedNotif.tipo}</Text>
                                                    </Box>
                                                    <Box>
                                                        <Text fontWeight="bold" fontSize="sm">Tasa de Apertura</Text>
                                                        <Text fontSize="sm" color="green.600" fontWeight="bold">
                                                            {Math.round((selectedNotif.leidas / selectedNotif.total) * 100)}%
                                                        </Text>
                                                    </Box>
                                                </SimpleGrid>
                                            </Box>

                                            <Divider />

                                            <Box>
                                                <Text fontSize="xs" textTransform="uppercase" fontWeight="bold" color="gray.500" mb={3}>
                                                    Vista Previa del Mensaje
                                                </Text>
                                                <Box
                                                    border="1px solid"
                                                    borderColor="gray.200"
                                                    borderRadius="lg"
                                                    p={4}
                                                    bg="white"
                                                    boxShadow="sm"
                                                >
                                                    <NotificacionCard notificacion={{
                                                        ...selectedNotif,
                                                        leida: false // Just for preview
                                                    }} />
                                                </Box>
                                            </Box>
                                        </VStack>
                                    </TabPanel>

                                    {/* Panel de Destinatarios */}
                                    <TabPanel p={0}>
                                        <Box p={4} borderBottom="1px solid" borderColor="gray.100">
                                            <InputGroup>
                                                <InputLeftElement pointerEvents="none">
                                                    <FaSearch color="gray.300" />
                                                </InputLeftElement>
                                                <Input
                                                    placeholder="Buscar destinatario..."
                                                    value={searchTerm}
                                                    onChange={(e) => setSearchTerm(e.target.value)}
                                                />
                                            </InputGroup>
                                        </Box>
                                        <Box overflowY="auto" maxH="400px">
                                            <Table variant="simple" size="sm">
                                                <Thead position="sticky" top={0} bg="white" zIndex={1}>
                                                    <Tr>
                                                        <Th>Usuario</Th>
                                                        <Th>Estado</Th>
                                                        <Th>Fecha Lectura</Th>
                                                    </Tr>
                                                </Thead>
                                                <Tbody>
                                                    {filteredRecipients?.map((recipient) => (
                                                        <Tr key={recipient.id}>
                                                            <Td>
                                                                <HStack>
                                                                    <Avatar size="xs" name={recipient.nombre} />
                                                                    <VStack align="start" spacing={0}>
                                                                        <Text fontWeight="500">{recipient.nombre}</Text>
                                                                        <Text fontSize="xs" color="gray.500">{recipient.email}</Text>
                                                                    </VStack>
                                                                </HStack>
                                                            </Td>
                                                            <Td>
                                                                <Badge colorScheme={recipient.leido ? 'green' : 'gray'}>
                                                                    {recipient.leido ? 'Leído' : 'Enviado'}
                                                                </Badge>
                                                            </Td>
                                                            <Td fontSize="sm" color="gray.600">
                                                                {recipient.fechaLectura}
                                                            </Td>
                                                        </Tr>
                                                    ))}
                                                </Tbody>
                                            </Table>
                                        </Box>
                                    </TabPanel>
                                </TabPanels>
                            </Tabs>
                        )}
                    </ModalBody>
                </ModalContent>
            </Modal>
        </Box>
    );
};

export default NotificacionesHistorial;
