import { useState, useEffect } from 'react';
import {
    Box,
    Button,
    Heading,
    Table,
    Thead,
    Tbody,
    Tr,
    Th,
    Td,
    Badge,
    IconButton,
    useDisclosure,
    Modal,
    ModalOverlay,
    ModalContent,
    ModalHeader,
    ModalFooter,
    ModalBody,
    ModalCloseButton,
    FormControl,
    FormLabel,
    Input,
    Select,
    useToast,
    HStack,
    Text,
    Textarea,
    VStack,
    SimpleGrid,
    Image,
    Tabs,
    TabList,
    Tab,
    TabPanel,
    TabPanels,
    Divider,
} from '@chakra-ui/react';
import { FiEdit2, FiTrash2, FiPlus, FiEye, FiStar } from 'react-icons/fi';
import { getOrders, getClients, getEmployees, getServicesByCompany, createOrder, updateOrder, getEvidencesByService } from '../../services/api';

const OrdenesServicio = () => {
    const [orders, setOrders] = useState([]);
    const [clients, setClients] = useState([]);
    const [employees, setEmployees] = useState([]);
    const [servicesCatalog, setServicesCatalog] = useState([]);
    const [currentOrder, setCurrentOrder] = useState(null);
    const [isAdmin, setIsAdmin] = useState(false);
    const { isOpen, onOpen, onClose } = useDisclosure();
    const toast = useToast();

    // Disclosure for View Modal
    const { isOpen: isViewOpen, onOpen: onViewOpen, onClose: onViewClose } = useDisclosure();
    const [selectedOrderForView, setSelectedOrderForView] = useState(null);
    const [evidences, setEvidences] = useState([]);

    // Función para traducir estados a leyendas amigables
    const getStatusLabel = (status) => {
        const statusMap = {
            'PENDING': 'Pendiente',
            'IN_PROGRESS': 'En Progreso',
            'FINISHED': 'Finalizado',
            'CANCELLED': 'Cancelado',
            'CERRADO_ADMIN': 'Cerrado',
            'IN_REVIEW': 'Reapertura'
        };
        return statusMap[status] || status;
    };

    const fetchData = async () => {
        try {
            const user = JSON.parse(sessionStorage.getItem('user'));
            const companyId = user ? user.company_id : null;
            
            // Verificar si el usuario es administrador
            setIsAdmin(user?.role === 'admin');

            if (companyId) {
                const [allOrders, allClients, allEmployees, companyServices] = await Promise.all([
                    getOrders(),
                    getClients(),
                    getEmployees(),
                    getServicesByCompany(companyId)
                ]);

                setOrders(allOrders.filter(o => o.company_id === companyId).sort((a, b) => b.id - a.id));
                setClients(allClients.filter(c => c.company_id === companyId));
                setEmployees(allEmployees.filter(e => e.company_id === companyId));
                setServicesCatalog(companyServices);
            }
        } catch (error) {
            console.error(error);
            toast({
                title: 'Error al cargar datos',
                status: 'error',
                duration: 3000,
            });
        }
    };

    useEffect(() => {
        fetchData();
    }, []);

    const getClientName = (id) => clients.find(c => c.user_id === id)?.name || 'Desconocido';
    const getEmployeeName = (id) => employees.find(e => e.id === id)?.name || 'Sin asignar';
    const getServiceName = (id) => servicesCatalog.find(s => s.id === id)?.name || 'Desconocido';

    const handleViewDetails = async (order) => {
        setSelectedOrderForView(order);
        try {
            const serviceEvidences = await getEvidencesByService(order.id);
            setEvidences(serviceEvidences);
        } catch (error) {
            console.error("Error fetching evidences:", error);
            setEvidences([]);
        }
        onViewOpen();
    };

    const handleEdit = (order) => {
        setCurrentOrder(order);
        onOpen();
    };

    const handleCreate = () => {
        const user = JSON.parse(sessionStorage.getItem('user'));
        setCurrentOrder({
            service_number: `SRV-${new Date().getFullYear()}-${orders.length + 1}`,
            client_id: '',
            service_catalogo_id: '',
            status: 'PENDING',
            request_date: new Date().toISOString().split('T')[0],
            deadline_date: '',
            employee_id: '',
            company_id: user ? user.company_id : null
        });
        onOpen();
    };

    const handleSave = async () => {
        try {
            if (currentOrder.id) {
                await updateOrder(currentOrder.id, currentOrder);
                toast({
                    title: 'Orden actualizada',
                    status: 'success',
                    duration: 2000,
                });
            } else {
                await createOrder(currentOrder);
                toast({
                    title: 'Orden creada',
                    status: 'success',
                    duration: 2000,
                });
            }
            fetchData();
            onClose();
        } catch (error) {
            toast({
                title: 'Error al guardar',
                description: error.message,
                status: 'error',
                duration: 3000,
            });
        }
    };

    return (
        <Box>
            <HStack justify="space-between" mb={6}>
                <Heading>Órdenes de Servicio</Heading>
                <Button leftIcon={<FiPlus />} colorScheme="brand" onClick={handleCreate}>
                    Nueva Orden
                </Button>
            </HStack>

            <Box bg="white" borderRadius="lg" boxShadow="sm" overflowX="auto">
                <Table variant="simple">
                    <Thead bg="gray.50">
                        <Tr>
                            <Th>Folio</Th>
                            <Th>Cliente</Th>
                            <Th>Servicio</Th>
                            <Th>Empleado</Th>
                            <Th>Estado</Th>
                            <Th>Acciones</Th>
                        </Tr>
                    </Thead>
                    <Tbody>
                        {orders.map((order) => (
                            <Tr key={order.id}>
                                <Td fontWeight="bold">{order.service_number}</Td>
                                <Td>{getClientName(order.client_id)}</Td>
                                <Td>{getServiceName(order.service_catalogo_id)}</Td>
                                <Td>{getEmployeeName(order.employee_id)}</Td>
                                <Td>
                                    <Badge
                                        colorScheme={
                                            order.status === 'FINISHED' ? 'green' :
                                                order.status === 'IN_PROGRESS' ? 'blue' :
                                                    order.status === 'PENDING' ? 'orange' : 
                                                    order.status === 'IN_REVIEW' ? 'purple' : 'gray'
                                        }
                                    >
                                        {getStatusLabel(order.status)}
                                    </Badge>
                                </Td>
                                <Td>
                                    <HStack spacing={2}>
                                        <IconButton
                                            icon={<FiEdit2 />}
                                            size="sm"
                                            aria-label="Editar"
                                            onClick={() => handleEdit(order)}
                                        />
                                        <IconButton
                                            icon={<FiEye />}
                                            size="sm"
                                            aria-label="Ver Detalles"
                                            onClick={() => handleViewDetails(order)}
                                        />
                                    </HStack>
                                </Td>
                            </Tr>
                        ))}
                    </Tbody>
                </Table>
            </Box>

            {/* Modal */}
            <Modal isOpen={isOpen} onClose={onClose} size="lg">
                <ModalOverlay />
                <ModalContent>
                    <ModalHeader>{currentOrder?.id ? 'Editar Orden' : 'Nueva Orden'}</ModalHeader>
                    <ModalCloseButton />
                    <ModalBody>
                        <Box display="flex" flexDirection="column" gap={4}>
                            <FormControl>
                                <FormLabel>Cliente</FormLabel>
                                <Select
                                    value={currentOrder?.client_id || ''}
                                    onChange={(e) => setCurrentOrder({ ...currentOrder, client_id: e.target.value })}
                                >
                                    <option value="">Seleccionar...</option>
                                    {clients.map(c => (
                                        <option key={c.user_id} value={c.user_id}>{c.name}</option>
                                    ))}
                                </Select>
                            </FormControl>

                            <FormControl>
                                <FormLabel>Servicio</FormLabel>
                                <Select
                                    value={currentOrder?.service_catalogo_id || ''}
                                    onChange={(e) => setCurrentOrder({ ...currentOrder, service_catalogo_id: parseInt(e.target.value) })}
                                >
                                    <option value="">Seleccionar...</option>
                                    {servicesCatalog.map(s => (
                                        <option key={s.id} value={s.id}>{s.name}</option>
                                    ))}
                                </Select>
                            </FormControl>

                            <FormControl>
                                <FormLabel>Asignar Empleado</FormLabel>
                                <Select
                                    value={currentOrder?.employee_id || ''}
                                    onChange={(e) => setCurrentOrder({ ...currentOrder, employee_id: e.target.value ? parseInt(e.target.value) : '' })}
                                >
                                    <option value="">Sin asignar</option>
                                    {employees.map(e => (
                                        <option key={e.id} value={e.id}>{e.name} ({e.specialty})</option>
                                    ))}
                                </Select>
                            </FormControl>

                            <HStack>
                                <FormControl>
                                    <FormLabel>Fecha Solicitud</FormLabel>
                                    <Input
                                        type="date"
                                        value={currentOrder?.request_date?.split('T')[0] || ''}
                                        onChange={(e) => setCurrentOrder({ ...currentOrder, request_date: e.target.value })}
                                    />
                                </FormControl>

                                <FormControl>
                                    <FormLabel>Fecha Límite</FormLabel>
                                    <Input
                                        type="date"
                                        value={currentOrder?.deadline_date?.split('T')[0] || ''}
                                        onChange={(e) => setCurrentOrder({ ...currentOrder, deadline_date: e.target.value })}
                                    />
                                </FormControl>
                            </HStack>

                            <FormControl>
                                <FormLabel>Estado</FormLabel>
                                <Select
                                    value={currentOrder?.status || ''}
                                    onChange={(e) => {
                                        if (e.target.value === 'CANCELLED' && !isAdmin) {
                                            toast({
                                                title: 'Permiso denegado',
                                                description: 'Solo los administradores pueden cancelar servicios',
                                                status: 'warning',
                                                duration: 3000,
                                            });
                                            return;
                                        }
                                        setCurrentOrder({ ...currentOrder, status: e.target.value });
                                    }}
                                >
                                    <option value="PENDING">Pendiente</option>
                                    <option value="IN_PROGRESS">En Progreso</option>
                                    <option value="FINISHED">Finalizado</option>
                                    <option value="CANCELLED" disabled={!isAdmin}>
                                        {isAdmin ? 'Cancelado' : 'Cancelado (Solo Admin)'}
                                    </option>
                                    <option value="CERRADO_ADMIN">Cerrado</option>
                                </Select>
                                {!isAdmin && (
                                    <Text fontSize="xs" color="orange.600" mt={1}>
                                        ℹ️ Solo los administradores pueden cancelar servicios
                                    </Text>
                                )}
                            </FormControl>

                            {(currentOrder?.status === 'IN_REVIEW' || currentOrder?.status === 'CERRADO_ADMIN') && (
                                <Box mt={4} p={4} bg="gray.50" borderRadius="md">
                                    <Heading size="sm" mb={2}>Feedback del Cliente</Heading>
                                    {currentOrder.client_rating && (
                                        <HStack>
                                            <Text fontWeight="bold">Calificación:</Text>
                                            <HStack spacing={1}>
                                                {[...Array(5)].map((_, i) => (
                                                    <FiStar key={i} color={i < currentOrder.client_rating ? 'orange' : 'gray'} />
                                                ))}
                                            </HStack>
                                        </HStack>
                                    )}
                                    {currentOrder.client_comment && (
                                        <Box mt={2}>
                                            <Text fontWeight="bold">Comentario:</Text>
                                            <Text>{currentOrder.client_comment}</Text>
                                        </Box>
                                    )}
                                </Box>
                            )}
                        </Box>
                    </ModalBody>

                    <ModalFooter>
                        <Button variant="ghost" mr={3} onClick={onClose}>
                            Cancelar
                        </Button>
                        <Button colorScheme="brand" onClick={handleSave}>
                            Guardar
                        </Button>
                    </ModalFooter>
                </ModalContent>
            </Modal>

            {/* View Details Modal */}
            <Modal isOpen={isViewOpen} onClose={onViewClose} size="xl">
                <ModalOverlay />
                <ModalContent>
                    <ModalHeader>Detalles del Servicio</ModalHeader>
                    <ModalCloseButton />
                    <ModalBody>
                        {selectedOrderForView && (
                            <VStack align="stretch" spacing={4}>
                                <SimpleGrid columns={2} spacing={4}>
                                    <Box>
                                        <Text fontWeight="bold">Folio:</Text>
                                        <Text>{selectedOrderForView.service_number}</Text>
                                    </Box>
                                    <Box>
                                        <Text fontWeight="bold">Estado:</Text>
                                        <Badge colorScheme={
                                            selectedOrderForView.status === 'FINISHED' ? 'green' :
                                            selectedOrderForView.status === 'IN_PROGRESS' ? 'blue' :
                                            selectedOrderForView.status === 'PENDING' ? 'orange' : 
                                            selectedOrderForView.status === 'IN_REVIEW' ? 'purple' : 'gray'
                                        }>
                                            {getStatusLabel(selectedOrderForView.status)}
                                        </Badge>
                                    </Box>
                                </SimpleGrid>
                                <Divider />
                                <Tabs variant="enclosed">
                                    <TabList>
                                        <Tab>General</Tab>
                                        <Tab>Evidencias ({evidences.length})</Tab>
                                    </TabList>
                                    <TabPanels>
                                        <TabPanel>
                                            <VStack align="stretch" spacing={3}>
                                                <Box>
                                                    <Text fontWeight="bold">Cliente:</Text>
                                                    <Text>{getClientName(selectedOrderForView.client_id)}</Text>
                                                </Box>
                                                <Box>
                                                    <Text fontWeight="bold">Servicio:</Text>
                                                    <Text>{getServiceName(selectedOrderForView.service_catalogo_id)}</Text>
                                                </Box>
                                                <Box>
                                                    <Text fontWeight="bold">Empleado Asignado:</Text>
                                                    <Text>{getEmployeeName(selectedOrderForView.employee_id)}</Text>
                                                </Box>
                                                <Divider />
                                                {(selectedOrderForView.status === 'IN_REVIEW' || selectedOrderForView.status === 'CERRADO_ADMIN') && (
                                                    <Box p={3} bg="gray.50" borderRadius="md">
                                                        <Heading size="sm" mb={2}>Feedback del Cliente</Heading>
                                                        {selectedOrderForView.client_rating && (
                                                            <HStack>
                                                                <Text fontWeight="bold">Calificación:</Text>
                                                                <HStack spacing={1}>
                                                                    {[...Array(5)].map((_, i) => (
                                                                        <FiStar key={i} color={i < selectedOrderForView.client_rating ? 'orange' : 'gray'} />
                                                                    ))}
                                                                </HStack>
                                                            </HStack>
                                                        )}
                                                        {selectedOrderForView.client_comment && (
                                                            <Box mt={2}>
                                                                <Text fontWeight="bold">Comentario:</Text>
                                                                <Text whiteSpace="pre-wrap">{selectedOrderForView.client_comment}</Text>
                                                            </Box>
                                                        )}
                                                        
                                                        {evidences.filter(e => e.description?.startsWith("Evidencia de reapertura:")).length > 0 && (
                                                            <Box mt={3}>
                                                                <Text fontWeight="bold">Evidencias de Reapertura:</Text>
                                                                <SimpleGrid columns={3} spacing={2} mt={1}>
                                                                    {evidences.filter(e => e.description?.startsWith("Evidencia de reapertura:")).map(ev => (
                                                                        <Image key={ev.id} src={ev.file_url} alt="Evidencia de reapertura" borderRadius="md" />
                                                                    ))}
                                                                </SimpleGrid>
                                                            </Box>
                                                        )}
                                                    </Box>
                                                )}
                                            </VStack>
                                        </TabPanel>
                                        <TabPanel>
                                            {evidences.filter(e => !e.description?.startsWith("Evidencia de reapertura:")).length > 0 ? (
                                                <SimpleGrid columns={2} spacing={4}>
                                                    {evidences.filter(e => !e.description?.startsWith("Evidencia de reapertura:")).map(ev => (
                                                        <Box key={ev.id} borderWidth="1px" borderRadius="lg" p={2}>
                                                            <Image src={ev.file_url} alt={ev.description} borderRadius="md" mb={2} />
                                                            <Text fontSize="sm">{ev.description}</Text>
                                                        </Box>
                                                    ))}
                                                </SimpleGrid>
                                            ) : (
                                                <Text>No hay evidencias generales para este servicio.</Text>
                                            )}
                                        </TabPanel>
                                    </TabPanels>
                                </Tabs>
                            </VStack>
                        )}
                    </ModalBody>
                    <ModalFooter>
                        <Button colorScheme="brand" onClick={onViewClose}>
                            Cerrar
                        </Button>
                    </ModalFooter>
                </ModalContent>
            </Modal>
        </Box>
    );
};

export default OrdenesServicio;
