import React from 'react';
import {
  Box,
  Text,
  SimpleGrid,
  Card,
  CardBody,
  HStack,
  VStack,
  Heading,
  Icon,
  Button,
  Flex,
  Avatar,
  Badge,
  Container,
  Progress,
  Tag
} from '@chakra-ui/react';
import { FiBookOpen, FiCalendar, FiTarget, FiClock, FiAward, FiArrowRight, FiStar } from 'react-icons/fi';
import { useNavigate } from 'react-router-dom';
import { motion } from 'framer-motion';
import Layout from '../../components/Layout';
import { mockBitacoras, mockClases, mockMetas, mockMaterias, mockProfesores } from '../../data/mockData';

const MotionBox = motion(Box);
const MotionCard = motion(Card);

const containerVariants = {
  hidden: { opacity: 0 },
  visible: {
    opacity: 1,
    transition: {
      staggerChildren: 0.1
    }
  }
};

const itemVariants = {
  hidden: { y: 20, opacity: 0 },
  visible: {
    y: 0,
    opacity: 1,
    transition: {
      type: "spring",
      stiffness: 100
    }
  }
};

const alumnoLinks = [
  { nombre: 'Dashboard', path: '/alumno' },
  { nombre: 'Mis Clases', path: '/alumno/clases' },
  { nombre: 'Bitácoras', path: '/alumno/bitacoras' },
  { nombre: 'Pagos', path: '/alumno/pagos' },
];

// Simulando Alumno ID 1 (Sofía López)
const alumnoId = 1;
const misClases = mockClases.filter(c => c.id_alumno === alumnoId);
const misBitacoras = mockBitacoras.filter(b => b.id_alumno === alumnoId).sort((a, b) => new Date(b.fecha) - new Date(a.fecha));
const miMeta = mockMetas[1] || "Aprender nuevas técnicas";

const StatCard = ({ title, value, icon, color, subtext }) => (
  <MotionCard
    variants={itemVariants}
    bg="white"
    borderRadius="2xl"
    boxShadow="sm"
    border="1px solid"
    borderColor="gray.100"
    whileHover={{ y: -5, boxShadow: 'lg' }}
    transition={{ duration: 0.2 }}
  >
    <CardBody>
      <Flex justify="space-between" align="start" mb={4}>
        <Box
          p={3}
          bgGradient={`linear(to-br, ${color}.400, ${color}.600)`}
          color="white"
          borderRadius="xl"
          boxShadow={`0 4px 12px ${color}.200`}
        >
          <Icon as={icon} boxSize={6} />
        </Box>
        <Icon as={FiStar} color="yellow.400" boxSize={5} fill="currentColor" opacity={0.5} />
      </Flex>
      <VStack align="start" spacing={0}>
        <Text fontSize="3xl" fontWeight="800" color="gray.800" lineHeight="1">
          {value}
        </Text>
        <Text fontSize="sm" color="gray.500" fontWeight="600" mt={1}>
          {title}
        </Text>
        <Text fontSize="xs" color={`${color}.500`} fontWeight="700" mt={2}>
          {subtext}
        </Text>
      </VStack>
    </CardBody>
  </MotionCard>
);

const NextClassCard = ({ clase }) => {
  const materia = mockMaterias.find(m => m.id === clase.id_materia);
  const profesor = mockProfesores.find(p => p.id === clase.id_profesor);

  return (
    <MotionCard
      variants={itemVariants}
      bgGradient="linear(to-r, brand.500, purple.600)"
      color="white"
      borderRadius="2xl"
      overflow="hidden"
      position="relative"
      boxShadow="xl"
    >
      <Box position="absolute" top="-20px" right="-20px" opacity={0.1}>
        <Icon as={FiClock} boxSize="150px" />
      </Box>
      <CardBody position="relative" zIndex={1} p={8}>
        <Flex direction={{ base: 'column', md: 'row' }} justify="space-between" align="center" gap={6}>
          <Box>
            <Badge bg="whiteAlpha.300" color="white" mb={2} px={3} py={1} borderRadius="full">
              Próxima Clase
            </Badge>
            <Heading size="lg" mb={2}>{materia?.nombre}</Heading>
            <HStack spacing={4} fontSize="lg" fontWeight="500">
              <HStack>
                <Icon as={FiCalendar} />
                <Text>{clase.dia_semana}</Text>
              </HStack>
              <HStack>
                <Icon as={FiClock} />
                <Text>{clase.hora_inicio}</Text>
              </HStack>
            </HStack>
          </Box>
          <HStack bg="whiteAlpha.200" p={4} borderRadius="xl" spacing={4}>
            <Avatar size="md" src={profesor?.foto_url} name={profesor?.nombre} border="2px solid white" />
            <Box>
              <Text fontSize="sm" opacity={0.8}>Profesor</Text>
              <Text fontWeight="bold">{profesor?.nombre}</Text>
            </Box>
          </HStack>
        </Flex>
      </CardBody>
    </MotionCard>
  );
};

const FeedbackItem = ({ bitacora }) => {
  const materia = mockMaterias.find(m => m.id === bitacora.id_materia);
  return (
    <MotionCard
      variants={itemVariants}
      bg="white"
      border="1px solid"
      borderColor="gray.100"
      borderRadius="xl"
      whileHover={{ scale: 1.02, borderColor: 'orange.200' }}
      transition={{ duration: 0.2 }}
    >
      <CardBody>
        <Flex justify="space-between" mb={2}>
          <Badge colorScheme="orange" borderRadius="full" px={2}>Evaluación</Badge>
          <Text fontSize="xs" color="gray.500">{bitacora.fecha}</Text>
        </Flex>
        <Heading size="sm" mb={2} color="gray.800">{materia?.nombre}</Heading>
        <Text fontSize="sm" color="gray.600" noOfLines={3}>
          "{bitacora.actividad}"
        </Text>
      </CardBody>
    </MotionCard>
  );
};

const AlumnoDashboard = () => {
  const navigate = useNavigate();
  const nextClass = misClases[0];

  return (
    <Layout links={alumnoLinks} userType="Alumno" userName="Sofía López" userPhoto="https://images.unsplash.com/photo-1517841905240-472988babdf9?ixlib=rb-1.2.1&auto=format&fit=facearea&facepad=2&w=256&h=256&q=80">
      <Container maxW="container.xl" px={0}>
        <MotionBox
          initial="hidden"
          animate="visible"
          variants={containerVariants}
        >
          <Flex justify="space-between" align="center" mb={8}>
            <Box>
              <Heading size="lg" color="gray.800" mb={2}>¡Hola, Sofía! 🎨</Heading>
              <Text color="gray.600">¿Lista para seguir creando hoy?</Text>
            </Box>
          </Flex>

          <SimpleGrid columns={{ base: 1, lg: 3 }} spacing={8} mb={8}>
            {/* Main Column */}
            <Box gridColumn={{ lg: "span 2" }}>
              <SimpleGrid columns={{ base: 1, md: 2 }} spacing={6} mb={8}>
                <StatCard
                  title="Bitácoras"
                  value={misBitacoras.length}
                  icon={FiBookOpen}
                  color="orange"
                  subtext="Evaluaciones recibidas"
                />
                <StatCard
                  title="Clases Semanales"
                  value={misClases.length}
                  icon={FiCalendar}
                  color="blue"
                  subtext="Horas de aprendizaje"
                />
              </SimpleGrid>

              {nextClass && (
                <Box mb={8}>
                  <Heading size="md" mb={4} color="gray.800">Tu Próxima Clase</Heading>
                  <NextClassCard clase={nextClass} />
                </Box>
              )}

              <Box>
                <Flex justify="space-between" align="center" mb={4}>
                  <Heading size="md" color="gray.800">Últimas Evaluaciones</Heading>
                  <Button variant="ghost" size="sm" colorScheme="orange" onClick={() => navigate('/alumno/bitacoras')}>
                    Ver todas
                  </Button>
                </Flex>
                <SimpleGrid columns={{ base: 1, md: 2 }} spacing={4}>
                  {misBitacoras.slice(0, 4).map((bitacora) => (
                    <FeedbackItem key={bitacora.id} bitacora={bitacora} />
                  ))}
                </SimpleGrid>
              </Box>
            </Box>

            {/* Sidebar Column */}
            <Box>
              <MotionCard
                variants={itemVariants}
                bg="white"
                border="1px solid"
                borderColor="gray.100"
                borderRadius="2xl"
                mb={6}
                overflow="hidden"
              >
                <Box h="6px" bgGradient="linear(to-r, orange.400, pink.500)" />
                <CardBody>
                  <Flex align="center" mb={4}>
                    <Icon as={FiTarget} color="pink.500" boxSize={6} mr={3} />
                    <Heading size="sm" color="gray.800">Mi Meta Actual</Heading>
                  </Flex>
                  <Text fontSize="md" fontWeight="600" color="gray.700" mb={6}>
                    "{miMeta}"
                  </Text>
                  <Box mb={2}>
                    <Flex justify="space-between" mb={2}>
                      <Text fontSize="xs" fontWeight="700" color="gray.500">PROGRESO</Text>
                      <Text fontSize="xs" fontWeight="700" color="pink.500">75%</Text>
                    </Flex>
                    <Progress
                      value={75}
                      size="sm"
                      colorScheme="pink"
                      borderRadius="full"
                      bg="gray.100"
                      hasStripe
                      isAnimated
                    />
                  </Box>
                </CardBody>
              </MotionCard>

              <MotionCard
                variants={itemVariants}
                bgGradient="linear(to-br, #667eea 0%, #764ba2 100%)"
                color="white"
                borderRadius="2xl"
                position="relative"
                overflow="hidden"
              >
                <Box position="absolute" top="-10px" right="-10px" opacity={0.2}>
                  <Icon as={FiAward} boxSize="100px" />
                </Box>
                <CardBody textAlign="center" py={10}>
                  <Icon as={FiAward} boxSize={12} mb={4} color="yellow.300" />
                  <Heading size="md" mb={2}>¡Sigue así!</Heading>
                  <Text fontSize="sm" opacity={0.9}>
                    Has asistido a todas tus clases esta semana. ¡Gran trabajo!
                  </Text>
                </CardBody>
              </MotionCard>
            </Box>
          </SimpleGrid>
        </MotionBox>
      </Container>
    </Layout>
  );
};

export default AlumnoDashboard;

