import React from 'react';
import {
    Box,
    Container,
    Heading,
    Text,
    SimpleGrid,
    Card,
    CardBody,
    Badge,
    VStack,
    HStack,
    Icon,
    Button,
    Flex,
    Divider
} from '@chakra-ui/react';
import { FiDollarSign, FiCalendar, FiCheckCircle, FiAlertCircle, FiClock } from 'react-icons/fi';
import { motion } from 'framer-motion';
import Layout from '../../components/Layout';
import { mockPagos } from '../../data/mockData';

const MotionCard = motion(Card);

const containerVariants = {
    hidden: { opacity: 0 },
    visible: {
        opacity: 1,
        transition: {
            staggerChildren: 0.1
        }
    }
};

const itemVariants = {
    hidden: { y: 20, opacity: 0 },
    visible: {
        y: 0,
        opacity: 1,
        transition: {
            type: "spring",
            stiffness: 100
        }
    }
};

const alumnoLinks = [
    { nombre: 'Dashboard', path: '/alumno' },
    { nombre: 'Mis Clases', path: '/alumno/clases' },
    { nombre: 'Bitácoras', path: '/alumno/bitacoras' },
    { nombre: 'Pagos', path: '/alumno/pagos' },
];

// Simulando alumno ID 1
const alumnoId = 1;
const misPagos = mockPagos.filter(p => p.id_alumno === alumnoId);

const getStatusColor = (status) => {
    switch (status) {
        case 'Aprobado':
            return 'green';
        case 'Pendiente':
            return 'orange';
        case 'Rechazado':
            return 'red';
        default:
            return 'gray';
    }
};

const getStatusIcon = (status) => {
    switch (status) {
        case 'Aprobado':
            return FiCheckCircle;
        case 'Pendiente':
            return FiClock;
        case 'Rechazado':
            return FiAlertCircle;
        default:
            return FiDollarSign;
    }
};

const PagoCard = ({ pago }) => (
    <MotionCard
        variants={itemVariants}
        bg="white"
        borderRadius="2xl"
        boxShadow="sm"
        border="1px solid"
        borderColor="gray.100"
        whileHover={{ y: -2, boxShadow: 'md' }}
        transition={{ duration: 0.2 }}
    >
        <CardBody>
            <Flex justify="space-between" align="start" mb={4}>
                <VStack align="start" spacing={1}>
                    <Text fontSize="lg" fontWeight="700" color="gray.800">
                        ${pago.monto.toLocaleString()}
                    </Text>
                    <Text fontSize="sm" color="gray.500">
                        Mensualidad
                    </Text>
                </VStack>
                <Badge
                    colorScheme={getStatusColor(pago.estatus)}
                    borderRadius="full"
                    px={3}
                    py={1}
                    fontSize="xs"
                    fontWeight="600"
                >
                    <HStack spacing={1}>
                        <Icon as={getStatusIcon(pago.estatus)} boxSize={3} />
                        <Text>{pago.estatus}</Text>
                    </HStack>
                </Badge>
            </Flex>

            <Divider mb={4} />

            <VStack align="stretch" spacing={3}>
                <HStack>
                    <Icon as={FiCalendar} color="gray.400" boxSize={4} />
                    <Text fontSize="sm" color="gray.600">
                        Fecha de pago: <strong>{pago.fecha_pago}</strong>
                    </Text>
                </HStack>

                {pago.comprobante_url && (
                    <Button
                        size="sm"
                        variant="outline"
                        colorScheme="blue"
                        rightIcon={<Icon as={FiCheckCircle} />}
                    >
                        Ver comprobante
                    </Button>
                )}
            </VStack>
        </CardBody>
    </MotionCard>
);

const PagosAlumno = () => {
    const pagosPendientes = misPagos.filter(p => p.estatus === 'Pendiente').length;
    const pagosAprobados = misPagos.filter(p => p.estatus === 'Aprobado').length;
    const totalPagado = misPagos
        .filter(p => p.estatus === 'Aprobado')
        .reduce((acc, p) => acc + p.monto, 0);

    return (
        <Layout
            links={alumnoLinks}
            userType="Alumno"
            userName="Sofía López"
            userPhoto="https://images.unsplash.com/photo-1517841905240-472988babdf9?ixlib=rb-1.2.1&auto=format&fit=facearea&facepad=2&w=256&h=256&q=80"
        >
            <Container maxW="container.xl" px={0}>
                <motion.div
                    initial="hidden"
                    animate="visible"
                    variants={containerVariants}
                >
                    <Box mb={8}>
                        <Heading size="lg" color="gray.800" mb={2}>
                            Mis Pagos
                        </Heading>
                        <Text color="gray.600">
                            Historial y estado de tus pagos mensuales
                        </Text>
                    </Box>

                    {/* Stats */}
                    <SimpleGrid columns={{ base: 1, md: 3 }} spacing={6} mb={8}>
                        <MotionCard
                            variants={itemVariants}
                            bg="white"
                            borderRadius="2xl"
                            p={6}
                            border="1px solid"
                            borderColor="gray.100"
                        >
                            <VStack align="start" spacing={2}>
                                <Icon as={FiDollarSign} boxSize={8} color="green.500" />
                                <Text fontSize="3xl" fontWeight="800" color="gray.800">
                                    ${totalPagado.toLocaleString()}
                                </Text>
                                <Text fontSize="sm" color="gray.500" fontWeight="600">
                                    Total Pagado
                                </Text>
                            </VStack>
                        </MotionCard>

                        <MotionCard
                            variants={itemVariants}
                            bg="white"
                            borderRadius="2xl"
                            p={6}
                            border="1px solid"
                            borderColor="gray.100"
                        >
                            <VStack align="start" spacing={2}>
                                <Icon as={FiCheckCircle} boxSize={8} color="blue.500" />
                                <Text fontSize="3xl" fontWeight="800" color="gray.800">
                                    {pagosAprobados}
                                </Text>
                                <Text fontSize="sm" color="gray.500" fontWeight="600">
                                    Pagos Aprobados
                                </Text>
                            </VStack>
                        </MotionCard>

                        <MotionCard
                            variants={itemVariants}
                            bg="white"
                            borderRadius="2xl"
                            p={6}
                            border="1px solid"
                            borderColor="gray.100"
                        >
                            <VStack align="start" spacing={2}>
                                <Icon as={FiClock} boxSize={8} color="orange.500" />
                                <Text fontSize="3xl" fontWeight="800" color="gray.800">
                                    {pagosPendientes}
                                </Text>
                                <Text fontSize="sm" color="gray.500" fontWeight="600">
                                    Pagos Pendientes
                                </Text>
                            </VStack>
                        </MotionCard>
                    </SimpleGrid>

                    {/* Payment List */}
                    <Box>
                        <Heading size="md" mb={4} color="gray.800">
                            Historial de Pagos
                        </Heading>
                        <SimpleGrid columns={{ base: 1, md: 2, lg: 3 }} spacing={6}>
                            {misPagos.map((pago) => (
                                <PagoCard key={pago.id} pago={pago} />
                            ))}
                        </SimpleGrid>
                    </Box>
                </motion.div>
            </Container>
        </Layout>
    );
};

export default PagosAlumno;
