import { useState, useEffect } from 'react';
import {
    Box,
    SimpleGrid,
    Stat,
    StatLabel,
    StatNumber,
    StatHelpText,
    Card,
    CardBody,
    Heading,
    Text,
    Icon,
    Flex,
    Badge,
    VStack,
    Button,
    HStack,
    useToast,
    useDisclosure,
    Modal,
    ModalOverlay,
    ModalContent,
    ModalHeader,
    ModalCloseButton,
    ModalBody,
    ModalFooter
} from '@chakra-ui/react';
import { FiClock, FiCheckCircle, FiPlusCircle, FiAlertCircle } from 'react-icons/fi';
import { Link as RouterLink } from 'react-router-dom';
import { getContractedServicesByClient, getServicesByCompany } from '../../services/api';

// Función para traducir estados a leyendas amigables
const getStatusLabel = (status) => {
    const statusMap = {
        'PENDING': 'Pendiente',
        'IN_PROGRESS': 'En Progreso',
        'FINISHED': 'Finalizado',
        'CANCELLED': 'Cancelado',
        'CERRADO_ADMIN': 'Cerrado',
        'IN_REVIEW': 'Reapertura'
    };
    return statusMap[status] || status;
};

const ClientDashboard = () => {
    const [selectedService, setSelectedService] = useState(null);
    const [evidences, setEvidences] = useState([]);
    const [myServices, setMyServices] = useState([]);
    const [servicesCatalog, setServicesCatalog] = useState([]);
    const { isOpen, onOpen, onClose } = useDisclosure();
    const toast = useToast();

    useEffect(() => {
        const fetchData = async () => {
            try {
                const user = JSON.parse(sessionStorage.getItem('user'));
                if (user) {
                    const [services, companyServices] = await Promise.all([
                        getContractedServicesByClient(user.id),
                        getServicesByCompany(user.company_id)
                    ]);
                    // Sort services by ID descending (newest first)
                    setMyServices(services.sort((a, b) => b.id - a.id));
                    setServicesCatalog(companyServices);
                }
            } catch (error) {
                console.error(error);
                toast({
                    title: 'Error al cargar dashboard',
                    status: 'error',
                    duration: 3000,
                });
            }
        };
        fetchData();
    }, []);

    const activeServices = myServices.filter(s => s.status === 'IN_PROGRESS' || s.status === 'PENDING').length;
    const finishedServices = myServices.filter(s => s.status === 'FINISHED').length;
    // Calculate warranties: services that are FINISHED and have a warranty_end_date in the future
    const activeWarranties = myServices.filter(s =>
        s.status === 'FINISHED' &&
        s.warranty_end_date &&
        new Date(s.warranty_end_date) > new Date()
    ).length;

    const getServiceName = (id) => servicesCatalog.find(s => s.id === id)?.name || 'Desconocido';

    const handleViewDetails = async (service) => {
        setSelectedService(service);
        try {
            const { getEvidencesByService } = await import('../../services/api');
            const serviceEvidences = await getEvidencesByService(service.id);
            setEvidences(serviceEvidences);
        } catch (error) {
            console.error(error);
            setEvidences([]);
        }
        onOpen();
    };

    return (
        <Box>
            <Flex justify="space-between" align="center" mb={6}>
                <Heading>Mi Resumen</Heading>
                <Button
                    as={RouterLink}
                    to="/client/solicitar"
                    leftIcon={<FiPlusCircle />}
                    colorScheme="brand"
                >
                    Solicitar Nuevo Servicio
                </Button>
            </Flex>

            {/* Stats Grid */}
            <SimpleGrid columns={{ base: 1, md: 3 }} spacing={6} mb={8}>
                <Card>
                    <CardBody>
                        <Stat>
                            <Flex alignItems="center" mb={2}>
                                <Box p={2} bg="blue.100" borderRadius="md" mr={3}>
                                    <Icon as={FiClock} color="blue.500" boxSize={6} />
                                </Box>
                                <StatLabel fontSize="lg">Servicios Activos</StatLabel>
                            </Flex>
                            <StatNumber>{activeServices}</StatNumber>
                            <StatHelpText>En proceso o pendientes</StatHelpText>
                        </Stat>
                    </CardBody>
                </Card>

                <Card>
                    <CardBody>
                        <Stat>
                            <Flex alignItems="center" mb={2}>
                                <Box p={2} bg="green.100" borderRadius="md" mr={3}>
                                    <Icon as={FiCheckCircle} color="green.500" boxSize={6} />
                                </Box>
                                <StatLabel fontSize="lg">Finalizados</StatLabel>
                            </Flex>
                            <StatNumber>{finishedServices}</StatNumber>
                            <StatHelpText>Total histórico</StatHelpText>
                        </Stat>
                    </CardBody>
                </Card>

                <Card>
                    <CardBody>
                        <Stat>
                            <Flex alignItems="center" mb={2}>
                                <Box p={2} bg="purple.100" borderRadius="md" mr={3}>
                                    <Icon as={FiAlertCircle} color="purple.500" boxSize={6} />
                                </Box>
                                <StatLabel fontSize="lg">Garantías</StatLabel>
                            </Flex>
                            <StatNumber>{activeWarranties}</StatNumber>
                            <StatHelpText>Vigentes</StatHelpText>
                        </Stat>
                    </CardBody>
                </Card>
            </SimpleGrid>

            {/* Active Services List */}
            <Heading size="md" mb={4}>Servicios en Curso</Heading>
            <VStack spacing={4} align="stretch">
                {myServices
                    .filter(s => s.status === 'IN_PROGRESS' || s.status === 'PENDING' || s.status === 'FINISHED')
                    .map((service) => (
                        <Card key={service.id} borderLeft="4px solid" borderColor={
                            service.status === 'IN_PROGRESS' ? 'blue.500' :
                            service.status === 'PENDING' ? 'orange.500' : 'green.500'
                        }>
                            <CardBody>
                                <Flex justify="space-between" align="center" wrap="wrap" gap={4}>
                                    <Box>
                                        <HStack mb={1}>
                                            <Badge colorScheme={
                                                service.status === 'IN_PROGRESS' ? 'blue' :
                                                service.status === 'PENDING' ? 'orange' : 'green'
                                            }>
                                                {service.status === 'IN_PROGRESS' ? 'EN PROGRESO' : 
                                                 service.status === 'PENDING' ? 'PENDIENTE' : 'FINALIZADO (Acción requerida)'}
                                            </Badge>
                                            <Text fontSize="sm" color="gray.500">Folio: {service.service_number}</Text>
                                        </HStack>
                                        <Heading size="sm" mb={1}>{getServiceName(service.service_catalogo_id)}</Heading>
                                        <Text fontSize="sm" color="gray.600">
                                            Fecha Solicitud: {new Date(service.request_date).toLocaleDateString()}
                                        </Text>
                                    </Box>
                                    <Box textAlign="right">
                                        <Button
                                            as={RouterLink}
                                            to="/client/historial"
                                            size="sm"
                                            variant="outline"
                                            colorScheme="brand"
                                        >
                                            {service.status === 'FINISHED' ? 'Revisar y Cerrar' : 'Ver Detalles'}
                                        </Button>
                                    </Box>
                                </Flex>
                            </CardBody>
                        </Card>
                    ))}
                {activeServices === 0 && (
                    <Card>
                        <CardBody textAlign="center" py={8}>
                            <Text color="gray.500">No tienes servicios activos en este momento.</Text>
                            <Button
                                as={RouterLink}
                                to="/client/solicitar"
                                variant="link"
                                colorScheme="brand"
                                mt={2}
                            >
                                Solicitar uno ahora
                            </Button>
                        </CardBody>
                    </Card>
                )}
            </VStack>

            {/* Modal Detalles (Reused from HistorialServicios logic) */}
            <Modal isOpen={isOpen} onClose={onClose} size="lg">
                <ModalOverlay />
                <ModalContent>
                    <ModalHeader>Detalle del Servicio</ModalHeader>
                    <ModalCloseButton />
                    <ModalBody>
                        {selectedService && (
                            <VStack align="stretch" spacing={4}>
                                <Flex justify="space-between">
                                    <Box>
                                        <Text fontSize="sm" color="gray.500">Servicio</Text>
                                        <Heading size="sm">{getServiceName(selectedService.service_catalogo_id)}</Heading>
                                    </Box>
                                    <Box textAlign="right">
                                        <Text fontSize="sm" color="gray.500">Folio</Text>
                                        <Text fontWeight="bold">{selectedService.service_number}</Text>
                                    </Box>
                                </Flex>

                                <Box h="1px" bg="gray.200" />

                                <SimpleGrid columns={2} spacing={4}>
                                    <Box>
                                        <Text fontSize="sm" color="gray.500">Fecha Solicitud</Text>
                                        <Text>{new Date(selectedService.request_date).toLocaleDateString()}</Text>
                                    </Box>
                                    <Box>
                                        <Text fontSize="sm" color="gray.500">Estado</Text>
                                        <Badge>{getStatusLabel(selectedService.status)}</Badge>
                                    </Box>
                                </SimpleGrid>

                                {evidences.length > 0 && (
                                    <Box>
                                        <Text fontWeight="bold" mb={2}>Evidencias</Text>
                                        <SimpleGrid columns={2} spacing={2}>
                                            {evidences.map(ev => (
                                                <Box key={ev.id} border="1px solid" borderColor="gray.200" p={1} borderRadius="md">
                                                    <Box
                                                        height="100px"
                                                        width="100%"
                                                        mb={1}
                                                        bgImage={`url(${ev.file_url})`}
                                                        bgSize="cover"
                                                        bgPosition="center"
                                                        borderRadius="sm"
                                                    />
                                                    <Text fontSize="xs" noOfLines={2}>{ev.description}</Text>
                                                </Box>
                                            ))}
                                        </SimpleGrid>
                                    </Box>
                                )}
                            </VStack>
                        )}
                    </ModalBody>
                    <ModalFooter>
                        <Button variant="ghost" mr={3} onClick={onClose}>
                            Cerrar
                        </Button>
                    </ModalFooter>
                </ModalContent>
            </Modal>
        </Box>
    );
};

export default ClientDashboard;
