import {
    Box,
    Heading,
    Table,
    Thead,
    Tbody,
    Tr,
    Th,
    Td,
    Badge,
    Button,
    useDisclosure,
    Modal,
    ModalOverlay,
    ModalContent,
    ModalHeader,
    ModalFooter,
    ModalBody,
    ModalCloseButton,
    VStack,
    HStack,
    Text,
    Divider,
    SimpleGrid,
    Image,
    useToast,
    Textarea,
    IconButton,
    FormControl,
    FormLabel,
} from '@chakra-ui/react';
import { useState, useEffect } from 'react';
import { FiEye, FiDownload, FiCheckCircle, FiAlertCircle, FiStar } from 'react-icons/fi';
import { getContractedServicesByClient, getServicesByCompany, getEvidencesByService, updateOrder } from '../../services/api';
import { FilePond, registerPlugin } from 'react-filepond';
import FilePondPluginImagePreview from 'filepond-plugin-image-preview';
import FilePondPluginFileValidateType from 'filepond-plugin-file-validate-type';
import 'filepond/dist/filepond.min.css';
import "filepond-plugin-image-preview/dist/filepond-plugin-image-preview.css";

// Register FilePond plugins
registerPlugin(FilePondPluginImagePreview, FilePondPluginFileValidateType);

// Función para traducir estados a leyendas amigables
const getStatusLabel = (status) => {
    const statusMap = {
        'PENDING': 'Pendiente',
        'IN_PROGRESS': 'En Progreso',
        'FINISHED': 'Finalizado',
        'CANCELLED': 'Cancelado',
        'CERRADO_ADMIN': 'Cerrado',
        'IN_REVIEW': 'Reapertura'
    };
    return statusMap[status] || status;
};

const StarRating = ({ rating, setRating }) => {
    return (
        <HStack spacing={1}>
            {[1, 2, 3, 4, 5].map((star) => (
                <IconButton
                    key={star}
                    aria-label={`Calificar con ${star} estrellas`}
                    icon={<FiStar color={star <= rating ? 'orange.400' : 'gray.300'} fill={star <= rating ? 'orange.400' : 'none'} />}
                    onClick={() => setRating(star)}
                    variant="ghost"
                    size="md"
                    _hover={{ bg: 'transparent' }}
                    _focus={{ boxShadow: 'none' }}
                />
            ))}
        </HStack>
    );
};

const HistorialServicios = () => {
    const [myServices, setMyServices] = useState([]);
    const [servicesCatalog, setServicesCatalog] = useState([]);
    const [selectedService, setSelectedService] = useState(null);
    const [evidences, setEvidences] = useState([]);
    const { isOpen, onOpen, onClose } = useDisclosure();
    const toast = useToast();

    // Modales adicionales
    const { isOpen: isCloseOpen, onOpen: onCloseOpen, onClose: onCloseClose } = useDisclosure();
    const { isOpen: isReopenOpen, onOpen: onReopenOpen, onClose: onReopenClose } = useDisclosure();

    // Estado para calificación y comentarios
    const [rating, setRating] = useState(0);
    const [comment, setComment] = useState("");
    
    // Estado para el modal de reapertura
    const [reopenReason, setReopenReason] = useState("");
    const [reopenFiles, setReopenFiles] = useState([]);

    useEffect(() => {
        const fetchData = async () => {
            try {
                const user = JSON.parse(sessionStorage.getItem('user'));
                if (user) {
                    const [services, companyServices] = await Promise.all([
                        getContractedServicesByClient(user.id),
                        getServicesByCompany(user.company_id)
                    ]);
                    setMyServices(services.sort((a, b) => b.id - a.id));
                    setServicesCatalog(companyServices);
                }
            } catch (error) {
                console.error(error);
                toast({
                    title: 'Error al cargar historial',
                    status: 'error',
                    duration: 3000,
                });
            }
        };
        fetchData();
    }, []);

    const getServiceName = (id) => servicesCatalog.find(s => s.id === id)?.name || 'Desconocido';

    const handleViewDetails = async (service) => {
        setSelectedService(service);
        try {
            const serviceEvidences = await getEvidencesByService(service.id);
            setEvidences(serviceEvidences);
        } catch (error) {
            console.error(error);
            setEvidences([]);
        }
        onOpen();
    };

    const handleUpdateService = async (updateData) => {
        try {
            const updatedService = await updateOrder(selectedService.id, updateData);
            
            // Actualizar la lista de servicios en el estado local
            setMyServices(myServices.map(s => s.id === selectedService.id ? updatedService : s));
            setSelectedService(updatedService); // Actualizar también el servicio seleccionado

            toast({
                title: "Estado actualizado",
                status: "success",
                duration: 2000,
            });

            // Cerrar todos los modales
            onClose();
            onCloseClose();
            onReopenClose();
        } catch (error) {
            toast({
                title: "Error al actualizar",
                description: error.message,
                status: "error",
                duration: 3000,
            });
        }
    };

    const handleCloseService = () => {
        if (rating === 0) {
            toast({
                title: "Calificación requerida",
                description: "Por favor, selecciona una calificación.",
                status: "warning",
                duration: 3000,
            });
            return;
        }
        handleUpdateService({
            status: 'CERRADO_ADMIN', // Usar el valor correcto de la BD
            client_rating: rating,
            client_comment: comment,
        });
    };

    const handleReopenService = async () => {
        if (!reopenReason.trim()) {
            toast({
                title: "Motivo requerido",
                description: "Por favor, explica por qué reabres el servicio.",
                status: "warning",
                duration: 3000,
            });
            return;
        }
    
        try {
            // Paso 1: Subir evidencias si existen
            if (reopenFiles.length > 0) {
                const formData = new FormData();
                formData.append("contracted_service_id", selectedService.id);
                formData.append("evidence_type", "PHOTO");
    
                reopenFiles.forEach((file) => {
                    formData.append("evidences", file.file);
                    formData.append(
                        "descriptions",
                        `Evidencia de reapertura: ${reopenReason.substring(0, 50)}`,
                    );
                });
    
                const response = await fetch("/api/evidences", {
                    method: "POST",
                    headers: {
                        Authorization: `Bearer ${sessionStorage.getItem("token")}`,
                    },
                    body: formData,
                });
    
                if (!response.ok) {
                    throw new Error("Error al subir las nuevas evidencias");
                }
            }
    
            // Paso 2: Actualizar el estado del servicio
            await handleUpdateService({
                status: 'IN_REVIEW',
                client_comment: `Motivo de reapertura: ${reopenReason}`,
            });
    
            // Limpiar estados
            setReopenReason("");
            setReopenFiles([]);
    
        } catch (error) {
            toast({
                title: "Error al reabrir",
                description: error.message,
                status: "error",
                duration: 3000,
            });
        }
    };

    const serviceEvidences = evidences;

    return (
        <Box>
            <Heading mb={6}>Historial de Servicios</Heading>

            <Box bg="white" borderRadius="lg" boxShadow="sm" overflowX="auto">
                <Table variant="simple">
                    <Thead bg="gray.50">
                        <Tr>
                            <Th>Folio</Th>
                            <Th>Servicio</Th>
                            <Th>Fecha Solicitud</Th>
                            <Th>Fecha Cierre</Th>
                            <Th>Estado</Th>
                            <Th>Acciones</Th>
                        </Tr>
                    </Thead>
                    <Tbody>
                        {myServices.map((service) => (
                            <Tr key={service.id}>
                                <Td fontWeight="bold">{service.service_number}</Td>
                                <Td>{getServiceName(service.service_catalogo_id)}</Td>
                                <Td>{new Date(service.request_date).toLocaleDateString()}</Td>
                                <Td>
                                    {service.end_date ? new Date(service.end_date).toLocaleDateString() : '-'}
                                </Td>
                                <Td>
                                    <Badge
                                        colorScheme={
                                            service.status === 'FINISHED' ? 'green' :
                                                service.status === 'IN_PROGRESS' ? 'blue' :
                                                    service.status === 'PENDING' ? 'orange' : 
                                                    service.status === 'IN_REVIEW' ? 'purple' : 'gray'
                                        }
                                    >
                                        {service.status}
                                    </Badge>
                                </Td>
                                <Td>
                                    <Button
                                        leftIcon={<FiEye />}
                                        size="sm"
                                        variant="ghost"
                                        onClick={() => handleViewDetails(service)}
                                    >
                                        Ver
                                    </Button>
                                </Td>
                            </Tr>
                        ))}
                    </Tbody>
                </Table>
            </Box>

            {/* Modal Detalles */}
            <Modal isOpen={isOpen} onClose={onClose} size="lg">
                <ModalOverlay />
                <ModalContent>
                    <ModalHeader>Detalle del Servicio</ModalHeader>
                    <ModalCloseButton />
                    <ModalBody>
                        {selectedService && (
                            <VStack align="stretch" spacing={4}>
                                <HStack justify="space-between">
                                    <Box>
                                        <Text fontSize="sm" color="gray.500">Servicio</Text>
                                        <Heading size="sm">{getServiceName(selectedService.service_catalogo_id)}</Heading>
                                    </Box>
                                    <Box textAlign="right">
                                        <Text fontSize="sm" color="gray.500">Folio</Text>
                                        <Text fontWeight="bold">{selectedService.service_number}</Text>
                                    </Box>
                                </HStack>

                                <Divider />

                                <SimpleGrid columns={2} spacing={4}>
                                    <Box>
                                        <Text fontSize="sm" color="gray.500">Fecha Solicitud</Text>
                                        <Text>{new Date(selectedService.request_date).toLocaleDateString()}</Text>
                                    </Box>
                                    <Box>
                                        <Text fontSize="sm" color="gray.500">Estado</Text>
                                        <Badge>{getStatusLabel(selectedService.status)}</Badge>
                                    </Box>
                                </SimpleGrid>

                                {serviceEvidences.length > 0 && (
                                    <Box>
                                        <Text fontWeight="bold" mb={2}>Evidencias</Text>
                                        <SimpleGrid columns={2} spacing={2}>
                                            {serviceEvidences.map(ev => (
                                                <Box key={ev.id} border="1px solid" borderColor="gray.200" p={1} borderRadius="md">
                                                    <Image src={ev.file_url} height="100px" width="100%" objectFit="cover" mb={1} />
                                                    <Text fontSize="xs" noOfLines={2}>{ev.description}</Text>
                                                </Box>
                                            ))}
                                        </SimpleGrid>
                                    </Box>
                                )}
                            </VStack>
                        )}
                    </ModalBody>
                    <ModalFooter>
                        <Button variant="ghost" mr={3} onClick={onClose}>
                            Cerrar
                        </Button>
                        {selectedService?.status === 'FINISHED' && (
                            <HStack>
                                <Button
                                    leftIcon={<FiAlertCircle />}
                                    colorScheme="orange"
                                    onClick={onReopenOpen}
                                >
                                    Reabrir para Revisión
                                </Button>
                                <Button
                                    leftIcon={<FiCheckCircle />}
                                    colorScheme="brand"
                                    onClick={onCloseOpen}
                                >
                                    Cerrar Servicio
                                </Button>
                            </HStack>
                        )}
                    </ModalFooter>
                </ModalContent>
            </Modal>

            {/* Modal para Cerrar Servicio */}
            <Modal isOpen={isCloseOpen} onClose={onCloseClose}>
                <ModalOverlay />
                <ModalContent>
                    <ModalHeader>Calificar y Cerrar Servicio</ModalHeader>
                    <ModalCloseButton />
                    <ModalBody>
                        <VStack spacing={4}>
                            <Text>
                                Gracias por confirmar la finalización. Por favor, califica el
                                servicio recibido.
                            </Text>
                            <FormControl>
                                <FormLabel>Calificación (obligatorio)</FormLabel>
                                <StarRating rating={rating} setRating={setRating} />
                            </FormControl>
                            <FormControl>
                                <FormLabel>Comentario (opcional)</FormLabel>
                                <Textarea
                                    value={comment}
                                    onChange={(e) => setComment(e.target.value)}
                                    placeholder="¿Cómo fue tu experiencia?"
                                />
                            </FormControl>
                        </VStack>
                    </ModalBody>
                    <ModalFooter>
                        <Button variant="ghost" mr={3} onClick={onCloseClose}>
                            Cancelar
                        </Button>
                        <Button
                            colorScheme="brand"
                            onClick={handleCloseService}
                            isDisabled={rating === 0}
                        >
                            Confirmar y Cerrar
                        </Button>
                    </ModalFooter>
                </ModalContent>
            </Modal>

            {/* Modal para Reabrir Servicio */}
            <Modal isOpen={isReopenOpen} onClose={onReopenClose} size="lg">
                <ModalOverlay />
                <ModalContent>
                    <ModalHeader>Reabrir para Revisión</ModalHeader>
                    <ModalCloseButton />
                    <ModalBody>
                        <VStack spacing={4}>
                            <FormControl isRequired>
                                <FormLabel>Motivo de la reapertura</FormLabel>
                                <Textarea
                                    value={reopenReason}
                                    onChange={(e) => setReopenReason(e.target.value)}
                                    placeholder="Describe el problema o la razón por la que no estás satisfecho con el servicio."
                                />
                            </FormControl>
                            <FormControl>
                                <FormLabel>Adjuntar evidencia (opcional, hasta 4 fotos)</FormLabel>
                                <FilePond
                                    files={reopenFiles}
                                    onupdatefiles={setReopenFiles}
                                    allowMultiple={true}
                                    maxFiles={4}
                                    acceptedFileTypes={["image/*"]}
                                    name="reopenEvidences"
                                    labelIdle='Arrastra y suelta fotos aquí o <span class="filepond--label-action">Examina</span>'
                                    credits={false}
                                />
                            </FormControl>
                        </VStack>
                    </ModalBody>
                    <ModalFooter>
                        <Button variant="ghost" mr={3} onClick={onReopenClose}>
                            Cancelar
                        </Button>
                        <Button
                            colorScheme="orange"
                            onClick={handleReopenService}
                            isDisabled={!reopenReason.trim()}
                        >
                            Confirmar Reapertura
                        </Button>
                    </ModalFooter>
                </ModalContent>
            </Modal>
        </Box>
    );
};

export default HistorialServicios;
