import { useState, useEffect } from 'react';
import {
    Box,
    Heading,
    SimpleGrid,
    Card,
    CardBody,
    CardHeader,
    Text,
    Button,
    Badge,
    useDisclosure,
    Modal,
    ModalOverlay,
    ModalContent,
    ModalHeader,
    ModalFooter,
    ModalBody,
    ModalCloseButton,
    FormControl,
    FormLabel,
    Input,
    Textarea,
    useToast,
    VStack
} from '@chakra-ui/react';
import { getServicesByCompany, createOrder } from '../../services/api';

const SolicitarServicio = () => {
    const [servicesCatalog, setServicesCatalog] = useState([]);
    const [selectedService, setSelectedService] = useState(null);
    const { isOpen, onOpen, onClose } = useDisclosure();
    const toast = useToast();
    const [requestDetails, setRequestDetails] = useState({
        date: '',
        notes: ''
    });

    useEffect(() => {
        const fetchServices = async () => {
            try {
                const user = JSON.parse(sessionStorage.getItem('user'));
                if (user && user.company_id) {
                    const services = await getServicesByCompany(user.company_id);
                    setServicesCatalog(services);
                }
            } catch (error) {
                console.error(error);
                toast({
                    title: 'Error al cargar servicios',
                    status: 'error',
                    duration: 3000,
                });
            }
        };
        fetchServices();
    }, []);

    const handleSelectService = (service) => {
        setSelectedService(service);
        onOpen();
    };

    const handleSubmit = async () => {
        if (!requestDetails.date) {
            toast({
                title: 'Error',
                description: 'Por favor selecciona una fecha preferente',
                status: 'error',
                duration: 2000,
            });
            return;
        }

        try {
            const user = JSON.parse(sessionStorage.getItem('user'));
            await createOrder({
                service_number: `REQ-${Date.now()}`,
                client_id: user.id,
                service_catalogo_id: selectedService.id,
                status: 'PENDING',
                request_date: requestDetails.date, // Using request_date as preferred date
                company_id: user.company_id,
                // notes: requestDetails.notes // Assuming we might add notes field later or use it in description if available
            });

            toast({
                title: 'Solicitud enviada',
                description: `Hemos recibido tu solicitud para ${selectedService.name}. Te contactaremos pronto.`,
                status: 'success',
                duration: 4000,
            });
            onClose();
            setRequestDetails({ date: '', notes: '' });
        } catch (error) {
            toast({
                title: 'Error al enviar solicitud',
                description: error.message,
                status: 'error',
                duration: 3000,
            });
        }
    };

    // Agrupar servicios por categoría
    const categories = [...new Set(servicesCatalog.map(s => s.category))];

    return (
        <Box>
            <Heading mb={6}>Solicitar Nuevo Servicio</Heading>
            <Text mb={8} color="gray.600">
                Selecciona el servicio que necesitas de nuestro catálogo.
            </Text>

            {categories.map(category => (
                <Box key={category} mb={8}>
                    <Heading size="md" mb={4} color="brand.600">{category}</Heading>
                    <SimpleGrid columns={{ base: 1, md: 2, lg: 3 }} spacing={6}>
                        {servicesCatalog
                            .filter(s => s.category === category)
                            .map(service => (
                                <Card key={service.id} _hover={{ shadow: 'md', transform: 'translateY(-2px)', transition: 'all 0.2s' }}>
                                    <CardHeader pb={2}>
                                        <Heading size="sm">{service.name}</Heading>
                                        {service.has_warranty && (
                                            <Badge colorScheme="green" mt={2} fontSize="xs">
                                                Garantía: {service.warranty_months} Meses
                                            </Badge>
                                        )}
                                    </CardHeader>
                                    <CardBody>
                                        <Text fontSize="sm" color="gray.600" mb={4} noOfLines={3}>
                                            {service.description}
                                        </Text>
                                        <Button
                                            colorScheme="brand"
                                            width="full"
                                            variant="outline"
                                            onClick={() => handleSelectService(service)}
                                        >
                                            Solicitar
                                        </Button>
                                    </CardBody>
                                </Card>
                            ))}
                    </SimpleGrid>
                </Box>
            ))}

            {/* Modal de Solicitud */}
            <Modal isOpen={isOpen} onClose={onClose}>
                <ModalOverlay />
                <ModalContent>
                    <ModalHeader>Solicitar {selectedService?.name}</ModalHeader>
                    <ModalCloseButton />
                    <ModalBody>
                        <VStack spacing={4}>
                            <Text fontSize="sm" color="gray.600">
                                Completa los detalles para agendar tu servicio. Un asesor confirmará la disponibilidad.
                            </Text>

                            <FormControl isRequired>
                                <FormLabel>Fecha Preferente</FormLabel>
                                <Input
                                    type="date"
                                    min={new Date().toISOString().split('T')[0]}
                                    value={requestDetails.date}
                                    onChange={(e) => setRequestDetails({ ...requestDetails, date: e.target.value })}
                                />
                            </FormControl>

                            <FormControl>
                                <FormLabel>Notas Adicionales</FormLabel>
                                <Textarea
                                    placeholder="Describe brevemente el problema o requerimientos especiales..."
                                    value={requestDetails.notes}
                                    onChange={(e) => setRequestDetails({ ...requestDetails, notes: e.target.value })}
                                />
                            </FormControl>
                        </VStack>
                    </ModalBody>

                    <ModalFooter>
                        <Button variant="ghost" mr={3} onClick={onClose}>
                            Cancelar
                        </Button>
                        <Button colorScheme="brand" onClick={handleSubmit}>
                            Enviar Solicitud
                        </Button>
                    </ModalFooter>
                </ModalContent>
            </Modal>
        </Box>
    );
};

export default SolicitarServicio;
