import { useState, useEffect } from 'react';
import {
    Box,
    Heading,
    Card,
    CardBody,
    Text,
    Badge,
    VStack,
    HStack,
    Divider,
    useToast
} from '@chakra-ui/react';
import { getContractedServicesByEmployee, getServicesByCompany } from '../../services/api';

const CalendarioTrabajo = () => {
    const [myServices, setMyServices] = useState([]);
    const [servicesCatalog, setServicesCatalog] = useState([]);
    const toast = useToast();

    useEffect(() => {
        const fetchData = async () => {
            try {
                const user = JSON.parse(sessionStorage.getItem('user'));
                if (user) {
                    const { getEmployeeByUserId } = await import('../../services/api');
                    const employee = await getEmployeeByUserId(user.id);

                    if (employee) {
                        const [services, companyServices] = await Promise.all([
                            getContractedServicesByEmployee(employee.id),
                            getServicesByCompany(user.company_id)
                        ]);

                        setMyServices(services);
                        setServicesCatalog(companyServices);
                    }
                }
            } catch (error) {
                console.error(error);
                toast({
                    title: 'Error al cargar calendario',
                    status: 'error',
                    duration: 3000,
                });
            }
        };
        fetchData();
    }, []);

    const getServiceName = (id) => servicesCatalog.find(s => s.id === id)?.name || 'Desconocido';

    // Agrupar por fecha
    const today = new Date().toISOString().split('T')[0];

    const todayServices = myServices.filter(s => s.deadline_date && s.deadline_date.startsWith(today));
    const futureServices = myServices.filter(s => s.deadline_date && s.deadline_date > today);

    return (
        <Box>
            <Heading mb={6}>Calendario de Trabajo</Heading>

            <VStack spacing={6} align="stretch">
                <Box>
                    <Heading size="md" mb={4} color="brand.600">Para Hoy ({today})</Heading>
                    {todayServices.length > 0 ? (
                        todayServices.map(service => (
                            <Card key={service.id} mb={3} borderLeft="4px solid" borderColor="brand.500">
                                <CardBody>
                                    <HStack justify="space-between">
                                        <Box>
                                            <Heading size="sm">{getServiceName(service.service_catalogo_id)}</Heading>
                                            <Text fontSize="sm" color="gray.600">Folio: {service.service_number}</Text>
                                        </Box>
                                        <Badge>{service.status}</Badge>
                                    </HStack>
                                </CardBody>
                            </Card>
                        ))
                    ) : (
                        <Text color="gray.500">No hay servicios programados para hoy.</Text>
                    )}
                </Box>

                <Divider />

                <Box>
                    <Heading size="md" mb={4}>Próximos Días</Heading>
                    {futureServices.length > 0 ? (
                        futureServices.map(service => (
                            <Card key={service.id} mb={3}>
                                <CardBody>
                                    <HStack justify="space-between">
                                        <Box>
                                            <Text fontSize="xs" fontWeight="bold" color="gray.500">
                                                {new Date(service.deadline_date).toLocaleDateString()}
                                            </Text>
                                            <Heading size="sm">{getServiceName(service.service_catalogo_id)}</Heading>
                                        </Box>
                                        <Badge colorScheme="gray">{service.status}</Badge>
                                    </HStack>
                                </CardBody>
                            </Card>
                        ))
                    ) : (
                        <Text color="gray.500">No hay servicios futuros asignados.</Text>
                    )}
                </Box>
            </VStack>
        </Box>
    );
};

export default CalendarioTrabajo;
