import { useState, useMemo } from 'react';
import {
  Box,
  Heading,
  VStack,
  HStack,
  Text,
  Avatar,
  Badge,
  Button,
  Card,
  CardBody,
  Image,
  Flex,
  SimpleGrid,
  Divider,
  Icon,
  Modal,
  ModalOverlay,
  ModalContent,
  ModalCloseButton,
  useDisclosure,
  IconButton,
  AspectRatio
} from '@chakra-ui/react';
import { BiLeftArrow, BiRightArrow, BiVideo, BiMusic, BiFile, BiImage } from 'react-icons/bi';
import Layout from '../../components/Layout';
import { mockBitacoras, mockProfesores, mockAlumnos, mockMaterias } from '../../data/mockData';

const padreLinks = [
  { nombre: 'Dashboard', path: '/padre' },
  { nombre: 'Avance (Bitácoras)', path: '/padre/bitacoras' },
  { nombre: 'Metas', path: '/padre/metas' },
  { nombre: 'Pagos', path: '/padre/pagos' },
  { nombre: 'Calendario', path: '/padre/calendario' },
  { nombre: 'Notificaciones', path: '/padre/notificaciones' }
];

const ITEMS_PER_PAGE = 5;

const padreId = 1;
const misHijos = mockAlumnos.filter(a => a.id_padre === padreId);
const misBitacoras = mockBitacoras
  .filter(b => misHijos.some(h => h.id === b.id_alumno))
  .sort((a, b) => new Date(b.fecha) - new Date(a.fecha)); // Orden descendente

const ConsultaBitacorasHijo = () => {
  const [currentPage, setCurrentPage] = useState(1);
  const [galeriaAbierta, setGaleriaAbierta] = useState(null);
  const [indiceEvidencia, setIndiceEvidencia] = useState(0);
  const { isOpen, onOpen, onClose } = useDisclosure();

  // Paginación
  const totalPages = Math.ceil(misBitacoras.length / ITEMS_PER_PAGE);
  const bitacorasPaginadas = useMemo(() => {
    const startIndex = (currentPage - 1) * ITEMS_PER_PAGE;
    return misBitacoras.slice(startIndex, startIndex + ITEMS_PER_PAGE);
  }, [currentPage]);

  // Obtener evidencias (soporta formato nuevo y antiguo)
  const obtenerEvidencias = (bitacora) => {
    if (bitacora.evidencias && bitacora.evidencias.length > 0) {
      return bitacora.evidencias;
    }
    if (bitacora.evidencia_url) {
      return [{ id: 1, tipo: 'image', url: bitacora.evidencia_url, thumbnail: bitacora.evidencia_url }];
    }
    return [];
  };

  // Abrir galería
  const abrirGaleria = (bitacora, indice = 0) => {
    const evidencias = obtenerEvidencias(bitacora);
    if (evidencias.length > 0) {
      setGaleriaAbierta(evidencias);
      setIndiceEvidencia(indice);
      onOpen();
    }
  };

  // Navegar en galería
  const siguienteEvidencia = () => {
    if (galeriaAbierta && indiceEvidencia < galeriaAbierta.length - 1) {
      setIndiceEvidencia(indiceEvidencia + 1);
    }
  };

  const anteriorEvidencia = () => {
    if (galeriaAbierta && indiceEvidencia > 0) {
      setIndiceEvidencia(indiceEvidencia - 1);
    }
  };

  const evidenciaActual = galeriaAbierta?.[indiceEvidencia];

  return (
    <Layout links={padreLinks} userType="Padre/Tutor" userName="Roberto López">
      <Box>
        {/* Instrucciones */}
        <Text fontSize="sm" color="gray.600" mb={6}>
          Consulta el progreso de tus hijos. <strong>Las evidencias se muestran directamente en cada card.</strong>
        </Text>

        {/* Resultados */}
        <Text fontSize="sm" color="gray.600" mb={4}>
          {misBitacoras.length > 0 ? (
            <>
              Mostrando <strong>{((currentPage - 1) * ITEMS_PER_PAGE) + 1}</strong> - <strong>{Math.min(currentPage * ITEMS_PER_PAGE, misBitacoras.length)}</strong> de <strong>{misBitacoras.length}</strong> bitácora{misBitacoras.length !== 1 ? 's' : ''}
            </>
          ) : (
            <>No hay bitácoras para mostrar</>
          )}
        </Text>

        {/* Cards de Bitácoras */}
        <VStack align="stretch" spacing={5}>
          {bitacorasPaginadas.map((bitacora) => {
            const evidencias = obtenerEvidencias(bitacora);
            const profesor = mockProfesores.find(p => p.id === bitacora.id_profesor);
            const alumno = mockAlumnos.find(a => a.id === bitacora.id_alumno);
            const materia = mockMaterias.find(m => m.id === bitacora.id_materia);
            
            return (
              <Card 
                key={bitacora.id} 
                border="1px solid" 
                borderColor="gray.200" 
                boxShadow="0 1px 3px 0 rgba(0, 0, 0, 0.1), 0 1px 2px 0 rgba(0, 0, 0, 0.06)"
                transition="all 0.2s ease-in-out"
                _hover={{
                  boxShadow: '0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06)',
                  transform: 'translateY(-2px)',
                }}
              >
                <CardBody p={6}>
                  {/* Header con Avatar y Fecha */}
                  <Flex justify="space-between" align="start" mb={5}>
                    <HStack spacing={5}>
                      <Avatar 
                        size="2xl" 
                        src={alumno?.foto_url} 
                        name={alumno?.nombre}
                        boxSize="100px"
                        border="4px solid"
                        borderColor="brand.200"
                        boxShadow="0 4px 12px rgba(14, 165, 233, 0.25)"
                      />
                      <VStack align="start" spacing={2}>
                        <Text fontWeight="700" fontSize="20px" color="gray.900">
                          {alumno?.nombre}
                        </Text>
                        <HStack spacing={3}>
                          <Badge 
                            colorScheme="purple" 
                            bg="purple.500" 
                            color="white"
                            px={2}
                            py={1}
                            borderRadius="6px"
                            fontSize="11px"
                            fontWeight="600"
                          >
                            {materia?.nombre || 'Sin materia'}
                          </Badge>
                          <HStack spacing={2}>
                            <Text fontSize="13px" color="gray.500" fontWeight="500">
                              Profesor:
                            </Text>
                            <Text fontSize="13px" color="gray.700" fontWeight="600">
                              {profesor?.nombre}
                            </Text>
                          </HStack>
                        </HStack>
                      </VStack>
                    </HStack>
                    <Badge 
                      colorScheme="brand" 
                      bg="brand.500" 
                      color="white"
                      px={3}
                      py={1}
                      borderRadius="6px"
                      fontSize="12px"
                      fontWeight="600"
                    >
                      {bitacora.fecha}
                    </Badge>
                  </Flex>

                  {/* Contenido y Evidencia en Grid */}
                  <SimpleGrid columns={{ base: 1, lg: 2 }} spacing={6}>
                    {/* Información de la Bitácora con Scroll */}
                    <Box
                      maxH="500px"
                      overflowY="auto"
                      pr={2}
                      css={{
                        '&::-webkit-scrollbar': {
                          width: '8px',
                        },
                        '&::-webkit-scrollbar-track': {
                          background: '#f1f1f1',
                          borderRadius: '10px',
                        },
                        '&::-webkit-scrollbar-thumb': {
                          background: '#888',
                          borderRadius: '10px',
                        },
                        '&::-webkit-scrollbar-thumb:hover': {
                          background: '#555',
                        },
                      }}
                    >
                      <VStack align="stretch" spacing={4}>
                        <Box>
                          <Text fontWeight="700" mb={2} fontSize="14px" color="gray.700" textTransform="uppercase" letterSpacing="0.5px">
                            Objetivo
                          </Text>
                          <Text color="gray.700" fontSize="14px" lineHeight="1.6">{bitacora.objetivo}</Text>
                        </Box>
                        <Divider borderColor="gray.200" />
                        <Box>
                          <Text fontWeight="700" mb={2} fontSize="14px" color="gray.700" textTransform="uppercase" letterSpacing="0.5px">
                            Actividad
                          </Text>
                          <Text color="gray.700" fontSize="14px" lineHeight="1.6">{bitacora.actividad}</Text>
                        </Box>
                        <Divider borderColor="gray.200" />
                        <Box>
                          <Text fontWeight="700" mb={2} fontSize="14px" color="gray.700" textTransform="uppercase" letterSpacing="0.5px">
                            Evaluación
                          </Text>
                          <Text color="gray.700" fontSize="14px" lineHeight="1.6">{bitacora.evaluacion}</Text>
                        </Box>
                        <Divider borderColor="gray.200" />
                        <Box>
                          <Text fontWeight="700" mb={2} fontSize="14px" color="gray.700" textTransform="uppercase" letterSpacing="0.5px">
                            Tarea
                          </Text>
                          <Text color="gray.700" fontSize="14px" lineHeight="1.6">{bitacora.tarea}</Text>
                        </Box>
                      </VStack>
                      
                      {/* Documentos de Tarea */}
                      {bitacora.documentos_tarea && bitacora.documentos_tarea.length > 0 && (
                        <>
                          <Divider borderColor="gray.200" mt={4} />
                          <Box>
                            <Text fontWeight="700" mb={2} fontSize="12px" color="gray.600" textTransform="uppercase" letterSpacing="0.5px">
                              Documentos de Tarea
                            </Text>
                            <SimpleGrid columns={2} spacing={2}>
                              {bitacora.documentos_tarea.map((doc) => (
                                <Box
                                  key={doc.id}
                                  as="a"
                                  href={doc.url}
                                  target="_blank"
                                  rel="noopener noreferrer"
                                  p={2}
                                  borderRadius="6px"
                                  border="1px solid"
                                  borderColor="gray.200"
                                  bg="gray.50"
                                  _hover={{
                                    bg: 'brand.50',
                                    borderColor: 'brand.300',
                                    transform: 'translateY(-1px)',
                                  }}
                                  transition="all 0.2s ease-in-out"
                                  display="flex"
                                  alignItems="center"
                                  gap={2}
                                  cursor="pointer"
                                >
                                  <Icon 
                                    as={doc.tipo === 'pdf' ? BiFile : BiImage} 
                                    fontSize="16px" 
                                    color={doc.tipo === 'pdf' ? 'red.500' : 'brand.500'} 
                                  />
                                  <Text 
                                    fontSize="11px" 
                                    color="gray.700" 
                                    fontWeight="500"
                                    noOfLines={1}
                                    flex={1}
                                  >
                                    {doc.nombre}
                                  </Text>
                                </Box>
                              ))}
                            </SimpleGrid>
                          </Box>
                        </>
                      )}
                    </Box>

                    {/* Galería de Evidencias */}
                    <Box>
                      <Flex justify="space-between" align="center" mb={3}>
                        <Text fontWeight="700" fontSize="14px" color="gray.700" textTransform="uppercase" letterSpacing="0.5px">
                          Evidencias ({evidencias.length})
                        </Text>
                        {evidencias.length > 1 && (
                          <Button
                            size="xs"
                            variant="ghost"
                            colorScheme="brand"
                            onClick={() => abrirGaleria(bitacora, 0)}
                            fontSize="11px"
                          >
                            Ver todas
                          </Button>
                        )}
                      </Flex>
                      
                      {evidencias.length > 0 ? (
                        <VStack spacing={3} align="stretch">
                          {/* Imagen principal o primera evidencia */}
                          <Box
                            borderRadius="12px"
                            overflow="hidden"
                            border="1px solid"
                            borderColor="gray.200"
                            bg="gray.50"
                            transition="all 0.2s ease-in-out"
                            _hover={{
                              borderColor: 'brand.300',
                              boxShadow: '0 4px 12px rgba(14, 165, 233, 0.15)',
                            }}
                            cursor="pointer"
                            onClick={() => abrirGaleria(bitacora, 0)}
                            position="relative"
                          >
                            {evidencias[0].tipo === 'image' && (
                              <Image
                                src={evidencias[0].url}
                                alt="Evidencia de bitácora"
                                w="100%"
                                h="auto"
                                objectFit="cover"
                                minH="300px"
                                maxH="400px"
                                transition="all 0.3s ease-in-out"
                                _hover={{
                                  transform: 'scale(1.02)',
                                }}
                              />
                            )}
                            {evidencias[0].tipo === 'video' && (
                              <AspectRatio ratio={16 / 9}>
                                <Box
                                  bg="gray.800"
                                  display="flex"
                                  alignItems="center"
                                  justifyContent="center"
                                  position="relative"
                                >
                                  <Icon as={BiVideo} fontSize="48px" color="white" />
                                  {evidencias[0].thumbnail && (
                                    <Image
                                      src={evidencias[0].thumbnail}
                                      alt="Video thumbnail"
                                      position="absolute"
                                      top={0}
                                      left={0}
                                      w="100%"
                                      h="100%"
                                      objectFit="cover"
                                      opacity={0.5}
                                    />
                                  )}
                                </Box>
                              </AspectRatio>
                            )}
                            {evidencias[0].tipo === 'audio' && (
                              <Box
                                bg="gray.100"
                                p={8}
                                display="flex"
                                alignItems="center"
                                justifyContent="center"
                                minH="200px"
                              >
                                <VStack spacing={3}>
                                  <Icon as={BiMusic} fontSize="48px" color="brand.500" />
                                  <Text fontSize="14px" color="gray.600" fontWeight="500">
                                    Audio disponible
                                  </Text>
                                </VStack>
                              </Box>
                            )}
                            {evidencias.length > 1 && (
                              <Box
                                position="absolute"
                                top={2}
                                right={2}
                                bg="blackAlpha.700"
                                color="white"
                                px={2}
                                py={1}
                                borderRadius="6px"
                                fontSize="11px"
                                fontWeight="600"
                              >
                                +{evidencias.length - 1} más
                              </Box>
                            )}
                          </Box>

                          {/* Thumbnails de otras evidencias */}
                          {evidencias.length > 1 && (
                            <SimpleGrid columns={4} spacing={2}>
                              {evidencias.slice(1, 5).map((evidencia, idx) => (
                                <Box
                                  key={evidencia.id}
                                  borderRadius="8px"
                                  overflow="hidden"
                                  border="2px solid"
                                  borderColor="gray.200"
                                  cursor="pointer"
                                  transition="all 0.2s ease-in-out"
                                  _hover={{
                                    borderColor: 'brand.400',
                                    transform: 'scale(1.05)',
                                  }}
                                  onClick={() => abrirGaleria(bitacora, idx + 1)}
                                  position="relative"
                                  aspectRatio="1"
                                >
                                  {evidencia.tipo === 'image' && evidencia.thumbnail && (
                                    <Image
                                      src={evidencia.thumbnail}
                                      alt={`Evidencia ${idx + 2}`}
                                      w="100%"
                                      h="100%"
                                      objectFit="cover"
                                    />
                                  )}
                                  {evidencia.tipo === 'video' && (
                                    <Box
                                      bg="gray.800"
                                      w="100%"
                                      h="100%"
                                      display="flex"
                                      alignItems="center"
                                      justifyContent="center"
                                    >
                                      <Icon as={BiVideo} fontSize="20px" color="white" />
                                    </Box>
                                  )}
                                  {evidencia.tipo === 'audio' && (
                                    <Box
                                      bg="gray.100"
                                      w="100%"
                                      h="100%"
                                      display="flex"
                                      alignItems="center"
                                      justifyContent="center"
                                    >
                                      <Icon as={BiMusic} fontSize="20px" color="brand.500" />
                                    </Box>
                                  )}
                                </Box>
                              ))}
                            </SimpleGrid>
                          )}
                        </VStack>
                      ) : (
                        <Box
                          borderRadius="12px"
                          border="1px dashed"
                          borderColor="gray.300"
                          bg="gray.50"
                          p={8}
                          textAlign="center"
                        >
                          <Text fontSize="13px" color="gray.500">
                            No hay evidencias disponibles
                          </Text>
                        </Box>
                      )}
                    </Box>
                  </SimpleGrid>
                </CardBody>
              </Card>
            );
          })}
        </VStack>

        {misBitacoras.length === 0 && (
          <Card border="1px solid" borderColor="gray.200" mt={6}>
            <CardBody p={10} textAlign="center">
              <Text fontSize="16px" color="gray.500" fontWeight="500">
                No hay bitácoras registradas para tus hijos
              </Text>
            </CardBody>
          </Card>
        )}

        {/* Paginación */}
        {misBitacoras.length > 0 && totalPages > 1 && (
          <Flex justify="center" align="center" mt={6} gap={2}>
            <IconButton
              icon={<Icon as={BiLeftArrow} />}
              onClick={() => setCurrentPage(Math.max(1, currentPage - 1))}
              isDisabled={currentPage === 1}
              aria-label="Página anterior"
              borderRadius="8px"
              size="sm"
            />
            <HStack spacing={2}>
              {Array.from({ length: totalPages }, (_, i) => i + 1).map((page) => (
                <Button
                  key={page}
                  size="sm"
                  onClick={() => setCurrentPage(page)}
                  colorScheme={currentPage === page ? 'brand' : 'gray'}
                  variant={currentPage === page ? 'solid' : 'outline'}
                  borderRadius="8px"
                  minW="40px"
                >
                  {page}
                </Button>
              ))}
            </HStack>
            <IconButton
              icon={<Icon as={BiRightArrow} />}
              onClick={() => setCurrentPage(Math.min(totalPages, currentPage + 1))}
              isDisabled={currentPage === totalPages}
              aria-label="Página siguiente"
              borderRadius="8px"
              size="sm"
            />
          </Flex>
        )}

        {/* Modal de Galería Fancybox */}
        <Modal 
          isOpen={isOpen} 
          onClose={onClose} 
          size="6xl" 
          motionPreset="scale"
          closeOnOverlayClick={true}
        >
          <ModalOverlay bg="blackAlpha.800" backdropFilter="blur(4px)" />
          <ModalContent 
            borderRadius="16px" 
            boxShadow="2xl"
            bg="black"
            maxW="90vw"
            maxH="90vh"
            position="relative"
          >
            <ModalCloseButton 
              color="white" 
              bg="blackAlpha.500"
              _hover={{ bg: 'blackAlpha.700' }}
              zIndex={10}
            />
            {evidenciaActual && (
              <Box position="relative" w="100%" h="90vh" display="flex" alignItems="center" justifyContent="center">
                {/* Navegación anterior */}
                {galeriaAbierta.length > 1 && indiceEvidencia > 0 && (
                  <IconButton
                    icon={<Icon as={BiLeftArrow} />}
                    onClick={anteriorEvidencia}
                    aria-label="Anterior"
                    position="absolute"
                    left={4}
                    top="50%"
                    transform="translateY(-50%)"
                    bg="blackAlpha.600"
                    color="white"
                    _hover={{ bg: 'blackAlpha.800' }}
                    size="lg"
                    borderRadius="full"
                    zIndex={10}
                  />
                )}

                {/* Contenido de la evidencia */}
                <Box w="100%" h="100%" display="flex" alignItems="center" justifyContent="center" p={4}>
                  {evidenciaActual.tipo === 'image' && (
                    <Image
                      src={evidenciaActual.url}
                      alt="Evidencia"
                      maxH="85vh"
                      maxW="100%"
                      objectFit="contain"
                      borderRadius="8px"
                    />
                  )}
                  {evidenciaActual.tipo === 'video' && (
                    <AspectRatio ratio={16 / 9} w="100%" maxW="1200px">
                      <Box as="video" src={evidenciaActual.url} controls autoPlay />
                    </AspectRatio>
                  )}
                  {evidenciaActual.tipo === 'audio' && (
                    <Box
                      bg="gray.900"
                      borderRadius="12px"
                      p={8}
                      w="100%"
                      maxW="600px"
                      textAlign="center"
                    >
                      <Icon as={BiMusic} fontSize="64px" color="brand.400" mb={4} />
                      <Box as="audio" src={evidenciaActual.url} controls w="100%" />
                    </Box>
                  )}
                </Box>

                {/* Navegación siguiente */}
                {galeriaAbierta.length > 1 && indiceEvidencia < galeriaAbierta.length - 1 && (
                  <IconButton
                    icon={<Icon as={BiRightArrow} />}
                    onClick={siguienteEvidencia}
                    aria-label="Siguiente"
                    position="absolute"
                    right={4}
                    top="50%"
                    transform="translateY(-50%)"
                    bg="blackAlpha.600"
                    color="white"
                    _hover={{ bg: 'blackAlpha.800' }}
                    size="lg"
                    borderRadius="full"
                    zIndex={10}
                  />
                )}

                {/* Indicador de posición */}
                {galeriaAbierta.length > 1 && (
                  <Box
                    position="absolute"
                    bottom={4}
                    left="50%"
                    transform="translateX(-50%)"
                    bg="blackAlpha.700"
                    px={4}
                    py={2}
                    borderRadius="full"
                    color="white"
                    fontSize="14px"
                    fontWeight="600"
                  >
                    {indiceEvidencia + 1} / {galeriaAbierta.length}
                  </Box>
                )}

                {/* Thumbnails en la parte inferior */}
                {galeriaAbierta.length > 1 && (
                  <Box
                    position="absolute"
                    bottom={16}
                    left="50%"
                    transform="translateX(-50%)"
                    display="flex"
                    gap={2}
                    maxW="90%"
                    overflowX="auto"
                    pb={2}
                  >
                    {galeriaAbierta.map((evidencia, idx) => (
                      <Box
                        key={evidencia.id}
                        w="60px"
                        h="60px"
                        borderRadius="8px"
                        overflow="hidden"
                        border="2px solid"
                        borderColor={idx === indiceEvidencia ? 'brand.400' : 'transparent'}
                        cursor="pointer"
                        onClick={() => setIndiceEvidencia(idx)}
                        flexShrink={0}
                        transition="all 0.2s ease-in-out"
                        _hover={{
                          borderColor: 'brand.300',
                          transform: 'scale(1.1)',
                        }}
                      >
                        {evidencia.tipo === 'image' && evidencia.thumbnail && (
                          <Image src={evidencia.thumbnail} alt={`Thumbnail ${idx + 1}`} w="100%" h="100%" objectFit="cover" />
                        )}
                        {evidencia.tipo === 'video' && (
                          <Box bg="gray.800" w="100%" h="100%" display="flex" alignItems="center" justifyContent="center">
                            <Icon as={BiVideo} fontSize="20px" color="white" />
                          </Box>
                        )}
                        {evidencia.tipo === 'audio' && (
                          <Box bg="gray.700" w="100%" h="100%" display="flex" alignItems="center" justifyContent="center">
                            <Icon as={BiMusic} fontSize="20px" color="brand.400" />
                          </Box>
                        )}
                      </Box>
                    ))}
                  </Box>
                )}
              </Box>
            )}
          </ModalContent>
        </Modal>
      </Box>
    </Layout>
  );
};

export default ConsultaBitacorasHijo;

