import { useState } from 'react';
import {
  Box,
  Heading,
  SimpleGrid,
  Card,
  CardBody,
  Avatar,
  Text,
  VStack,
  HStack,
  Button,
  Modal,
  ModalOverlay,
  ModalContent,
  ModalHeader,
  ModalBody,
  ModalFooter,
  ModalCloseButton,
  useDisclosure,
  FormControl,
  FormLabel,
  Textarea,
  useToast,
  Icon,
  Badge,
  Flex,
  Tooltip,
  Divider
} from '@chakra-ui/react';
import { FaUserGraduate, FaBullseye, FaEnvelope, FaPhone, FaEdit, FaPalette, FaClock } from 'react-icons/fa';
import Layout from '../../components/Layout';
import { mockAlumnos, mockClases, mockMetas, mockMaterias } from '../../data/mockData';

const profesorLinks = [
  { nombre: 'Mis Alumnos', path: '/profesor/alumnos' },
  { nombre: 'Registrar Bitácora', path: '/profesor/bitacora' },
  { nombre: 'Mi Calendario', path: '/profesor/calendario' },
  { nombre: 'Notificaciones', path: '/profesor/notificaciones' }
];

const profesorId = 1;
const misClases = mockClases.filter(c => c.id_profesor === profesorId);
const misAlumnosIds = [...new Set(misClases.map(c => c.id_alumno))];
const misAlumnos = mockAlumnos.filter(a => misAlumnosIds.includes(a.id));

const MisAlumnosMetas = () => {
  const [alumnoSeleccionado, setAlumnoSeleccionado] = useState(null);
  const [metas, setMetas] = useState(mockMetas);
  const { isOpen, onOpen, onClose } = useDisclosure();
  const toast = useToast();

  const handleAbrirMetas = (alumno) => {
    setAlumnoSeleccionado(alumno);
    onOpen();
  };

  const handleGuardarMetas = (e) => {
    e.preventDefault();
    const formData = new FormData(e.target);
    const nuevasMetas = formData.get('metas');

    if (alumnoSeleccionado) {
      setMetas({ ...metas, [alumnoSeleccionado.id]: nuevasMetas });
    }

    toast({
      title: 'Metas guardadas',
      description: 'Las metas del alumno han sido actualizadas',
      status: 'success',
      duration: 2000,
      isClosable: true,
    });
    onClose();
  };

  const getMateriaAlumno = (alumnoId) => {
    const clase = misClases.find(c => c.id_alumno === alumnoId);
    if (clase) {
      const materia = mockMaterias.find(m => m.id === clase.id_materia);
      return materia ? materia.nombre : 'Sin materia';
    }
    return 'Sin materia';
  };

  const getHorarioAlumno = (alumnoId) => {
    const clase = misClases.find(c => c.id_alumno === alumnoId);
    return clase ? `${clase.dia_semana} ${clase.hora_inicio}` : '';
  };

  return (
    <Layout links={profesorLinks} userType="Profesor" userName="María González">
      <Box>
        <VStack align="start" spacing={2} mb={8}>
          <Heading size="lg" bgGradient="linear(to-r, brand.500, brand.300)" bgClip="text">
            Mis Alumnos y Metas
          </Heading>
          <Text fontSize="md" color="gray.600">
            Gestiona el progreso y objetivos de tus estudiantes para impulsar su desarrollo artístico.
          </Text>
        </VStack>

        <SimpleGrid columns={{ base: 1, md: 2, lg: 3 }} spacing={8}>
          {misAlumnos.map((alumno) => {
            const materia = getMateriaAlumno(alumno.id);
            const horario = getHorarioAlumno(alumno.id);
            const metaActual = metas[alumno.id] || 'Sin metas definidas';

            return (
              <Card
                key={alumno.id}
                borderRadius="2xl"
                overflow="hidden"
                boxShadow="lg"
                transition="all 0.3s"
                _hover={{ transform: 'translateY(-5px)', boxShadow: '2xl' }}
                bg="white"
                border="1px solid"
                borderColor="gray.100"
              >
                <Box h="100px" bgGradient="linear(to-r, brand.400, brand.600)" position="relative">
                  <Box
                    position="absolute"
                    bottom="-40px"
                    left="50%"
                    transform="translateX(-50%)"
                    p={1}
                    bg="white"
                    borderRadius="full"
                  >
                    <Avatar size="xl" src={alumno.foto_url} name={alumno.nombre} border="2px solid white" />
                  </Box>
                </Box>

                <CardBody pt={12} pb={6}>
                  <VStack spacing={3}>
                    <Box textAlign="center">
                      <Heading size="md" color="gray.800">{alumno.nombre}</Heading>
                      <HStack justify="center" spacing={2} mt={1}>
                        <Badge colorScheme="purple" borderRadius="full" px={2}>
                          {alumno.edad} años
                        </Badge>
                        <Badge colorScheme="blue" borderRadius="full" px={2}>
                          {materia}
                        </Badge>
                      </HStack>
                    </Box>

                    <HStack fontSize="sm" color="gray.500" spacing={4}>
                      <HStack>
                        <Icon as={FaClock} />
                        <Text>{horario}</Text>
                      </HStack>
                    </HStack>

                    <Divider my={2} />

                    <Box w="100%" bg="gray.50" p={3} borderRadius="lg" borderLeft="4px solid" borderColor="brand.500">
                      <HStack mb={1}>
                        <Icon as={FaBullseye} color="brand.500" />
                        <Text fontWeight="bold" fontSize="sm" color="gray.700">Meta Actual</Text>
                      </HStack>
                      <Text fontSize="sm" color="gray.600" noOfLines={2}>
                        {metaActual}
                      </Text>
                    </Box>

                    <HStack w="100%" justify="center" spacing={4} pt={2}>
                      <Tooltip label={alumno.email}>
                        <Button size="sm" variant="ghost" colorScheme="gray" borderRadius="full">
                          <Icon as={FaEnvelope} />
                        </Button>
                      </Tooltip>
                      <Tooltip label={alumno.telefono}>
                        <Button size="sm" variant="ghost" colorScheme="gray" borderRadius="full">
                          <Icon as={FaPhone} />
                        </Button>
                      </Tooltip>
                      <Button
                        size="sm"
                        colorScheme="brand"
                        leftIcon={<Icon as={FaEdit} />}
                        onClick={() => handleAbrirMetas(alumno)}
                        borderRadius="full"
                        px={6}
                        boxShadow="md"
                      >
                        Editar Metas
                      </Button>
                    </HStack>
                  </VStack>
                </CardBody>
              </Card>
            );
          })}
        </SimpleGrid>

        <Modal isOpen={isOpen} onClose={onClose} size="xl" isCentered>
          <ModalOverlay backdropFilter="blur(5px)" />
          <ModalContent borderRadius="2xl">
            <ModalHeader bg="gray.50" borderTopRadius="2xl">
              <HStack>
                <Icon as={FaUserGraduate} color="brand.500" />
                <Text>Metas de {alumnoSeleccionado?.nombre}</Text>
              </HStack>
            </ModalHeader>
            <ModalCloseButton />
            <form onSubmit={handleGuardarMetas}>
              <ModalBody py={6}>
                <FormControl>
                  <FormLabel fontWeight="bold" color="gray.700">
                    Definir Objetivos y Metas
                  </FormLabel>
                  <Text fontSize="sm" color="gray.500" mb={3}>
                    Establece metas claras y alcanzables para guiar el desarrollo artístico del alumno.
                  </Text>
                  <Textarea
                    name="metas"
                    defaultValue={metas[alumnoSeleccionado?.id] || ''}
                    rows={8}
                    placeholder="Ej: Completar 3 obras de paisaje, mejorar técnica de sombreado..."
                    borderRadius="xl"
                    focusBorderColor="brand.500"
                    _hover={{ borderColor: 'brand.300' }}
                  />
                </FormControl>
              </ModalBody>
              <ModalFooter bg="gray.50" borderBottomRadius="2xl">
                <Button variant="ghost" mr={3} onClick={onClose} borderRadius="full">Cancelar</Button>
                <Button colorScheme="brand" type="submit" borderRadius="full" px={6} boxShadow="md">
                  Guardar Cambios
                </Button>
              </ModalFooter>
            </form>
          </ModalContent>
        </Modal>
      </Box>
    </Layout>
  );
};

export default MisAlumnosMetas;

