import { useState, useMemo } from 'react';
import {
  Box,
  VStack,
  HStack,
  Text,
  Heading,
  Tabs,
  TabList,
  TabPanels,
  Tab,
  TabPanel,
  Badge,
  Button,
  Input,
  InputGroup,
  InputLeftElement,
  Select,
  Grid,
  Icon,
  Flex,
  useToast,
  SlideFade,
  IconButton,
  Tooltip
} from '@chakra-ui/react';
import {
  FiSearch,
  FiFilter,
  FiCheckCircle,
  FiMail,
  FiArchive,
  FiUsers,
  FiCalendar,
  FiDollarSign,
  FiBook,
  FiGift,
  FiX,
  FiInbox
} from 'react-icons/fi';
import Layout from '../../components/Layout';
import NotificacionCard from '../../components/NotificacionCard';

const profesorLinks = [
  { nombre: 'Mis Alumnos', path: '/profesor/alumnos' },
  { nombre: 'Registrar Bitácora', path: '/profesor/bitacora' },
  { nombre: 'Mi Calendario', path: '/profesor/calendario' },
  { nombre: 'Notificaciones', path: '/profesor/notificaciones' }
];

const initialNotificaciones = [
  {
    id: 1,
    titulo: 'Nuevo alumno asignado',
    mensaje: 'Se te ha asignado un nuevo alumno: Sofía López. Puedes comenzar a registrar sus bitácoras y establecer sus metas.',
    fecha: '2024-01-20',
    leida: false,
    tipo: 'alumno',
    archivada: false
  },
  {
    id: 2,
    titulo: 'Recordatorio de clase',
    mensaje: 'Recuerda tu clase con Diego Ramírez el próximo lunes a las 14:00',
    fecha: '2024-01-19',
    leida: false,
    tipo: 'clase',
    archivada: false
  },
  {
    id: 3,
    titulo: 'Pago recibido',
    mensaje: 'El padre de Sofía López ha registrado un nuevo pago',
    fecha: '2024-01-18',
    leida: true,
    tipo: 'pago',
    archivada: false
  },
  {
    id: 4,
    titulo: '🎨 Nuevo Curso Disponible: Técnicas Avanzadas de Acuarela',
    html: `
      <p>¡Tenemos excelentes noticias! Se ha abierto un nuevo curso especializado que podría interesarte:</p>
      <h3>🎨 Técnicas Avanzadas de Acuarela</h3>
      <p><strong>Duración:</strong> 8 semanas</p>
      <p><strong>Horario:</strong> Martes y Jueves de 18:00 a 20:00</p>
      <p><strong>Incluye:</strong></p>
      <ul>
        <li>Materiales profesionales incluidos</li>
        <li>Certificado al finalizar</li>
        <li>Grupo reducido (máximo 8 alumnos)</li>
      </ul>
      <p style="margin-top: 16px;"><strong>¡Inscríbete antes del 25 de enero y obtén un 15% de descuento!</strong></p>
    `,
    fecha: '2024-01-17',
    leida: false,
    tipo: 'curso',
    imagen: 'https://images.unsplash.com/photo-1541961017774-22349e4a1262?w=800&h=400&fit=crop',
    archivada: false
  },
  {
    id: 5,
    titulo: '🎉 Promoción Especial: Descuento en Materiales',
    html: `
      <p>Como agradecimiento a nuestros profesores, tenemos una <strong>promoción especial</strong> este mes:</p>
      <div style="background: linear-gradient(135deg, #FEF3C7 0%, #FDE68A 100%); padding: 16px; border-radius: 8px; margin: 16px 0;">
        <h3 style="color: #92400E; margin: 0 0 8px 0;">20% de descuento</h3>
        <p style="color: #78350F; margin: 0;">En todos los materiales de arte en nuestra tienda asociada</p>
      </div>
      <p>Código de descuento: <strong style="color: #B45309;">PROF2024</strong></p>
      <p>Válido hasta el 31 de enero de 2024</p>
    `,
    fecha: '2024-01-16',
    leida: false,
    tipo: 'promocion',
    imagen: 'https://images.unsplash.com/photo-1513475382585-d06e58bcb0e0?w=800&h=400&fit=crop',
    archivada: false
  }
];

const NotificacionesProfesor = () => {
  const [notificaciones, setNotificaciones] = useState(initialNotificaciones);
  const [searchTerm, setSearchTerm] = useState('');
  const [filterTipo, setFilterTipo] = useState('todas');
  const toast = useToast();

  // Estadísticas
  const stats = useMemo(() => {
    const noLeidas = notificaciones.filter(n => !n.leida && !n.archivada).length;
    const archivadas = notificaciones.filter(n => n.archivada).length;
    const total = notificaciones.length;
    return { noLeidas, archivadas, total };
  }, [notificaciones]);

  // Filtrar notificaciones
  const filteredNotificaciones = useMemo(() => {
    return notificaciones.filter(notif => {
      const matchesSearch = notif.titulo.toLowerCase().includes(searchTerm.toLowerCase()) ||
        (notif.mensaje && notif.mensaje.toLowerCase().includes(searchTerm.toLowerCase()));
      const matchesTipo = filterTipo === 'todas' || notif.tipo === filterTipo;
      return matchesSearch && matchesTipo;
    });
  }, [notificaciones, searchTerm, filterTipo]);

  // Separar por tabs
  const noLeidas = filteredNotificaciones.filter(n => !n.leida && !n.archivada);
  const archivadas = filteredNotificaciones.filter(n => n.archivada);
  const todas = filteredNotificaciones.filter(n => !n.archivada);

  // Marcar como leída
  const marcarComoLeida = (id) => {
    setNotificaciones(prev =>
      prev.map(n => n.id === id ? { ...n, leida: true } : n)
    );
  };

  // Marcar todas como leídas
  const marcarTodasLeidas = () => {
    setNotificaciones(prev =>
      prev.map(n => ({ ...n, leida: true }))
    );
    toast({
      title: 'Notificaciones marcadas',
      description: 'Todas las notificaciones han sido marcadas como leídas',
      status: 'success',
      duration: 3000,
      isClosable: true,
    });
  };

  // Archivar notificación
  const archivarNotificacion = (id) => {
    setNotificaciones(prev =>
      prev.map(n => n.id === id ? { ...n, archivada: true, leida: true } : n)
    );
    toast({
      title: 'Notificación archivada',
      status: 'info',
      duration: 2000,
      isClosable: true,
    });
  };

  const limpiarFiltros = () => {
    setSearchTerm('');
    setFilterTipo('todas');
  };

  const EmptyState = ({ icon, title, description }) => (
    <Box
      textAlign="center"
      py={16}
      px={6}
      bg="white"
      borderRadius="2xl"
      boxShadow="sm"
      border="1px dashed"
      borderColor="gray.200"
      display="flex"
      flexDirection="column"
      alignItems="center"
      justifyContent="center"
      minH="300px"
    >
      <Box
        p={4}
        bg="gray.50"
        borderRadius="full"
        mb={4}
        color="gray.400"
      >
        <Icon as={icon} boxSize={10} />
      </Box>
      <Text fontSize="xl" fontWeight="bold" color="gray.700" mb={2}>
        {title}
      </Text>
      <Text color="gray.500" maxW="sm">
        {description}
      </Text>
      {(searchTerm || filterTipo !== 'todas') && (
        <Button
          mt={6}
          variant="outline"
          colorScheme="purple"
          size="sm"
          onClick={limpiarFiltros}
          leftIcon={<FiX />}
        >
          Limpiar filtros
        </Button>
      )}
    </Box>
  );

  return (
    <Layout links={profesorLinks} userType="Profesor" userName="María González">
      <Box maxW="1200px" mx="auto">
        {/* Header con estadísticas */}
        <Box mb={8}>
          <Heading size="lg" mb={2} bgGradient="linear(to-r, purple.600, pink.500)" bgClip="text">
            📬 Centro de Notificaciones
          </Heading>
          <Text color="gray.600" mb={6}>Gestiona tus avisos y comunicaciones escolares</Text>

          <Grid templateColumns={{ base: '1fr', md: 'repeat(3, 1fr)' }} gap={4} mb={6}>
            {/* Stat: No leídas */}
            <Box
              bg="linear-gradient(135deg, #667eea 0%, #764ba2 100%)"
              p={5}
              borderRadius="2xl"
              color="white"
              boxShadow="lg"
              position="relative"
              overflow="hidden"
              transition="transform 0.2s"
              _hover={{ transform: 'translateY(-2px)' }}
            >
              <Box position="absolute" top={0} right={0} p={4} opacity={0.2}>
                <Icon as={FiMail} boxSize={24} transform="rotate(-15deg)" />
              </Box>
              <HStack spacing={4} position="relative" zIndex={1}>
                <Box p={3} bg="whiteAlpha.200" borderRadius="xl">
                  <Icon as={FiMail} boxSize={6} />
                </Box>
                <Box>
                  <Text fontSize="3xl" fontWeight="800" lineHeight="1">{stats.noLeidas}</Text>
                  <Text fontSize="sm" fontWeight="medium" opacity={0.9}>No leídas</Text>
                </Box>
              </HStack>
            </Box>

            {/* Stat: Total */}
            <Box
              bg="linear-gradient(135deg, #f093fb 0%, #f5576c 100%)"
              p={5}
              borderRadius="2xl"
              color="white"
              boxShadow="lg"
              position="relative"
              overflow="hidden"
              transition="transform 0.2s"
              _hover={{ transform: 'translateY(-2px)' }}
            >
              <Box position="absolute" top={0} right={0} p={4} opacity={0.2}>
                <Icon as={FiInbox} boxSize={24} transform="rotate(-15deg)" />
              </Box>
              <HStack spacing={4} position="relative" zIndex={1}>
                <Box p={3} bg="whiteAlpha.200" borderRadius="xl">
                  <Icon as={FiInbox} boxSize={6} />
                </Box>
                <Box>
                  <Text fontSize="3xl" fontWeight="800" lineHeight="1">{stats.total}</Text>
                  <Text fontSize="sm" fontWeight="medium" opacity={0.9}>Total recibidas</Text>
                </Box>
              </HStack>
            </Box>

            {/* Stat: Archivadas */}
            <Box
              bg="linear-gradient(135deg, #4facfe 0%, #00f2fe 100%)"
              p={5}
              borderRadius="2xl"
              color="white"
              boxShadow="lg"
              position="relative"
              overflow="hidden"
              transition="transform 0.2s"
              _hover={{ transform: 'translateY(-2px)' }}
            >
              <Box position="absolute" top={0} right={0} p={4} opacity={0.2}>
                <Icon as={FiArchive} boxSize={24} transform="rotate(-15deg)" />
              </Box>
              <HStack spacing={4} position="relative" zIndex={1}>
                <Box p={3} bg="whiteAlpha.200" borderRadius="xl">
                  <Icon as={FiArchive} boxSize={6} />
                </Box>
                <Box>
                  <Text fontSize="3xl" fontWeight="800" lineHeight="1">{stats.archivadas}</Text>
                  <Text fontSize="sm" fontWeight="medium" opacity={0.9}>Archivadas</Text>
                </Box>
              </HStack>
            </Box>
          </Grid>

          {/* Filtros y búsqueda */}
          <Flex
            direction={{ base: 'column', md: 'row' }}
            gap={4}
            mb={6}
            bg="white"
            p={4}
            borderRadius="2xl"
            boxShadow="sm"
            border="1px solid"
            borderColor="gray.100"
            alignItems="center"
          >
            <InputGroup flex={1}>
              <InputLeftElement pointerEvents="none">
                <Icon as={FiSearch} color="gray.400" />
              </InputLeftElement>
              <Input
                placeholder="Buscar por título o contenido..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                bg="gray.50"
                border="none"
                _focus={{ bg: 'white', boxShadow: 'outline' }}
                borderRadius="lg"
              />
              {searchTerm && (
                <InputGroup size="sm" width="auto" position="absolute" right={2} top={2} zIndex={2}>
                  <IconButton
                    size="xs"
                    aria-label="Limpiar búsqueda"
                    icon={<FiX />}
                    onClick={() => setSearchTerm('')}
                    variant="ghost"
                    isRound
                  />
                </InputGroup>
              )}
            </InputGroup>

            <HStack spacing={3} w={{ base: 'full', md: 'auto' }}>
              <Select
                value={filterTipo}
                onChange={(e) => setFilterTipo(e.target.value)}
                w={{ base: 'full', md: '200px' }}
                bg="gray.50"
                border="none"
                borderRadius="lg"
                icon={<FiFilter />}
                _focus={{ boxShadow: 'outline' }}
              >
                <option value="todas">Todos los tipos</option>
                <option value="alumno">👥 Alumnos</option>
                <option value="clase">📅 Clases</option>
                <option value="pago">💰 Pagos</option>
                <option value="curso">📚 Cursos</option>
                <option value="promocion">🎁 Promociones</option>
              </Select>

              {stats.noLeidas > 0 && (
                <Tooltip label="Marcar todas como leídas">
                  <IconButton
                    icon={<FiCheckCircle />}
                    colorScheme="purple"
                    variant="ghost"
                    onClick={marcarTodasLeidas}
                    size="lg"
                    aria-label="Marcar todas como leídas"
                    isRound
                  />
                </Tooltip>
              )}
            </HStack>
          </Flex>
        </Box>

        {/* Tabs */}
        <Tabs colorScheme="purple" variant="soft-rounded" isLazy>
          <TabList mb={6} bg="white" p={2} borderRadius="2xl" boxShadow="sm" display="inline-flex">
            <Tab fontWeight="semibold" _selected={{ bg: 'purple.50', color: 'purple.600' }}>
              <HStack spacing={2}>
                <Icon as={FiMail} />
                <Text>No leídas</Text>
                {stats.noLeidas > 0 && (
                  <Badge colorScheme="purple" borderRadius="full" px={2}>{stats.noLeidas}</Badge>
                )}
              </HStack>
            </Tab>
            <Tab fontWeight="semibold" _selected={{ bg: 'gray.100', color: 'gray.800' }}>
              <HStack spacing={2}>
                <Icon as={FiInbox} />
                <Text>Todas</Text>
                <Badge colorScheme="gray" borderRadius="full" px={2}>{todas.length}</Badge>
              </HStack>
            </Tab>
            <Tab fontWeight="semibold" _selected={{ bg: 'blue.50', color: 'blue.600' }}>
              <HStack spacing={2}>
                <Icon as={FiArchive} />
                <Text>Archivadas</Text>
                {stats.archivadas > 0 && (
                  <Badge colorScheme="blue" borderRadius="full" px={2}>{stats.archivadas}</Badge>
                )}
              </HStack>
            </Tab>
          </TabList>

          <TabPanels>
            {/* Panel: No leídas */}
            <TabPanel px={0} py={0}>
              <SlideFade in={true} offsetY="20px">
                {noLeidas.length === 0 ? (
                  <EmptyState
                    icon={FiCheckCircle}
                    title="¡Todo al día!"
                    description={searchTerm || filterTipo !== 'todas' ? "No hay notificaciones sin leer con los filtros actuales." : "No tienes notificaciones sin leer. ¡Excelente!"}
                  />
                ) : (
                  <VStack align="stretch" spacing={4}>
                    {noLeidas.map((notif) => (
                      <NotificacionCard
                        key={notif.id}
                        notificacion={notif}
                        onMarcarLeida={() => marcarComoLeida(notif.id)}
                        onArchivar={() => archivarNotificacion(notif.id)}
                      />
                    ))}
                  </VStack>
                )}
              </SlideFade>
            </TabPanel>

            {/* Panel: Todas */}
            <TabPanel px={0} py={0}>
              <SlideFade in={true} offsetY="20px">
                {todas.length === 0 ? (
                  <EmptyState
                    icon={FiInbox}
                    title="Bandeja vacía"
                    description={searchTerm || filterTipo !== 'todas' ? "No se encontraron notificaciones con los filtros seleccionados." : "Aún no has recibido ninguna notificación."}
                  />
                ) : (
                  <VStack align="stretch" spacing={4}>
                    {todas.map((notif) => (
                      <NotificacionCard
                        key={notif.id}
                        notificacion={notif}
                        onMarcarLeida={() => marcarComoLeida(notif.id)}
                        onArchivar={() => archivarNotificacion(notif.id)}
                      />
                    ))}
                  </VStack>
                )}
              </SlideFade>
            </TabPanel>

            {/* Panel: Archivadas */}
            <TabPanel px={0} py={0}>
              <SlideFade in={true} offsetY="20px">
                {archivadas.length === 0 ? (
                  <EmptyState
                    icon={FiArchive}
                    title="Archivo vacío"
                    description={searchTerm || filterTipo !== 'todas' ? "No hay notificaciones archivadas que coincidan con tu búsqueda." : "No has archivado ninguna notificación todavía."}
                  />
                ) : (
                  <VStack align="stretch" spacing={4}>
                    {archivadas.map((notif) => (
                      <NotificacionCard
                        key={notif.id}
                        notificacion={notif}
                        onMarcarLeida={() => marcarComoLeida(notif.id)}
                        onArchivar={() => archivarNotificacion(notif.id)}
                      />
                    ))}
                  </VStack>
                )}
              </SlideFade>
            </TabPanel>
          </TabPanels>
        </Tabs>
      </Box>
    </Layout>
  );
};

export default NotificacionesProfesor;
