import React from 'react';
import {
  Box,
  Text,
  SimpleGrid,
  Card,
  CardBody,
  HStack,
  VStack,
  Heading,
  Icon,
  Button,
  Flex,
  Avatar,
  Badge,
  Container,
  Divider,
  useColorModeValue
} from '@chakra-ui/react';
import { FiUsers, FiCalendar, FiFileText, FiClock, FiPlus, FiArrowRight, FiActivity } from 'react-icons/fi';
import { useNavigate } from 'react-router-dom';
import { motion } from 'framer-motion';
import Layout from '../../components/Layout';
import { mockClases, mockBitacoras, mockAlumnos, mockMaterias } from '../../data/mockData';

const MotionBox = motion(Box);
const MotionCard = motion(Card);

const containerVariants = {
  hidden: { opacity: 0 },
  visible: {
    opacity: 1,
    transition: {
      staggerChildren: 0.1
    }
  }
};

const itemVariants = {
  hidden: { y: 20, opacity: 0 },
  visible: {
    y: 0,
    opacity: 1,
    transition: {
      type: "spring",
      stiffness: 100
    }
  }
};

const profesorLinks = [
  { nombre: 'Dashboard', path: '/profesor' },
  { nombre: 'Mis Alumnos', path: '/profesor/alumnos' },
  { nombre: 'Registrar Bitácora', path: '/profesor/bitacora' },
  { nombre: 'Mi Calendario', path: '/profesor/calendario' },
  { nombre: 'Notificaciones', path: '/profesor/notificaciones' }
];

// Simulando que el profesor tiene ID 1
const profesorId = 1;
const misClases = mockClases.filter(c => c.id_profesor === profesorId);
const misBitacoras = mockBitacoras.filter(b => b.id_profesor === profesorId).sort((a, b) => new Date(b.fecha) - new Date(a.fecha));
const misAlumnosIds = [...new Set(misClases.map(c => c.id_alumno))];
const misAlumnos = mockAlumnos.filter(a => misAlumnosIds.includes(a.id));

const StatCard = ({ title, value, icon, color, subtext }) => (
  <MotionCard
    variants={itemVariants}
    bg="white"
    border="1px solid"
    borderColor="gray.100"
    shadow="sm"
    borderRadius="2xl"
    whileHover={{ y: -5, boxShadow: 'lg' }}
    transition={{ duration: 0.2 }}
  >
    <CardBody>
      <Flex justifyContent="space-between" alignItems="flex-start" mb={4}>
        <Box
          p={3}
          bgGradient={`linear(to-br, ${color}.400, ${color}.600)`}
          color="white"
          borderRadius="xl"
          boxShadow={`0 4px 12px ${color}.200`}
        >
          <Icon as={icon} boxSize={6} />
        </Box>
      </Flex>
      <VStack align="start" spacing={0}>
        <Text fontSize="3xl" fontWeight="800" color="gray.800" lineHeight="1">
          {value}
        </Text>
        <Text fontSize="sm" color="gray.500" fontWeight="600" mt={1}>
          {title}
        </Text>
        {subtext && (
          <Text fontSize="xs" color={`${color}.500`} fontWeight="700" mt={2}>
            {subtext}
          </Text>
        )}
      </VStack>
    </CardBody>
  </MotionCard>
);

const ClassCard = ({ clase }) => {
  const alumno = mockAlumnos.find(a => a.id === clase.id_alumno);
  const materia = mockMaterias.find(m => m.id === clase.id_materia);

  return (
    <MotionBox
      variants={itemVariants}
      p={4}
      bg="white"
      border="1px solid"
      borderColor="gray.100"
      borderRadius="2xl"
      display="flex"
      alignItems="center"
      justifyContent="space-between"
      whileHover={{ scale: 1.02, borderColor: 'purple.200' }}
      transition={{ duration: 0.2 }}
      boxShadow="sm"
    >
      <HStack spacing={4}>
        <Box
          p={3}
          bg="purple.50"
          color="purple.600"
          borderRadius="xl"
          textAlign="center"
          minW="80px"
        >
          <Text fontSize="xs" fontWeight="bold" textTransform="uppercase">{clase.dia_semana}</Text>
          <Text fontSize="sm" fontWeight="800">{clase.hora_inicio}</Text>
        </Box>
        <Box>
          <Text fontWeight="700" color="gray.800" fontSize="md">{materia?.nombre}</Text>
          <HStack spacing={2} mt={1}>
            <Avatar size="xs" src={alumno?.foto_url} name={alumno?.nombre} />
            <Text fontSize="sm" color="gray.500" fontWeight="500">{alumno?.nombre}</Text>
          </HStack>
        </Box>
      </HStack>
      <Badge colorScheme="purple" variant="solid" borderRadius="full" px={3}>
        En Curso
      </Badge>
    </MotionBox>
  );
};

const ActivityItem = ({ bitacora }) => {
  const alumno = mockAlumnos.find(a => a.id === bitacora.id_alumno);
  const materia = mockMaterias.find(m => m.id === bitacora.id_materia);

  return (
    <Flex gap={4} py={3}>
      <Box position="relative">
        <Box
          position="absolute"
          left="50%"
          top="0"
          bottom="0"
          w="2px"
          bg="gray.100"
          transform="translateX(-50%)"
          zIndex={0}
        />
        <Box
          position="relative"
          zIndex={1}
          boxSize={4}
          bg="teal.400"
          borderRadius="full"
          border="3px solid white"
          mt={1}
          boxShadow="sm"
        />
      </Box>
      <Box flex={1} pb={4}>
        <HStack justify="space-between" mb={1}>
          <Text fontSize="sm" fontWeight="700" color="gray.800">
            Bitácora Registrada
          </Text>
          <Text fontSize="xs" color="gray.400" fontWeight="500">
            {bitacora.fecha}
          </Text>
        </HStack>
        <Text fontSize="sm" color="gray.600" mb={2}>
          Para <Text as="span" fontWeight="600" color="teal.600">{alumno?.nombre}</Text> en {materia?.nombre}
        </Text>
        <Box
          bg="gray.50"
          p={3}
          borderRadius="xl"
          borderLeft="4px solid"
          borderLeftColor="teal.400"
        >
          <Text fontSize="xs" color="gray.600" noOfLines={2} fontStyle="italic">
            "{bitacora.actividad}"
          </Text>
        </Box>
      </Box>
    </Flex>
  );
};

const ProfesorDashboard = () => {
  const navigate = useNavigate();

  return (
    <Layout links={profesorLinks} userType="Profesor" userName="María González" userPhoto="https://images.unsplash.com/photo-1494790108377-be9c29b29330?ixlib=rb-1.2.1&auto=format&fit=facearea&facepad=2&w=256&h=256&q=80">
      <Container maxW="container.xl" px={0}>
        <MotionBox
          initial="hidden"
          animate="visible"
          variants={containerVariants}
        >
          {/* Header Section */}
          <Flex justify="space-between" align="center" mb={8}>
            <Box>
              <Heading size="lg" color="gray.800" mb={2}>Hola, María 👋</Heading>
              <Text color="gray.600">Tu espacio creativo y académico.</Text>
            </Box>
            <Button
              leftIcon={<FiPlus />}
              colorScheme="teal"
              size="md"
              onClick={() => navigate('/profesor/bitacora')}
              shadow="md"
              borderRadius="xl"
              _hover={{ transform: 'translateY(-2px)', shadow: 'lg' }}
            >
              Nueva Bitácora
            </Button>
          </Flex>

          {/* Stats Grid */}
          <SimpleGrid columns={{ base: 1, md: 3 }} spacing={6} mb={8}>
            <StatCard
              title="Alumnos Activos"
              value={misAlumnos.length}
              icon={FiUsers}
              color="teal"
              subtext="Total asignados"
            />
            <StatCard
              title="Clases Semanales"
              value={misClases.length}
              icon={FiCalendar}
              color="purple"
              subtext="Horas programadas"
            />
            <StatCard
              title="Bitácoras Realizadas"
              value={misBitacoras.length}
              icon={FiFileText}
              color="pink"
              subtext="En este periodo"
            />
          </SimpleGrid>

          <SimpleGrid columns={{ base: 1, lg: 3 }} spacing={8}>
            {/* Main Content - Upcoming Classes */}
            <Box gridColumn={{ lg: "span 2" }}>
              <Flex justify="space-between" align="center" mb={6}>
                <Heading size="md" color="gray.800">Próximas Clases</Heading>
                <Button variant="ghost" size="sm" rightIcon={<FiArrowRight />} colorScheme="purple">
                  Ver Calendario
                </Button>
              </Flex>
              <VStack spacing={4} align="stretch">
                {misClases.slice(0, 4).map((clase) => (
                  <ClassCard key={clase.id} clase={clase} />
                ))}
              </VStack>
            </Box>

            {/* Sidebar - Recent Activity */}
            <Box>
              <Flex justify="space-between" align="center" mb={6}>
                <Heading size="md" color="gray.800">Actividad Reciente</Heading>
              </Flex>
              <MotionCard
                variants={itemVariants}
                variant="outline"
                borderRadius="2xl"
                borderColor="gray.200"
                overflow="hidden"
              >
                <CardBody>
                  <VStack align="stretch" spacing={0}>
                    {misBitacoras.slice(0, 4).map((bitacora) => (
                      <ActivityItem key={bitacora.id} bitacora={bitacora} />
                    ))}
                    {misBitacoras.length === 0 && (
                      <Text color="gray.500" textAlign="center" py={4}>
                        No hay actividad reciente
                      </Text>
                    )}
                  </VStack>
                  <Button variant="link" colorScheme="teal" size="sm" w="full" mt={4}>
                    Ver todo el historial
                  </Button>
                </CardBody>
              </MotionCard>
            </Box>
          </SimpleGrid>
        </MotionBox>
      </Container>
    </Layout>
  );
};

export default ProfesorDashboard;

