import { useState, useEffect } from 'react';
import {
  Box,
  Heading,
  FormControl,
  FormLabel,
  Select,
  Input,
  Textarea,
  Button,
  VStack,
  HStack,
  useToast,
  Text,
  SimpleGrid,
  Card,
  CardBody,
  Icon,
  Avatar,
  Divider,
  Badge,
  InputGroup,
  InputLeftElement,
  Flex,
  useColorModeValue,
  Tooltip,
  Progress,
  Link
} from '@chakra-ui/react';
import {
  FaMicrophone,
  FaStop,
  FaSave,
  FaUserGraduate,
  FaCalendarAlt,
  FaBullseye,
  FaPalette,
  FaClipboardCheck,
  FaHome,
  FaCloudUploadAlt,
  FaMagic,
  FaRobot,
  FaBookOpen,
  FaFileUpload,
  FaTrash,
  FaFilePdf,
  FaFileImage,
  FaFileAlt,
  FaHistory,
  FaExternalLinkAlt
} from 'react-icons/fa';
import { Link as RouterLink } from 'react-router-dom';
import Layout from '../../components/Layout';
import { mockAlumnos, mockClases, mockBitacoras } from '../../data/mockData';

const profesorLinks = [
  { nombre: 'Dashboard', path: '/profesor' },
  { nombre: 'Mis Alumnos', path: '/profesor/alumnos' },
  { nombre: 'Registrar Bitácora', path: '/profesor/bitacora' },
  { nombre: 'Mi Calendario', path: '/profesor/calendario' },
  { nombre: 'Notificaciones', path: '/profesor/notificaciones' }
];

const profesorId = 1;
const misClases = mockClases.filter(c => c.id_profesor === profesorId);
const misAlumnosIds = [...new Set(misClases.map(c => c.id_alumno))];
const misAlumnos = mockAlumnos.filter(a => misAlumnosIds.includes(a.id));

const RegistroBitacora = () => {
  const [formData, setFormData] = useState({
    alumno: '',
    fecha: new Date().toISOString().split('T')[0],
    objetivo: '',
    actividad: '',
    evaluacion: '',
    tarea: '',
    tareaArchivos: []
  });
  const [grabando, setGrabando] = useState(false);
  const [procesando, setProcesando] = useState(false);
  const [selectedStudent, setSelectedStudent] = useState(null);
  const [lastBitacora, setLastBitacora] = useState(null);
  const toast = useToast();

  // Colors
  const cardBg = useColorModeValue('white', 'gray.700');
  const borderColor = useColorModeValue('gray.100', 'gray.600');

  useEffect(() => {
    if (formData.alumno) {
      const student = misAlumnos.find(a => a.id === parseInt(formData.alumno));
      setSelectedStudent(student);

      // Find last bitacora for this student
      const studentBitacoras = mockBitacoras.filter(b => b.id_alumno === parseInt(formData.alumno));
      // Sort by date descending
      studentBitacoras.sort((a, b) => new Date(b.fecha) - new Date(a.fecha));
      setLastBitacora(studentBitacoras.length > 0 ? studentBitacoras[0] : null);
    } else {
      setSelectedStudent(null);
      setLastBitacora(null);
    }
  }, [formData.alumno]);

  const handleIniciarGrabacion = () => {
    setGrabando(true);
    setProcesando(false);

    // Simulación de grabación y procesamiento
    setTimeout(() => {
      setGrabando(false);
      setProcesando(true);

      setTimeout(() => {
        // Rellenar automáticamente con datos de ejemplo
        setFormData(prev => ({
          ...prev,
          objetivo: 'Mejorar la técnica de digitación y expresión musical. Trabajar en la interpretación de obras clásicas con mayor fluidez y precisión.',
          actividad: 'Practicamos escalas mayores y menores en diferentes tonalidades. El alumno trabajó en una pieza de nivel intermedio, enfocándose en la dinámica y el fraseo. Se realizaron ejercicios de técnica de dedos y arpegios.',
          evaluacion: 'Excelente progreso. Se observa mejoría significativa en la velocidad y precisión. El alumno muestra gran dedicación y entusiasmo. Áreas a mejorar: mantener el tempo constante durante pasajes complejos.',
          tarea: 'Practicar la pieza estudiada durante 30 minutos diarios. Completar ejercicios de escalas en Do mayor, Sol mayor y Re mayor.'
        }));

        setProcesando(false);
        toast({
          title: '¡Magia completada! ✨',
          description: 'La IA ha generado el contenido de la bitácora basado en tu clase.',
          status: 'success',
          duration: 4000,
          isClosable: true,
          position: 'top-right'
        });
      }, 2500);
    }, 3000);
  };

  const handleSubmit = (e) => {
    e.preventDefault();
    toast({
      title: 'Bitácora registrada',
      description: 'La bitácora ha sido guardada correctamente en el expediente del alumno.',
      status: 'success',
      duration: 3000,
      isClosable: true,
    });

    setFormData({
      alumno: '',
      fecha: new Date().toISOString().split('T')[0],
      objetivo: '',
      actividad: '',
      evaluacion: '',
      tarea: '',
      tareaArchivos: []
    });
    setSelectedStudent(null);
    setLastBitacora(null);
  };

  return (
    <Layout links={profesorLinks} userType="Profesor" userName="María González">
      <Box maxW="1600px" mx="auto">
        <VStack align="start" spacing={2} mb={8}>
          <Heading size="lg" bgGradient="linear(to-r, brand.500, brand.300)" bgClip="text">
            Registro de Bitácora
          </Heading>
          <Text fontSize="md" color="gray.600">
            Documenta el progreso de tus alumnos. Utiliza nuestra IA para agilizar el proceso.
          </Text>
        </VStack>

        <SimpleGrid columns={{ base: 1, lg: 3 }} spacing={8}>
          {/* Columna Izquierda: Formulario Principal */}
          <Box gridColumn={{ lg: "span 2" }}>
            <Card bg={cardBg} boxShadow="xl" borderRadius="2xl" border="1px solid" borderColor={borderColor}>
              <CardBody p={8}>
                <form onSubmit={handleSubmit}>
                  <VStack spacing={6}>
                    {/* Selección de Alumno y Fecha */}
                    <SimpleGrid columns={{ base: 1, md: 2 }} spacing={6} w="100%">
                      <FormControl isRequired>
                        <FormLabel fontWeight="bold" color="gray.700">
                          <HStack>
                            <Icon as={FaUserGraduate} color="brand.500" />
                            <Text>Alumno</Text>
                          </HStack>
                        </FormLabel>
                        <Select
                          value={formData.alumno}
                          onChange={(e) => setFormData({ ...formData, alumno: e.target.value })}
                          placeholder="Selecciona un alumno"
                          size="lg"
                          borderRadius="xl"
                          focusBorderColor="brand.500"
                        >
                          {misAlumnos.map(alumno => (
                            <option key={alumno.id} value={alumno.id}>{alumno.nombre}</option>
                          ))}
                        </Select>
                      </FormControl>

                      <FormControl isRequired>
                        <FormLabel fontWeight="bold" color="gray.700">
                          <HStack>
                            <Icon as={FaCalendarAlt} color="brand.500" />
                            <Text>Fecha de Clase</Text>
                          </HStack>
                        </FormLabel>
                        <Input
                          type="date"
                          value={formData.fecha}
                          onChange={(e) => setFormData({ ...formData, fecha: e.target.value })}
                          size="lg"
                          borderRadius="xl"
                          focusBorderColor="brand.500"
                        />
                      </FormControl>
                    </SimpleGrid>

                    <Divider />

                    {/* Campos de Texto */}
                    <FormControl isRequired>
                      <FormLabel fontWeight="bold" color="gray.700">
                        <HStack>
                          <Icon as={FaBullseye} color="red.400" />
                          <Text>Objetivo de la Clase</Text>
                        </HStack>
                      </FormLabel>
                      <Textarea
                        value={formData.objetivo}
                        onChange={(e) => setFormData({ ...formData, objetivo: e.target.value })}
                        placeholder="¿Qué se buscaba lograr hoy?"
                        rows={3}
                        borderRadius="xl"
                        focusBorderColor="brand.500"
                        bg="gray.50"
                      />
                    </FormControl>

                    <FormControl isRequired>
                      <FormLabel fontWeight="bold" color="gray.700">
                        <HStack>
                          <Icon as={FaPalette} color="orange.400" />
                          <Text>Actividades Realizadas</Text>
                        </HStack>
                      </FormLabel>
                      <Textarea
                        value={formData.actividad}
                        onChange={(e) => setFormData({ ...formData, actividad: e.target.value })}
                        placeholder="Describe los ejercicios y trabajos realizados..."
                        rows={4}
                        borderRadius="xl"
                        focusBorderColor="brand.500"
                        bg="gray.50"
                      />
                    </FormControl>

                    <FormControl isRequired>
                      <FormLabel fontWeight="bold" color="gray.700">
                        <HStack>
                          <Icon as={FaClipboardCheck} color="green.400" />
                          <Text>Evaluación / Observaciones</Text>
                        </HStack>
                      </FormLabel>
                      <Textarea
                        value={formData.evaluacion}
                        onChange={(e) => setFormData({ ...formData, evaluacion: e.target.value })}
                        placeholder="Desempeño, puntos fuertes y áreas de mejora..."
                        rows={4}
                        borderRadius="xl"
                        focusBorderColor="brand.500"
                        bg="gray.50"
                      />
                    </FormControl>

                    <Box bg="blue.50" p={5} borderRadius="xl" border="1px dashed" borderColor="blue.200">
                      <HStack mb={4} justify="space-between">
                        <HStack>
                          <Icon as={FaBookOpen} color="blue.500" />
                          <Heading size="sm" color="blue.700">Asignación de Tarea</Heading>
                        </HStack>
                        <Badge colorScheme="blue" borderRadius="full" px={2}>
                          {formData.tareaArchivos.length} archivos
                        </Badge>
                      </HStack>

                      <SimpleGrid columns={{ base: 1, md: 2 }} spacing={5}>
                        <FormControl>
                          <FormLabel fontSize="sm" fontWeight="bold" color="gray.600">Instrucciones</FormLabel>
                          <Textarea
                            value={formData.tarea}
                            onChange={(e) => setFormData({ ...formData, tarea: e.target.value })}
                            placeholder="Describe los ejercicios o trabajos para casa..."
                            rows={6}
                            bg="white"
                            borderRadius="lg"
                            focusBorderColor="blue.400"
                          />
                        </FormControl>

                        <VStack spacing={3} align="stretch">
                          <FormControl>
                            <FormLabel fontSize="sm" fontWeight="bold" color="gray.600">Material de Apoyo</FormLabel>
                            <Flex
                              bg="white"
                              h="80px"
                              borderRadius="lg"
                              border="2px dashed"
                              borderColor="gray.300"
                              align="center"
                              justify="center"
                              direction="column"
                              cursor="pointer"
                              transition="all 0.2s"
                              _hover={{ borderColor: 'blue.400', bg: 'blue.50' }}
                              onClick={() => document.getElementById('tarea-upload').click()}
                            >
                              <HStack color="blue.400">
                                <Icon as={FaFileUpload} />
                                <Text fontSize="sm" fontWeight="medium">Adjuntar Archivos</Text>
                              </HStack>
                              <Text fontSize="xs" color="gray.400" mt={1}>PDF, Imágenes (Max 5MB)</Text>
                              <Input
                                id="tarea-upload"
                                type="file"
                                multiple
                                accept=".pdf,image/*"
                                display="none"
                                onChange={(e) => {
                                  if (e.target.files.length > 0) {
                                    const newFiles = Array.from(e.target.files);
                                    setFormData(prev => ({
                                      ...prev,
                                      tareaArchivos: [...prev.tareaArchivos, ...newFiles]
                                    }));
                                    toast({
                                      title: 'Archivos adjuntados',
                                      description: `Se agregaron ${newFiles.length} archivos`,
                                      status: 'success',
                                      duration: 2000,
                                    });
                                  }
                                }}
                              />
                            </Flex>
                          </FormControl>

                          {/* Lista de archivos adjuntos */}
                          {formData.tareaArchivos.length > 0 && (
                            <VStack
                              align="stretch"
                              spacing={2}
                              maxH="150px"
                              overflowY="auto"
                              css={{
                                '&::-webkit-scrollbar': { width: '4px' },
                                '&::-webkit-scrollbar-track': { width: '6px' },
                                '&::-webkit-scrollbar-thumb': { background: '#CBD5E0', borderRadius: '24px' },
                              }}
                            >
                              {formData.tareaArchivos.map((file, index) => (
                                <HStack
                                  key={index}
                                  bg="white"
                                  p={2}
                                  borderRadius="md"
                                  justify="space-between"
                                  boxShadow="sm"
                                >
                                  <HStack spacing={3} overflow="hidden">
                                    <Icon
                                      as={file.type.includes('pdf') ? FaFilePdf : file.type.includes('image') ? FaFileImage : FaFileAlt}
                                      color={file.type.includes('pdf') ? "red.400" : "green.400"}
                                    />
                                    <Text fontSize="xs" noOfLines={1} title={file.name}>{file.name}</Text>
                                  </HStack>
                                  <Icon
                                    as={FaTrash}
                                    color="gray.400"
                                    cursor="pointer"
                                    _hover={{ color: 'red.500' }}
                                    onClick={() => {
                                      setFormData(prev => ({
                                        ...prev,
                                        tareaArchivos: prev.tareaArchivos.filter((_, i) => i !== index)
                                      }));
                                    }}
                                  />
                                </HStack>
                              ))}
                            </VStack>
                          )}
                        </VStack>
                      </SimpleGrid>
                    </Box>

                    <FormControl>
                      <FormLabel fontWeight="bold" color="gray.700">Evidencias (Fotos/Videos)</FormLabel>
                      <Flex
                        border="2px dashed"
                        borderColor="gray.300"
                        borderRadius="xl"
                        p={8}
                        align="center"
                        justify="center"
                        direction="column"
                        cursor="pointer"
                        transition="all 0.2s"
                        _hover={{ borderColor: 'brand.500', bg: 'brand.50' }}
                        onClick={() => document.getElementById('file-upload').click()}
                      >
                        <Icon as={FaCloudUploadAlt} w={10} h={10} color="gray.400" mb={2} />
                        <Text color="gray.500" fontWeight="medium">Haz clic para subir o arrastra archivos aquí</Text>
                        <Text fontSize="xs" color="gray.400">Soporta imágenes y audio</Text>
                        <Input
                          id="file-upload"
                          type="file"
                          accept="image/*,audio/*"
                          display="none"
                          onChange={(e) => {
                            if (e.target.files[0]) {
                              toast({
                                title: 'Archivo listo',
                                description: `${e.target.files[0].name} preparado para subir`,
                                status: 'info',
                                duration: 2000,
                              });
                            }
                          }}
                        />
                      </Flex>
                    </FormControl>

                    <Button
                      type="submit"
                      colorScheme="brand"
                      size="lg"
                      w="100%"
                      h="60px"
                      fontSize="xl"
                      borderRadius="xl"
                      boxShadow="lg"
                      leftIcon={<Icon as={FaSave} />}
                      _hover={{ transform: 'translateY(-2px)', boxShadow: 'xl' }}
                    >
                      Guardar Bitácora
                    </Button>
                  </VStack>
                </form>
              </CardBody>
            </Card>
          </Box>

          {/* Columna Derecha: Herramientas IA y Contexto */}
          <VStack spacing={6} align="stretch">
            {/* Tarjeta IA */}
            <Card bgGradient="linear(to-br, brand.600, brand.800)" color="white" borderRadius="2xl" boxShadow="xl" overflow="hidden">
              <CardBody p={6} position="relative">
                <Box position="absolute" top={-4} right={-4} opacity={0.1}>
                  <Icon as={FaRobot} w={40} h={40} />
                </Box>

                <VStack spacing={4} align="center" position="relative" zIndex={1}>
                  <HStack>
                    <Icon as={FaMagic} color="yellow.300" />
                    <Heading size="md">Asistente IA</Heading>
                  </HStack>

                  <Text fontSize="sm" textAlign="center" opacity={0.9}>
                    Graba un resumen de tu clase y deja que la IA complete el formulario por ti.
                  </Text>

                  <Box
                    w="80px"
                    h="80px"
                    borderRadius="full"
                    bg={grabando ? "red.500" : "whiteAlpha.300"}
                    display="flex"
                    alignItems="center"
                    justifyContent="center"
                    cursor={procesando ? "not-allowed" : "pointer"}
                    onClick={!procesando ? handleIniciarGrabacion : undefined}
                    transition="all 0.3s"
                    _hover={{ transform: 'scale(1.1)', bg: grabando ? "red.600" : "whiteAlpha.400" }}
                    animation={grabando ? "pulse 1.5s infinite" : "none"}
                    sx={{
                      "@keyframes pulse": {
                        "0%": { boxShadow: "0 0 0 0 rgba(255, 82, 82, 0.7)" },
                        "70%": { boxShadow: "0 0 0 20px rgba(255, 82, 82, 0)" },
                        "100%": { boxShadow: "0 0 0 0 rgba(255, 82, 82, 0)" },
                      }
                    }}
                  >
                    <Icon as={grabando ? FaStop : FaMicrophone} w={8} h={8} color="white" />
                  </Box>

                  <Text fontWeight="bold">
                    {grabando ? "Grabando..." : procesando ? "Procesando..." : "Tocar para Grabar"}
                  </Text>

                  {procesando && (
                    <Box w="100%">
                      <Progress size="xs" isIndeterminate colorScheme="yellow" borderRadius="full" />
                      <Text fontSize="xs" textAlign="center" mt={2}>Analizando audio y generando texto...</Text>
                    </Box>
                  )}
                </VStack>
              </CardBody>
            </Card>

            {/* Tarjeta de Alumno Seleccionado (Contexto) */}
            {selectedStudent ? (
              <Card bg="white" borderRadius="2xl" boxShadow="lg" border="1px solid" borderColor="gray.100">
                <CardBody>
                  <VStack spacing={4}>
                    <Avatar size="xl" src={selectedStudent.foto_url} name={selectedStudent.nombre} ring={4} ringColor="brand.50" />
                    <Box textAlign="center">
                      <Heading size="md">{selectedStudent.nombre}</Heading>
                      <Text color="gray.500" fontSize="sm">{selectedStudent.edad} años</Text>
                    </Box>

                    <Divider />

                    <Box w="100%">
                      <HStack justify="space-between" mb={2}>
                        <Heading size="xs" color="gray.500" textTransform="uppercase">Última Bitácora</Heading>
                        {lastBitacora && (
                          <Badge colorScheme="green" fontSize="xs">{lastBitacora.fecha}</Badge>
                        )}
                      </HStack>

                      {lastBitacora ? (
                        <VStack align="start" spacing={3}>
                          <Box>
                            <Text fontSize="xs" fontWeight="bold" color="gray.700">Objetivo:</Text>
                            <Text fontSize="sm" color="gray.600" noOfLines={2} title={lastBitacora.objetivo}>
                              {lastBitacora.objetivo}
                            </Text>
                          </Box>

                          <Box>
                            <Text fontSize="xs" fontWeight="bold" color="gray.700">Evaluación:</Text>
                            <Text fontSize="sm" color="gray.600" noOfLines={2} title={lastBitacora.evaluacion}>
                              {lastBitacora.evaluacion}
                            </Text>
                          </Box>
                        </VStack>
                      ) : (
                        <Text fontSize="sm" color="gray.400" fontStyle="italic" textAlign="center" py={2}>
                          No hay registros previos.
                        </Text>
                      )}
                    </Box>

                    <Divider />

                    <Button
                      as={RouterLink}
                      to={`/profesor/alumnos`}
                      width="100%"
                      variant="outline"
                      colorScheme="brand"
                      leftIcon={<Icon as={FaHistory} />}
                      size="sm"
                    >
                      Ver Historial Completo
                    </Button>
                  </VStack>
                </CardBody>
              </Card>
            ) : (
              <Card bg="gray.50" borderRadius="2xl" border="2px dashed" borderColor="gray.200">
                <CardBody display="flex" flexDirection="column" alignItems="center" justifyContent="center" py={10}>
                  <Icon as={FaUserGraduate} w={10} h={10} color="gray.300" mb={3} />
                  <Text color="gray.400" textAlign="center">
                    Selecciona un alumno para ver su información rápida
                  </Text>
                </CardBody>
              </Card>
            )}
          </VStack>
        </SimpleGrid>
      </Box>
    </Layout>
  );
};

export default RegistroBitacora;

