import { useState, useEffect } from 'react';
import {
    Box,
    Button,
    Heading,
    Table,
    Thead,
    Tbody,
    Tr,
    Th,
    Td,
    Badge,
    IconButton,
    useDisclosure,
    Modal,
    ModalOverlay,
    ModalContent,
    ModalHeader,
    ModalFooter,
    ModalBody,
    ModalCloseButton,
    FormControl,
    FormLabel,
    Input,
    Select,
    useToast,
    HStack,
    Text,
    Avatar
} from '@chakra-ui/react';
import { FiEdit2, FiTrash2, FiPlus, FiPower, FiRefreshCw } from 'react-icons/fi';
import { getUsers, getCompanies, createUser, updateUser } from '../../services/api';

const GestionAdministradores = () => {
    // Filtrar solo usuarios ADMIN
    const [admins, setAdmins] = useState([]);
    const [companies, setCompanies] = useState([]);
    const [currentAdmin, setCurrentAdmin] = useState(null);
    const { isOpen, onOpen, onClose } = useDisclosure();
    const toast = useToast();

    const fetchData = async () => {
        try {
            const [usersData, companiesData] = await Promise.all([
                getUsers(),
                getCompanies()
            ]);
            setAdmins(usersData.filter(u => u.role === 'ADMIN'));
            setCompanies(companiesData);
        } catch (error) {
            console.error(error);
            toast({
                title: 'Error al cargar datos',
                status: 'error',
                duration: 3000,
            });
        }
    };

    useEffect(() => {
        fetchData();
    }, []);

    const getCompanyName = (companyId) => {
        const company = companies.find(c => c.id === companyId);
        return company ? company.name : 'Desconocida';
    };

    const handleEdit = (admin) => {
        setCurrentAdmin(admin);
        onOpen();
    };

    const handleCreate = () => {
        setCurrentAdmin({
            email: '',
            role: 'ADMIN',
            company_id: '',
            is_active: true
        });
        onOpen();
    };

    const handleSave = async () => {
        try {
            if (currentAdmin.id) {
                const dataToUpdate = { ...currentAdmin };
                if (!dataToUpdate.password) {
                    delete dataToUpdate.password;
                }
                await updateUser(currentAdmin.id, dataToUpdate);
                toast({
                    title: 'Administrador actualizado',
                    status: 'success',
                    duration: 2000,
                });
            } else {
                if (!currentAdmin.password) {
                    toast({
                        title: 'Error',
                        description: 'La contraseña es requerida para nuevos administradores',
                        status: 'error',
                        duration: 3000,
                    });
                    return;
                }
                await createUser({
                    ...currentAdmin,
                    id: `usr-${Date.now()}`,
                    // password will be hashed by backend
                });
                toast({
                    title: 'Administrador creado',
                    status: 'success',
                    duration: 2000,
                });
            }
            fetchData();
            onClose();
        } catch (error) {
            toast({
                title: 'Error al guardar',
                description: error.message,
                status: 'error',
                duration: 3000,
            });
        }
    };

    const toggleStatus = async (id) => {
        const admin = admins.find(a => a.id === id);
        if (!admin) return;

        try {
            const newStatus = !admin.is_active;
            await updateUser(id, { ...admin, is_active: newStatus });

            toast({
                title: `Administrador ${newStatus ? 'activado' : 'desactivado'}`,
                status: newStatus ? 'success' : 'warning',
                duration: 2000,
            });
            fetchData();
        } catch (error) {
            toast({
                title: 'Error al actualizar estado',
                description: error.message,
                status: 'error',
                duration: 3000,
            });
        }
    };

    const resetPassword = (id) => {
        toast({
            title: 'Contraseña restablecida',
            description: 'Se ha enviado un correo con la nueva contraseña',
            status: 'info',
            duration: 3000,
        });
    };

    return (
        <Box>
            <HStack justify="space-between" mb={6}>
                <Heading>Gestión de Administradores</Heading>
                <Button leftIcon={<FiPlus />} colorScheme="brand" onClick={handleCreate}>
                    Nuevo Administrador
                </Button>
            </HStack>

            <Box bg="white" borderRadius="lg" boxShadow="sm" overflowX="auto">
                <Table variant="simple">
                    <Thead bg="gray.50">
                        <Tr>
                            <Th>Usuario</Th>
                            <Th>Empresa</Th>
                            <Th>Estado</Th>
                            <Th>Acciones</Th>
                        </Tr>
                    </Thead>
                    <Tbody>
                        {admins.map((admin) => (
                            <Tr key={admin.id}>
                                <Td>
                                    <HStack>
                                        <Avatar size="sm" name={admin.email} />
                                        <Box>
                                            <Text fontWeight="bold">{admin.email}</Text>
                                            <Text fontSize="xs" color="gray.500">ID: {admin.id}</Text>
                                        </Box>
                                    </HStack>
                                </Td>
                                <Td>
                                    <Text fontSize="sm" fontWeight="medium">
                                        {getCompanyName(admin.company_id)}
                                    </Text>
                                </Td>
                                <Td>
                                    <Badge colorScheme={admin.is_active ? 'green' : 'red'}>
                                        {admin.is_active ? 'ACTIVO' : 'INACTIVO'}
                                    </Badge>
                                </Td>
                                <Td>
                                    <HStack spacing={2}>
                                        <IconButton
                                            icon={<FiEdit2 />}
                                            size="sm"
                                            aria-label="Editar"
                                            onClick={() => handleEdit(admin)}
                                        />
                                        <IconButton
                                            icon={<FiRefreshCw />}
                                            size="sm"
                                            colorScheme="blue"
                                            variant="ghost"
                                            aria-label="Reset Password"
                                            title="Restablecer Contraseña"
                                            onClick={() => resetPassword(admin.id)}
                                        />
                                        <IconButton
                                            icon={<FiPower />}
                                            size="sm"
                                            colorScheme={admin.is_active ? 'red' : 'green'}
                                            aria-label="Cambiar estado"
                                            onClick={() => toggleStatus(admin.id)}
                                        />
                                    </HStack>
                                </Td>
                            </Tr>
                        ))}
                    </Tbody>
                </Table>
            </Box>

            {/* Modal */}
            <Modal isOpen={isOpen} onClose={onClose}>
                <ModalOverlay />
                <ModalContent>
                    <ModalHeader>{currentAdmin?.id ? 'Editar Administrador' : 'Nuevo Administrador'}</ModalHeader>
                    <ModalCloseButton />
                    <ModalBody>
                        <Box display="flex" flexDirection="column" gap={4}>
                            <FormControl>
                                <FormLabel>Correo Electrónico</FormLabel>
                                <Input
                                    type="email"
                                    value={currentAdmin?.email || ''}
                                    onChange={(e) => setCurrentAdmin({ ...currentAdmin, email: e.target.value })}
                                />
                            </FormControl>

                            <FormControl>
                                <FormLabel>Empresa Asignada</FormLabel>
                                <Select
                                    value={currentAdmin?.company_id || ''}
                                    onChange={(e) => setCurrentAdmin({ ...currentAdmin, company_id: parseInt(e.target.value) })}
                                >
                                    <option value="">Seleccionar empresa...</option>
                                    {companies.map(company => (
                                        <option key={company.id} value={company.id}>
                                            {company.name}
                                        </option>
                                    ))}
                                </Select>
                            </FormControl>

                            <FormControl>
                                <FormLabel>Contraseña {currentAdmin?.id && '(Dejar en blanco para no cambiar)'}</FormLabel>
                                <Input
                                    type="password"
                                    placeholder={currentAdmin?.id ? '••••••••' : 'Ingresar contraseña'}
                                    value={currentAdmin?.password || ''}
                                    onChange={(e) => setCurrentAdmin({ ...currentAdmin, password: e.target.value })}
                                />
                            </FormControl>
                        </Box>
                    </ModalBody>

                    <ModalFooter>
                        <Button variant="ghost" mr={3} onClick={onClose}>
                            Cancelar
                        </Button>
                        <Button colorScheme="brand" onClick={handleSave}>
                            Guardar
                        </Button>
                    </ModalFooter>
                </ModalContent>
            </Modal>
        </Box>
    );
};

export default GestionAdministradores;
