import { useState, useEffect } from 'react';
import {
    Box,
    Button,
    Heading,
    Table,
    Thead,
    Tbody,
    Tr,
    Th,
    Td,
    Badge,
    IconButton,
    useDisclosure,
    Modal,
    ModalOverlay,
    ModalContent,
    ModalHeader,
    ModalFooter,
    ModalBody,
    ModalCloseButton,
    FormControl,
    FormLabel,
    Input,
    Select,
    useToast,
    HStack,
    Text,
    Avatar
} from '@chakra-ui/react';
import { FiEdit2, FiTrash2, FiPlus, FiPower } from 'react-icons/fi';
import { getCompanies, createCompany, updateCompany } from '../../services/api';

const GestionEmpresas = () => {
    const [companies, setCompanies] = useState([]);
    const [currentCompany, setCurrentCompany] = useState(null);
    const { isOpen, onOpen, onClose } = useDisclosure();
    const toast = useToast();

    const fetchCompanies = async () => {
        try {
            const data = await getCompanies();
            setCompanies(data);
        } catch (error) {
            console.error(error);
            toast({
                title: 'Error al cargar empresas',
                status: 'error',
                duration: 3000,
            });
        }
    };

    useEffect(() => {
        fetchCompanies();
    }, []);

    const handleEdit = (company) => {
        setCurrentCompany(company);
        onOpen();
    };

    const handleCreate = () => {
        setCurrentCompany({
            name: '',
            email: '',
            phone: '',
            is_active: true,
            street: '',
            exterior_number: '',
            neighborhood: '',
            municipality: '',
            state: '',
            postal_code: ''
        });
        onOpen();
    };

    const handleSave = async () => {
        try {
            if (currentCompany.id) {
                await updateCompany(currentCompany.id, currentCompany);
                toast({
                    title: 'Empresa actualizada',
                    status: 'success',
                    duration: 2000,
                });
            } else {
                await createCompany({
                    ...currentCompany,
                    logo_url: 'https://via.placeholder.com/150' // Default logo
                });
                toast({
                    title: 'Empresa creada',
                    status: 'success',
                    duration: 2000,
                });
            }
            fetchCompanies();
            onClose();
        } catch (error) {
            toast({
                title: 'Error al guardar',
                description: error.message,
                status: 'error',
                duration: 3000,
            });
        }
    };

    const toggleStatus = async (id) => {
        const company = companies.find(c => c.id === id);
        if (!company) return;

        try {
            const newStatus = !company.is_active;
            await updateCompany(id, { ...company, is_active: newStatus });

            toast({
                title: `Empresa ${newStatus ? 'activada' : 'desactivada'}`,
                status: newStatus ? 'success' : 'warning',
                duration: 2000,
            });
            fetchCompanies();
        } catch (error) {
            toast({
                title: 'Error al actualizar estado',
                description: error.message,
                status: 'error',
                duration: 3000,
            });
        }
    };

    return (
        <Box>
            <HStack justify="space-between" mb={6}>
                <Heading>Gestión de Empresas</Heading>
                <Button leftIcon={<FiPlus />} colorScheme="brand" onClick={handleCreate}>
                    Nueva Empresa
                </Button>
            </HStack>

            <Box bg="white" borderRadius="lg" boxShadow="sm" overflowX="auto">
                <Table variant="simple">
                    <Thead bg="gray.50">
                        <Tr>
                            <Th>Empresa</Th>
                            <Th>Contacto</Th>
                            <Th>Ubicación</Th>
                            <Th>Estado</Th>
                            <Th>Acciones</Th>
                        </Tr>
                    </Thead>
                    <Tbody>
                        {companies.map((company) => (
                            <Tr key={company.id}>
                                <Td>
                                    <HStack>
                                        <Avatar size="sm" name={company.name} src={company.logo_url} />
                                        <Box>
                                            <Text fontWeight="bold">{company.name}</Text>
                                            <Text fontSize="xs" color="gray.500">ID: {company.id}</Text>
                                        </Box>
                                    </HStack>
                                </Td>
                                <Td>
                                    <Text fontSize="sm">{company.email}</Text>
                                    <Text fontSize="xs" color="gray.500">{company.phone}</Text>
                                </Td>
                                <Td>
                                    <Text fontSize="sm">{company.municipality}, {company.state}</Text>
                                </Td>
                                <Td>
                                    <Badge colorScheme={company.is_active ? 'green' : 'red'}>
                                        {company.is_active ? 'ACTIVA' : 'INACTIVA'}
                                    </Badge>
                                </Td>
                                <Td>
                                    <HStack spacing={2}>
                                        <IconButton
                                            icon={<FiEdit2 />}
                                            size="sm"
                                            aria-label="Editar"
                                            onClick={() => handleEdit(company)}
                                        />
                                        <IconButton
                                            icon={<FiPower />}
                                            size="sm"
                                            colorScheme={company.is_active ? 'red' : 'green'}
                                            aria-label="Cambiar estado"
                                            onClick={() => toggleStatus(company.id)}
                                        />
                                    </HStack>
                                </Td>
                            </Tr>
                        ))}
                    </Tbody>
                </Table>
            </Box>

            {/* Modal de Edición/Creación */}
            <Modal isOpen={isOpen} onClose={onClose} size="xl">
                <ModalOverlay />
                <ModalContent>
                    <ModalHeader>{currentCompany?.id ? 'Editar Empresa' : 'Nueva Empresa'}</ModalHeader>
                    <ModalCloseButton />
                    <ModalBody>
                        <Box display="grid" gridTemplateColumns="1fr 1fr" gap={4}>
                            <FormControl gridColumn="span 2">
                                <FormLabel>Nombre Comercial</FormLabel>
                                <Input
                                    value={currentCompany?.name || ''}
                                    onChange={(e) => setCurrentCompany({ ...currentCompany, name: e.target.value })}
                                />
                            </FormControl>

                            <FormControl>
                                <FormLabel>Email de Contacto</FormLabel>
                                <Input
                                    value={currentCompany?.email || ''}
                                    onChange={(e) => setCurrentCompany({ ...currentCompany, email: e.target.value })}
                                />
                            </FormControl>

                            <FormControl>
                                <FormLabel>Teléfono</FormLabel>
                                <Input
                                    value={currentCompany?.phone || ''}
                                    onChange={(e) => setCurrentCompany({ ...currentCompany, phone: e.target.value })}
                                />
                            </FormControl>

                            <FormControl>
                                <FormLabel>Calle</FormLabel>
                                <Input
                                    value={currentCompany?.street || ''}
                                    onChange={(e) => setCurrentCompany({ ...currentCompany, street: e.target.value })}
                                />
                            </FormControl>

                            <FormControl>
                                <FormLabel>Número Exterior</FormLabel>
                                <Input
                                    value={currentCompany?.exterior_number || ''}
                                    onChange={(e) => setCurrentCompany({ ...currentCompany, exterior_number: e.target.value })}
                                />
                            </FormControl>

                            <FormControl>
                                <FormLabel>Colonia</FormLabel>
                                <Input
                                    value={currentCompany?.neighborhood || ''}
                                    onChange={(e) => setCurrentCompany({ ...currentCompany, neighborhood: e.target.value })}
                                />
                            </FormControl>

                            <FormControl>
                                <FormLabel>Municipio</FormLabel>
                                <Input
                                    value={currentCompany?.municipality || ''}
                                    onChange={(e) => setCurrentCompany({ ...currentCompany, municipality: e.target.value })}
                                />
                            </FormControl>

                            <FormControl>
                                <FormLabel>Estado</FormLabel>
                                <Input
                                    value={currentCompany?.state || ''}
                                    onChange={(e) => setCurrentCompany({ ...currentCompany, state: e.target.value })}
                                />
                            </FormControl>

                            <FormControl>
                                <FormLabel>Código Postal</FormLabel>
                                <Input
                                    value={currentCompany?.postal_code || ''}
                                    onChange={(e) => setCurrentCompany({ ...currentCompany, postal_code: e.target.value })}
                                />
                            </FormControl>
                        </Box>
                    </ModalBody>

                    <ModalFooter>
                        <Button variant="ghost" mr={3} onClick={onClose}>
                            Cancelar
                        </Button>
                        <Button colorScheme="brand" onClick={handleSave}>
                            Guardar
                        </Button>
                    </ModalFooter>
                </ModalContent>
            </Modal>
        </Box>
    );
};

export default GestionEmpresas;
