const API_URL = import.meta.env.PROD ? 'https://keservicios.kemarketing.mx/api' : '/api';

// Helper para fetch
const fetchApi = async (endpoint, options = {}) => {
    const token = sessionStorage.getItem('token');
    const headers = {
        'Content-Type': 'application/json',
        ...options.headers
    };

    if (token) {
        headers['Authorization'] = `Bearer ${token}`;
    }

    const config = {
        ...options,
        headers
    };

    const response = await fetch(`${API_URL}${endpoint}`, config);
    if (!response.ok) {
        if (response.status === 401) {
            // Token expired or invalid
            sessionStorage.removeItem('token');
            sessionStorage.removeItem('user');
            // Redirect to login if not already there
            if (!window.location.pathname.includes('/login') && window.location.pathname !== '/') {
                window.location.href = '/login';
            }
        }
        const error = await response.json();
        throw new Error(error.error || 'Error en la petición');
    }
    return response.json();
};

export const login = async (email, password) => {
    const response = await fetch(`${API_URL}/auth/login`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ email, password })
    });
    if (!response.ok) {
        const error = await response.json();
        throw new Error(error.error || 'Error en el inicio de sesión');
    }
    return response.json();
};

export const clientLogin = async (clientIdentifier, serviceFolio) => {
    const response = await fetch(`${API_URL}/auth/client-login`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ clientIdentifier, serviceFolio })
    });
    if (!response.ok) {
        const error = await response.json();
        throw new Error(error.error || 'Error en el inicio de sesión');
    }
    return response.json();
};

// Companies
export const getCompanies = () => fetchApi('/companies');
export const getCompanyById = (id) => fetchApi(`/companies/${id}`);
export const createCompany = (data) => fetchApi('/companies', {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify(data)
});
export const updateCompany = (id, data) => fetchApi(`/companies/${id}`, {
    method: 'PUT',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify(data)
});
export const deleteCompany = (id) => fetchApi(`/companies/${id}`, {
    method: 'DELETE'
});
export const isCompanyActive = async (companyId) => {
    try {
        const company = await getCompanyById(companyId);
        return company.is_active;
    } catch (e) {
        return false;
    }
};

// Users
export const getUsers = () => fetchApi('/users');
export const getUserById = async (id) => {
    const users = await getUsers();
    return users.find(u => u.id === id);
};
export const createUser = (data) => fetchApi('/users', {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify(data)
});
export const updateUser = (id, data) => fetchApi(`/users/${id}`, {
    method: 'PUT',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify(data)
});
export const deleteUser = (id) => fetchApi(`/users/${id}`, {
    method: 'DELETE'
});
export const isUserActive = async (userId) => {
    try {
        const user = await getUserById(userId);
        return user ? user.is_active : false;
    } catch (e) {
        return false;
    }
};

// Clients
export const getClients = () => fetchApi('/clients');
export const getClientByUserId = async (userId) => {
    const clients = await getClients();
    return clients.find(c => c.user_id === userId);
};
export const createClient = (data) => fetchApi('/clients', {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify(data)
});
export const updateClient = (id, data) => fetchApi(`/clients/${id}`, {
    method: 'PUT',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify(data)
});
export const deleteClient = (id) => fetchApi(`/clients/${id}`, {
    method: 'DELETE'
});

// Employees
export const getEmployees = () => fetchApi('/employees');
export const getEmployeeByUserId = async (userId) => {
    const employees = await getEmployees();
    return employees.find(e => e.user_id === userId);
};
export const createEmployee = (data) => fetchApi('/employees', {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify(data)
});
export const updateEmployee = (id, data) => fetchApi(`/employees/${id}`, {
    method: 'PUT',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify(data)
});
export const deleteEmployee = (id) => fetchApi(`/employees/${id}`, {
    method: 'DELETE'
});

// Services
export const getServices = () => fetchApi('/services');
export const getServicesByCompany = (companyId) => fetchApi(`/companies/${companyId}/services`);
export const createService = (data) => fetchApi('/services', {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify(data)
});
export const updateService = (id, data) => fetchApi(`/services/${id}`, {
    method: 'PUT',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify(data)
});
export const deleteService = (id) => fetchApi(`/services/${id}`, {
    method: 'DELETE'
});

// Orders
export const getOrders = () => fetchApi('/orders');
export const createOrder = (data) => fetchApi('/orders', {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify(data)
});
export const updateOrder = (id, data) => fetchApi(`/orders/${id}`, {
    method: 'PUT',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify(data)
});
export const getContractedServicesByCompany = async (companyId) => {
    const orders = await getOrders();
    return orders.filter(o => o.company_id === parseInt(companyId));
};
export const getContractedServicesByClient = async (clientId) => {
    const orders = await getOrders();
    return orders.filter(o => o.client_id === clientId);
};
export const getContractedServicesByEmployee = (employeeId) => fetchApi(`/employees/${employeeId}/services`);

// Evidences
export const getEvidencesByService = (serviceId) => fetchApi(`/services/${serviceId}/evidences`);
export const createEvidence = (data) => fetchApi('/evidences', {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify(data)
});

// Modules
export const getModules = () => fetchApi('/modules');
export const getModulesByCompany = (companyId) => fetchApi(`/companies/${companyId}/modules`);
export const toggleCompanyModule = (companyId, moduleId, isEnabled) => fetchApi(`/companies/${companyId}/modules/${moduleId}`, {
    method: 'PUT',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({ is_enabled: isEnabled })
});

// Config
export const getSystemConfig = () => fetchApi('/config');
export const getSystemConfigValue = async (key) => {
    try {
        const config = await fetchApi(`/config/${key}`);
        return config ? config.config_value : null;
    } catch (e) {
        return null;
    }
};
export const updateSystemConfig = (key, value) => fetchApi(`/config/${key}`, {
    method: 'PUT',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({ config_value: value })
});

// Access Validations
export const canClientAccess = async (clientId) => {
    try {
        const client = await getClientByUserId(clientId);
        if (!client) return false;
        if (client.allow_access_without_service) return true;

        const services = await getContractedServicesByClient(clientId);
        const now = new Date();

        return services.some(service => {
            if (service.status === 'PENDING' || service.status === 'IN_PROGRESS') return true;
            if (service.warranty_end_date) {
                const warrantyEnd = new Date(service.warranty_end_date);
                return warrantyEnd > now;
            }
            return false;
        });
    } catch (e) {
        return false;
    }
};
