/*
 KeQuiz Production Schema - Archivo Único para Producción
 =========================================================
 
 Este archivo contiene TODO lo necesario para crear la base de datos desde cero en producción.
 Incluye todas las migraciones realizadas:
 
 ✅ Migración de campos de nombre:
    - first_name, last_name, second_last_name (sin campo name antiguo)
    - Índices para búsqueda por nombre
 
 ✅ Restricciones únicas:
    - Email UNIQUE
    - Teléfono único parcial (permite NULL pero no duplicados)
    - CURP único parcial (permite NULL pero no duplicados)
 
 ✅ Tabla de especialidades:
    - Tabla specialties con relación a instituciones
    - Índices de optimización
 
 ✅ Campos de exámenes:
    - tolerance_minutes en exams
    - attempt_number, scheduled_at, scheduled_end_at, completed_at en exam_assignments
 
 ✅ Permisos completos:
    - Todos los permisos necesarios para el usuario 'iger'
    - Permisos en secuencias, tablas y funciones
 
 INSTRUCCIONES PARA PRODUCCIÓN:
 1. Ejecutar este archivo SQL completo en la base de datos de producción
 2. No es necesario ejecutar ningún otro archivo de migración
 3. Este archivo crea todo desde cero (DROP TABLE IF EXISTS antes de crear)
 
 IMPORTANTE: Este archivo está diseñado para crear la base de datos desde cero.
 Si ya existe una base de datos con datos, hacer backup antes de ejecutar.
*/

-- ----------------------------
-- Base Schema (from kequizInicial.sql)
-- ----------------------------

DROP TABLE IF EXISTS "public"."specialties" CASCADE;
DROP TABLE IF EXISTS "public"."institution_modules" CASCADE;
DROP TABLE IF EXISTS "public"."modules" CASCADE;
DROP TABLE IF EXISTS "public"."system_config" CASCADE;
DROP TABLE IF EXISTS "public"."notifications" CASCADE;
DROP TABLE IF EXISTS "public"."exam_responses" CASCADE;
DROP TABLE IF EXISTS "public"."exam_questions" CASCADE;
DROP TABLE IF EXISTS "public"."exam_logs" CASCADE;
DROP TABLE IF EXISTS "public"."exam_assignments" CASCADE;
DROP TABLE IF EXISTS "public"."exams" CASCADE;
DROP TABLE IF EXISTS "public"."question_bank" CASCADE;
DROP TABLE IF EXISTS "public"."users" CASCADE;
DROP TABLE IF EXISTS "public"."roles" CASCADE;
DROP TABLE IF EXISTS "public"."institutions" CASCADE;

-- Sequences
DROP SEQUENCE IF EXISTS "public"."institutions_id_institution_seq";
CREATE SEQUENCE "public"."institutions_id_institution_seq" INCREMENT 1 MINVALUE 1 MAXVALUE 2147483647 START 1 CACHE 1;

DROP SEQUENCE IF EXISTS "public"."roles_id_role_seq";
CREATE SEQUENCE "public"."roles_id_role_seq" INCREMENT 1 MINVALUE 1 MAXVALUE 2147483647 START 1 CACHE 1;

DROP SEQUENCE IF EXISTS "public"."users_id_seq";
CREATE SEQUENCE "public"."users_id_seq" INCREMENT 1 MINVALUE 1 MAXVALUE 2147483647 START 1 CACHE 1;

-- Institutions
CREATE TABLE "public"."institutions" (
  "id_institution" int4 NOT NULL DEFAULT nextval('institutions_id_institution_seq'::regclass),
  "name" varchar(255) NOT NULL,
  "rfc" varchar(20),
  "phone" varchar(50),
  "email" varchar(255),
  "fiscal_address" text,
  "admin_user_id" int4, -- ID del usuario administrador de la institución (opcional)
  "logo_url" varchar(500),
  "tagline" varchar(500),
  "exam_footer" text,
  "executive_contact" jsonb,
  "street" varchar(255),
  "exterior_number" varchar(50),
  "neighborhood" varchar(255),
  "municipality" varchar(255),
  "state" varchar(255),
  "postal_code" varchar(20),
  "max_teachers" int4 DEFAULT 0,
  "max_students" int4 DEFAULT 0,
  "max_exams" int4 DEFAULT 0,
  "is_active" bool DEFAULT true,
  "settings" jsonb,
  "created_at" timestamp(6) DEFAULT CURRENT_TIMESTAMP,
  "updated_at" timestamp(6) DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY ("id_institution")
);

-- Roles
CREATE TABLE "public"."roles" (
  "id_role" int4 NOT NULL DEFAULT nextval('roles_id_role_seq'::regclass),
  "code" varchar(50) NOT NULL UNIQUE,
  "name" varchar(100) NOT NULL,
  "description" text,
  "is_active" bool DEFAULT true,
  "created_at" timestamp(6) DEFAULT CURRENT_TIMESTAMP,
  "updated_at" timestamp(6) DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY ("id_role")
);

-- Users
CREATE TABLE "public"."users" (
  "id" int4 NOT NULL DEFAULT nextval('users_id_seq'::regclass),
  "first_name" varchar(255) NOT NULL, -- Nombre(s) - Obligatorio
  "last_name" varchar(255) NOT NULL, -- Primer Apellido - Obligatorio
  "second_last_name" varchar(255), -- Segundo Apellido - Opcional
  "email" varchar(255) NOT NULL UNIQUE,
  "password_hash" varchar(255) NOT NULL,
  "role" varchar(50) NOT NULL,
  "phone" varchar(50), -- Added - Teléfono único (constraint agregado después)
  "specialty" varchar(255), -- Added (for professors)
  "client_number" varchar(50), -- Added (for students)
  "curp" varchar(18), -- Added - CURP único (constraint agregado después)
  "institution_id" int4,
  "photo_url" varchar(500),
  "is_active" bool DEFAULT true,
  "created_at" timestamp(6) DEFAULT CURRENT_TIMESTAMP,
  "updated_at" timestamp(6) DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY ("id"),
  FOREIGN KEY ("institution_id") REFERENCES "public"."institutions" ("id_institution") ON DELETE SET NULL
);

-- ----------------------------
-- Academic Tables (KeQuiz Specific)
-- ----------------------------

-- Question Bank
CREATE TABLE "public"."question_bank" (
  "id_question" SERIAL PRIMARY KEY,
  "institution_id" int4 NOT NULL,
  "professor_id" int4 NOT NULL,
  "category" varchar(100),
  "type" varchar(50) NOT NULL, -- 'MULTIPLE_CHOICE', 'TRUE_FALSE', 'OPEN'
  "content" jsonb NOT NULL, -- { "question": "...", "options": [...], "media_url": "..." }
  "correct_answer" text NOT NULL,
  "is_active" bool DEFAULT true,
  "created_at" timestamp(6) DEFAULT CURRENT_TIMESTAMP,
  "updated_at" timestamp(6) DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY ("institution_id") REFERENCES "public"."institutions" ("id_institution") ON DELETE CASCADE,
  FOREIGN KEY ("professor_id") REFERENCES "public"."users" ("id") ON DELETE CASCADE
);

-- Exams
CREATE TABLE "public"."exams" (
  "id_exam" SERIAL PRIMARY KEY,
  "institution_id" int4 NOT NULL,
  "professor_id" int4 NOT NULL,
  "title" varchar(255) NOT NULL,
  "description" text,
  "settings" jsonb, -- { "timer_minutes": 60, "randomize_questions": true, "show_results_immediately": false }
  "tolerance_minutes" int4 DEFAULT 15, -- Tiempo de tolerancia en minutos después de la hora programada
  "is_active" bool DEFAULT true,
  "created_at" timestamp(6) DEFAULT CURRENT_TIMESTAMP,
  "updated_at" timestamp(6) DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY ("institution_id") REFERENCES "public"."institutions" ("id_institution") ON DELETE CASCADE,
  FOREIGN KEY ("professor_id") REFERENCES "public"."users" ("id") ON DELETE CASCADE
);

-- Exam Assignments
CREATE TABLE "public"."exam_assignments" (
  "id_assignment" SERIAL PRIMARY KEY,
  "exam_id" int4 NOT NULL,
  "student_id" int4 NOT NULL,
  "attempt_number" int4 DEFAULT 1, -- Número de intento para permitir reprogramaciones
  "status" varchar(50) DEFAULT 'PENDING', -- 'PENDING', 'IN_PROGRESS', 'COMPLETED', 'CANCELLED'
  "score" numeric(5,2),
  "scheduled_at" timestamp(6), -- Fecha y hora programada para inicio (puede ser NULL si no está programado)
  "scheduled_end_at" timestamp(6), -- Fecha y hora de fin del período (para rangos de fechas)
  "started_at" timestamp(6), -- Fecha y hora real de inicio
  "finished_at" timestamp(6), -- Fecha y hora de finalización (legacy, usar completed_at)
  "completed_at" timestamp(6), -- Fecha y hora de finalización
  "created_at" timestamp(6) DEFAULT CURRENT_TIMESTAMP,
  "updated_at" timestamp(6) DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY ("exam_id") REFERENCES "public"."exams" ("id_exam") ON DELETE CASCADE,
  FOREIGN KEY ("student_id") REFERENCES "public"."users" ("id") ON DELETE CASCADE
);

-- Exam Questions (Relación entre exámenes y preguntas)
CREATE TABLE "public"."exam_questions" (
  "exam_id" int4 NOT NULL,
  "question_id" int4 NOT NULL,
  "order_index" int4 NOT NULL DEFAULT 0, -- Orden de las preguntas en el examen
  PRIMARY KEY ("exam_id", "question_id"),
  FOREIGN KEY ("exam_id") REFERENCES "public"."exams" ("id_exam") ON DELETE CASCADE,
  FOREIGN KEY ("question_id") REFERENCES "public"."question_bank" ("id_question") ON DELETE CASCADE
);

-- Exam Responses (Respuestas de los estudiantes)
CREATE TABLE "public"."exam_responses" (
  "id_response" SERIAL PRIMARY KEY,
  "assignment_id" int4 NOT NULL,
  "question_id" int4 NOT NULL,
  "student_answer" text, -- Respuesta del estudiante (texto o índice de opción)
  "is_correct" bool, -- NULL para preguntas abiertas hasta que el profesor califique
  "points_earned" numeric(5,2) DEFAULT 0,
  "feedback" text, -- Comentarios del profesor
  "created_at" timestamp(6) DEFAULT CURRENT_TIMESTAMP,
  "updated_at" timestamp(6) DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY ("assignment_id") REFERENCES "public"."exam_assignments" ("id_assignment") ON DELETE CASCADE,
  FOREIGN KEY ("question_id") REFERENCES "public"."question_bank" ("id_question") ON DELETE CASCADE,
  UNIQUE ("assignment_id", "question_id") -- Una respuesta por pregunta por asignación
);

-- Exam Logs (Audit/Monitoring)
CREATE TABLE "public"."exam_logs" (
  "id_log" SERIAL PRIMARY KEY,
  "assignment_id" int4 NOT NULL,
  "event_type" varchar(50) NOT NULL, -- 'START', 'FINISH', 'FOCUS_LOST', 'TAB_CHANGE', 'HEARTBEAT'
  "event_data" jsonb,
  "timestamp" timestamp(6) DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY ("assignment_id") REFERENCES "public"."exam_assignments" ("id_assignment") ON DELETE CASCADE
);

-- ----------------------------
-- Initial Data
-- ----------------------------

-- Roles
INSERT INTO "public"."roles" ("code", "name", "description") VALUES 
('ROOT', 'Root', 'Superusuario del sistema'),
('ADMIN', 'Admin', 'Administrador de institución'),
('PROFESOR', 'Profesor', 'Profesor que crea exámenes'),
('ESTUDIANTE', 'Estudiante', 'Estudiante que toma exámenes');

-- Institution Sompo
INSERT INTO "public"."institutions" ("name", "rfc", "email", "max_teachers", "max_students", "max_exams") VALUES 
('Sompo Seguros', 'SOM123456789', 'contacto@sompo.com.mx', 50, 500, 100);

-- Users
-- Password hash for 'demo123'
INSERT INTO "public"."users" ("first_name", "last_name", "second_last_name", "email", "password_hash", "role", "institution_id") VALUES 
('Root', 'Usuario', NULL, 'hola@kemarketing.mx', '$2b$10$ca45dkK506z3XhHDHPUzz.hauQzmuiHVAhqrLCeK/hzzia/7i02Mq', 'ROOT', NULL),
('Victoria', 'Pérez', NULL, 'victoria.perez@sompo.com.mx', '$2b$10$Ee8eU3/G1SJfnpgaF0HyJuahzEZUxwCM7jy/Qhu9ndOQSNWTe0KeK', 'ADMIN', 1);

-- Indexes
-- Nota: CREATE INDEX IF NOT EXISTS solo está disponible desde PostgreSQL 9.5+
-- Si se ejecuta en versiones anteriores, estos índices pueden fallar si ya existen
CREATE INDEX IF NOT EXISTS "idx_users_email" ON "public"."users" ("email");
CREATE INDEX IF NOT EXISTS "idx_users_institution_id" ON "public"."users" ("institution_id");
CREATE INDEX IF NOT EXISTS "idx_users_first_name" ON "public"."users" ("first_name");
CREATE INDEX IF NOT EXISTS "idx_users_last_name" ON "public"."users" ("last_name");

-- Constraints únicos para phone y curp (permiten NULL pero no duplicados)
-- Estos son índices únicos parciales que solo aplican cuando el valor NO es NULL
-- Nota: CREATE UNIQUE INDEX IF NOT EXISTS puede no funcionar en versiones antiguas de PostgreSQL
-- Si falla, usar: DROP INDEX IF EXISTS antes de crear
DROP INDEX IF EXISTS "public"."users_phone_key";
CREATE UNIQUE INDEX "users_phone_key" ON "public"."users" ("phone") WHERE "phone" IS NOT NULL;
DROP INDEX IF EXISTS "public"."users_curp_key";
CREATE UNIQUE INDEX "users_curp_key" ON "public"."users" ("curp") WHERE "curp" IS NOT NULL;

-- Índices para optimización de consultas
CREATE INDEX IF NOT EXISTS "idx_question_bank_institution" ON "public"."question_bank" ("institution_id");
CREATE INDEX IF NOT EXISTS "idx_question_bank_professor" ON "public"."question_bank" ("professor_id");
CREATE INDEX IF NOT EXISTS "idx_question_bank_category" ON "public"."question_bank" ("category");
CREATE INDEX IF NOT EXISTS "idx_exams_institution" ON "public"."exams" ("institution_id");
CREATE INDEX IF NOT EXISTS "idx_exam_assignments_student" ON "public"."exam_assignments" ("student_id");
CREATE INDEX IF NOT EXISTS "idx_exam_assignments_exam" ON "public"."exam_assignments" ("exam_id");
CREATE INDEX IF NOT EXISTS "idx_exam_assignments_status" ON "public"."exam_assignments" ("status");
CREATE INDEX IF NOT EXISTS "idx_exam_assignments_exam_student_attempt" ON "public"."exam_assignments" ("exam_id", "student_id", "attempt_number" DESC);
CREATE INDEX IF NOT EXISTS "idx_exam_assignments_scheduled" ON "public"."exam_assignments" ("scheduled_at") WHERE "scheduled_at" IS NOT NULL;
CREATE INDEX IF NOT EXISTS "idx_exam_questions_exam" ON "public"."exam_questions" ("exam_id");
CREATE INDEX IF NOT EXISTS "idx_exam_questions_question" ON "public"."exam_questions" ("question_id");
CREATE INDEX IF NOT EXISTS "idx_exam_responses_assignment" ON "public"."exam_responses" ("assignment_id");
CREATE INDEX IF NOT EXISTS "idx_exam_responses_question" ON "public"."exam_responses" ("question_id");
CREATE INDEX IF NOT EXISTS "idx_exam_logs_assignment" ON "public"."exam_logs" ("assignment_id");

-- ----------------------------
-- System & UI Tables
-- ----------------------------

-- Notifications
CREATE TABLE "public"."notifications" (
  "id" SERIAL PRIMARY KEY,
  "user_id" int4 NOT NULL,
  "type" varchar(50) NOT NULL,
  "title" varchar(255) NOT NULL,
  "message" text,
  "is_read" bool DEFAULT false,
  "related_service_id" int4,
  "created_at" timestamp(6) DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY ("user_id") REFERENCES "public"."users" ("id") ON DELETE CASCADE
);

-- System Config
CREATE TABLE "public"."system_config" (
  "config_key" varchar(100) PRIMARY KEY,
  "config_value" text,
  "description" text,
  "updated_at" timestamp(6) DEFAULT CURRENT_TIMESTAMP
);

-- Modules
CREATE TABLE "public"."modules" (
  "id_module" SERIAL PRIMARY KEY,
  "code" varchar(50) NOT NULL UNIQUE,
  "name" varchar(100) NOT NULL,
  "description" text,
  "is_active" bool DEFAULT true
);

-- Institution Modules (Feature flags per institution)
CREATE TABLE "public"."institution_modules" (
  "institution_id" int4 NOT NULL,
  "module_id" int4 NOT NULL,
  "is_enabled" bool DEFAULT true,
  "settings" jsonb,
  PRIMARY KEY ("institution_id", "module_id"),
  FOREIGN KEY ("institution_id") REFERENCES "public"."institutions" ("id_institution") ON DELETE CASCADE,
  FOREIGN KEY ("module_id") REFERENCES "public"."modules" ("id_module") ON DELETE CASCADE
);

-- Specialties (Catálogo de especialidades)
CREATE TABLE "public"."specialties" (
  "id_specialty" SERIAL PRIMARY KEY,
  "institution_id" int4 NOT NULL,
  "name" varchar(255) NOT NULL,
  "description" text,
  "is_active" bool DEFAULT true,
  "created_at" timestamp(6) DEFAULT CURRENT_TIMESTAMP,
  "updated_at" timestamp(6) DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY ("institution_id") REFERENCES "public"."institutions" ("id_institution") ON DELETE CASCADE,
  UNIQUE ("institution_id", "name") -- Nombre único por institución
);

-- ----------------------------
-- Initial Data Extension
-- ----------------------------

-- Modules
INSERT INTO "public"."modules" ("code", "name", "description") VALUES 
('EXAM_BUILDER', 'Constructor de Exámenes', 'Permite a los profesores crear exámenes personalizados'),
('LIVE_MONITOR', 'Monitor en Vivo', 'Supervisión en tiempo real de los estudiantes durante el examen'),
('QUESTION_BANK', 'Banco de Preguntas', 'Gestión centralizada de preguntas por categoría'),
('REPORTING', 'Reportes Avanzados', 'Estadísticas detalladas de rendimiento por estudiante e institución');

-- System Config
INSERT INTO "public"."system_config" ("config_key", "config_value", "description") VALUES 
('SITE_NAME', 'KeQuiz', 'Nombre de la plataforma'),
('ALLOW_REGISTRATION', 'false', 'Permitir registro público de estudiantes'),
('MAINTENANCE_MODE', 'false', 'Activar modo mantenimiento');

CREATE INDEX IF NOT EXISTS "idx_notifications_user" ON "public"."notifications" ("user_id");
CREATE INDEX IF NOT EXISTS "idx_notifications_is_read" ON "public"."notifications" ("is_read");
CREATE INDEX IF NOT EXISTS "idx_specialties_institution" ON "public"."specialties" ("institution_id");
CREATE INDEX IF NOT EXISTS "idx_specialties_active" ON "public"."specialties" ("is_active");

-- ----------------------------
-- Permisos para Usuario iger
-- ----------------------------

-- Otorgar permisos en el esquema public
GRANT USAGE ON SCHEMA "public" TO "iger";

-- Otorgar permisos en todas las secuencias
GRANT USAGE, SELECT ON ALL SEQUENCES IN SCHEMA "public" TO "iger";
ALTER DEFAULT PRIVILEGES IN SCHEMA "public" GRANT USAGE, SELECT ON SEQUENCES TO "iger";

-- Otorgar permisos en todas las tablas
GRANT SELECT, INSERT, UPDATE, DELETE ON ALL TABLES IN SCHEMA "public" TO "iger";
ALTER DEFAULT PRIVILEGES IN SCHEMA "public" GRANT SELECT, INSERT, UPDATE, DELETE ON TABLES TO "iger";

-- Otorgar permisos en todas las funciones (si existen)
GRANT EXECUTE ON ALL FUNCTIONS IN SCHEMA "public" TO "iger";
ALTER DEFAULT PRIVILEGES IN SCHEMA "public" GRANT EXECUTE ON FUNCTIONS TO "iger";

-- Permisos específicos por tabla (para mayor claridad)
GRANT SELECT, INSERT, UPDATE, DELETE ON TABLE "public"."institutions" TO "iger";
GRANT SELECT, INSERT, UPDATE, DELETE ON TABLE "public"."roles" TO "iger";
GRANT SELECT, INSERT, UPDATE, DELETE ON TABLE "public"."users" TO "iger";
GRANT SELECT, INSERT, UPDATE, DELETE ON TABLE "public"."question_bank" TO "iger";
GRANT SELECT, INSERT, UPDATE, DELETE ON TABLE "public"."exams" TO "iger";
GRANT SELECT, INSERT, UPDATE, DELETE ON TABLE "public"."exam_assignments" TO "iger";
GRANT SELECT, INSERT, UPDATE, DELETE ON TABLE "public"."exam_questions" TO "iger";
GRANT SELECT, INSERT, UPDATE, DELETE ON TABLE "public"."exam_responses" TO "iger";
GRANT SELECT, INSERT, UPDATE, DELETE ON TABLE "public"."exam_logs" TO "iger";
GRANT SELECT, INSERT, UPDATE, DELETE ON TABLE "public"."notifications" TO "iger";
GRANT SELECT, INSERT, UPDATE, DELETE ON TABLE "public"."system_config" TO "iger";
GRANT SELECT, INSERT, UPDATE, DELETE ON TABLE "public"."modules" TO "iger";
GRANT SELECT, INSERT, UPDATE, DELETE ON TABLE "public"."institution_modules" TO "iger";
GRANT SELECT, INSERT, UPDATE, DELETE ON TABLE "public"."specialties" TO "iger";

-- Permisos en secuencias específicas (explícitas y automáticas)
GRANT USAGE, SELECT ON SEQUENCE "public"."institutions_id_institution_seq" TO "iger";
GRANT USAGE, SELECT ON SEQUENCE "public"."roles_id_role_seq" TO "iger";
GRANT USAGE, SELECT ON SEQUENCE "public"."users_id_seq" TO "iger";
-- Secuencias creadas automáticamente por SERIAL (cubiertas por el comando general, pero listadas para referencia):
-- question_bank_id_question_seq, exams_id_exam_seq, exam_assignments_id_assignment_seq,
-- exam_responses_id_response_seq, exam_logs_id_log_seq, notifications_id_seq,
-- modules_id_module_seq, specialties_id_specialty_seq

-- Permisos para crear tablas y funciones (opcional, si se necesita)
-- GRANT CREATE ON SCHEMA "public" TO "iger";

