#!/bin/bash

# Script de Restauración de Base de Datos - KeQuiz
# 
# Este script restaura una base de datos PostgreSQL desde un archivo de backup
# 
# Uso: ./scripts/restore_database.sh [archivo_backup.sql]
# O: bash scripts/restore_database.sh [archivo_backup.sql]
# 
# Si no se especifica archivo, mostrará los backups disponibles

# Configuración (ajustar según tu entorno)
DB_HOST="${DB_HOST:-127.0.0.1}"
DB_PORT="${DB_PORT:-5432}"
DB_NAME="${DB_NAME:-kequiz}"
DB_USER="${DB_USER:-iger}"
BACKUP_DIR="${BACKUP_DIR:-/var/www/html/kequiz}"

# Colores para output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# Función para mostrar ayuda
show_help() {
    echo "Uso: $0 [opciones] [archivo_backup]"
    echo ""
    echo "Opciones:"
    echo "  -f, --file FILE    Archivo de backup a restaurar"
    echo "  -l, --list         Listar backups disponibles"
    echo "  -h, --help         Mostrar esta ayuda"
    echo ""
    echo "Ejemplos:"
    echo "  $0 backup_20250109_223000.sql"
    echo "  $0 -f /var/www/html/kequiz/backup.sql"
    echo "  $0 --list"
}

# Función para listar backups disponibles
list_backups() {
    echo "Backups disponibles en $BACKUP_DIR:"
    echo ""
    
    if [ ! -d "$BACKUP_DIR" ]; then
        echo -e "${RED}✗ Directorio de backup no existe: $BACKUP_DIR${NC}"
        return 1
    fi
    
    local backups=$(find "$BACKUP_DIR" -name "backup_*.sql" -o -name "backup_*.sql.gz" 2>/dev/null | sort -r)
    
    if [ -z "$backups" ]; then
        echo -e "${YELLOW}⚠ No se encontraron backups en $BACKUP_DIR${NC}"
        return 1
    fi
    
    local count=1
    echo "$backups" | while read -r backup; do
        local size=$(du -h "$backup" | cut -f1)
        local date=$(stat -f "%Sm" -t "%Y-%m-%d %H:%M:%S" "$backup" 2>/dev/null || stat -c "%y" "$backup" 2>/dev/null | cut -d' ' -f1,2)
        echo "  $count. $(basename "$backup") - $size - $date"
        count=$((count + 1))
    done
}

# Función para descomprimir si es necesario
decompress_if_needed() {
    local file="$1"
    local decompressed_file="${file%.gz}"
    
    if [[ "$file" == *.gz ]]; then
        echo -e "${YELLOW}Descomprimiendo backup...${NC}"
        if gunzip -c "$file" > "$decompressed_file"; then
            echo -e "${GREEN}✓ Backup descomprimido: $decompressed_file${NC}"
            echo "$decompressed_file"
        else
            echo -e "${RED}✗ Error al descomprimir el backup${NC}"
            return 1
        fi
    else
        echo "$file"
    fi
}

# Función para restaurar
restore_database() {
    local backup_file="$1"
    
    if [ ! -f "$backup_file" ]; then
        echo -e "${RED}✗ Archivo de backup no encontrado: $backup_file${NC}"
        return 1
    fi
    
    # Verificar si el archivo está comprimido
    local sql_file
    if [[ "$backup_file" == *.gz ]]; then
        sql_file=$(decompress_if_needed "$backup_file")
        if [ $? -ne 0 ]; then
            return 1
        fi
        local temp_file=true
    else
        sql_file="$backup_file"
        local temp_file=false
    fi
    
    echo ""
    echo -e "${YELLOW}⚠ ADVERTENCIA: Esta operación eliminará todos los datos actuales de la base de datos${NC}"
    echo -e "${YELLOW}Base de datos: $DB_NAME${NC}"
    echo -e "${YELLOW}Archivo: $backup_file${NC}"
    echo ""
    read -p "¿Estás seguro de que deseas continuar? (escribe 'SI' para confirmar): " confirm
    
    if [ "$confirm" != "SI" ]; then
        echo -e "${YELLOW}Operación cancelada${NC}"
        if [ "$temp_file" = true ]; then
            rm -f "$sql_file"
        fi
        return 1
    fi
    
    echo ""
    echo "Restaurando base de datos..."
    echo "Esto puede tardar varios minutos dependiendo del tamaño del backup..."
    
    # Crear base de datos si no existe (solo estructura)
    echo "Verificando/Creando base de datos..."
    psql -h "$DB_HOST" -p "$DB_PORT" -U "$DB_USER" -d postgres -c "SELECT 1 FROM pg_database WHERE datname='$DB_NAME'" | grep -q 1 || \
    psql -h "$DB_HOST" -p "$DB_PORT" -U "$DB_USER" -d postgres -c "CREATE DATABASE $DB_NAME"
    
    if [ $? -ne 0 ]; then
        echo -e "${RED}✗ Error al crear/verificar la base de datos${NC}"
        if [ "$temp_file" = true ]; then
            rm -f "$sql_file"
        fi
        return 1
    fi
    
    # Restaurar desde el archivo SQL
    if psql -h "$DB_HOST" -p "$DB_PORT" -U "$DB_USER" -d "$DB_NAME" -f "$sql_file" > /dev/null 2>&1; then
        echo -e "${GREEN}✓ Base de datos restaurada exitosamente${NC}"
        
        # Limpiar archivo temporal si se descomprimió
        if [ "$temp_file" = true ]; then
            rm -f "$sql_file"
            echo -e "${GREEN}✓ Archivo temporal eliminado${NC}"
        fi
        
        # Verificar que la restauración fue exitosa
        echo ""
        echo "Verificando restauración..."
        local table_count=$(psql -h "$DB_HOST" -p "$DB_PORT" -U "$DB_USER" -d "$DB_NAME" -t -c "SELECT COUNT(*) FROM information_schema.tables WHERE table_schema='public' AND table_type='BASE TABLE';" 2>/dev/null | tr -d ' ')
        
        if [ -n "$table_count" ] && [ "$table_count" -gt 0 ]; then
            echo -e "${GREEN}✓ Restauración verificada: $table_count tablas encontradas${NC}"
        else
            echo -e "${YELLOW}⚠ No se pudieron verificar las tablas (puede ser normal si la BD estaba vacía)${NC}"
        fi
        
        return 0
    else
        echo -e "${RED}✗ Error al restaurar la base de datos${NC}"
        echo "Revisa los logs anteriores para más detalles"
        
        if [ "$temp_file" = true ]; then
            rm -f "$sql_file"
        fi
        return 1
    fi
}

# Procesar argumentos
BACKUP_FILE=""

while [[ $# -gt 0 ]]; do
    case $1 in
        -f|--file)
            BACKUP_FILE="$2"
            shift 2
            ;;
        -l|--list)
            list_backups
            exit 0
            ;;
        -h|--help)
            show_help
            exit 0
            ;;
        *.sql|*.sql.gz)
            BACKUP_FILE="$1"
            shift
            ;;
        *)
            echo -e "${RED}Opción desconocida: $1${NC}"
            show_help
            exit 1
            ;;
    esac
done

# Si no se especificó archivo, listar y pedir selección
if [ -z "$BACKUP_FILE" ]; then
    echo "No se especificó archivo de backup."
    echo ""
    list_backups
    echo ""
    read -p "Ingresa el nombre del archivo de backup a restaurar: " BACKUP_FILE
    
    if [ -z "$BACKUP_FILE" ]; then
        echo -e "${RED}✗ No se especificó archivo${NC}"
        exit 1
    fi
    
    # Si no es ruta absoluta, buscar en el directorio de backups
    if [[ "$BACKUP_FILE" != /* ]]; then
        BACKUP_FILE="$BACKUP_DIR/$BACKUP_FILE"
    fi
fi

# Verificar que el archivo existe
if [ ! -f "$BACKUP_FILE" ]; then
    echo -e "${RED}✗ Archivo no encontrado: $BACKUP_FILE${NC}"
    echo ""
    echo "Backups disponibles:"
    list_backups
    exit 1
fi

# Ejecutar restauración
restore_database "$BACKUP_FILE"

exit $?

