import { UserModel } from '../src/models/user.js';
import pool from '../src/models/database.js';

async function testLegacyUserRead() {
    const testEmail = `legacy_${Date.now()}@example.com`;
    console.log(`Testing legacy user read for: ${testEmail}`);

    try {
        // 1. Manually insert a user with ONLY 'name' (simulating legacy data)
        // We bypass UserModel.create to force a "legacy-like" insertion if columns allow, 
        // or just rely on UserModel.create which now concatenates to name.
        // Since we know the schema only has 'name', UserModel.create will do exactly that.

        const newUser = {
            email: testEmail,
            password_hash: 'hash',
            role: 'ADMIN',
            first_name: 'Juan',
            last_name: 'Perez',
            second_last_name: 'Lopez',
            phone: '555' + Date.now().toString().slice(-7), // Unique phone
            is_active: true
        };

        const createdUser = await UserModel.create(newUser);
        console.log('User created (simulating legacy storage):', createdUser.id);
        console.log(' stored name:', createdUser.name);

        // 2. Fetch using UserModel.getById and check if fields are populated
        const fetchedUser = await UserModel.getById(createdUser.id);

        console.log('\nFetched User Data:');
        console.log(' id:', fetchedUser.id);
        console.log(' name (full):', fetchedUser.name);
        console.log(' first_name:', fetchedUser.first_name);
        console.log(' last_name:', fetchedUser.last_name);
        console.log(' second_last_name:', fetchedUser.second_last_name);

        let success = true;
        if (fetchedUser.first_name !== 'Juan') {
            console.error('FAIL: first_name mismatch. Expected Juan, got', fetchedUser.first_name);
            success = false;
        }
        if (fetchedUser.last_name !== 'Perez') {
            console.error('FAIL: last_name mismatch. Expected Perez, got', fetchedUser.last_name);
            success = false;
        }
        if (fetchedUser.second_last_name !== 'Lopez') {
            console.error('FAIL: second_last_name mismatch. Expected Lopez, got', fetchedUser.second_last_name);
            success = false;
        }

        if (success) {
            console.log('\nSUCCESS: Legacy name parsed correctly!');
        } else {
            console.log('\nFAILED: Name parsing incorrect.');
        }

        // Cleanup
        await UserModel.delete(createdUser.id);
        console.log('Test user deleted.');

    } catch (error) {
        console.error('Test FAILED with error:', error);
    } finally {
        pool.end();
    }
}

testLegacyUserRead();
