#!/usr/bin/env node

/*
 * Script de Verificación de Base de Datos - KeQuiz
 * 
 * Este script valida el estado de la base de datos y genera un reporte
 * de inconsistencias, tablas faltantes, columnas faltantes, etc.
 * 
 * Uso: node scripts/verify_database.js
 */

import { query } from '../src/models/database.js';
import fs from 'fs';
import path from 'path';

const REPORT_DIR = path.join(process.cwd(), 'logs');
const REPORT_FILE = path.join(REPORT_DIR, `database_verification_${new Date().toISOString().replace(/[:.]/g, '-')}.txt`);

// Estructura esperada de la base de datos
const EXPECTED_STRUCTURE = {
    tables: {
        'institutions': {
            columns: ['id_institution', 'name', 'rfc', 'phone', 'email', 'fiscal_address', 'logo_url', 
                     'tagline', 'exam_footer', 'executive_contact', 'street', 'exterior_number', 
                     'neighborhood', 'municipality', 'state', 'postal_code', 'max_teachers', 'max_students', 
                     'max_exams', 'is_active', 'settings', 'created_at', 'updated_at'],
            alternative: 'companies', // Tabla alternativa
            alternativeColumns: ['id', 'name', 'rfc', 'phone', 'email', 'fiscal_address', 'logo_url',
                                'tagline', 'exam_footer', 'executive_contact', 'street', 'exterior_number',
                                'neighborhood', 'municipality', 'state', 'postal_code', 'max_employees', 
                                'max_clients', 'max_exams', 'is_active', 'settings', 'created_at', 'updated_at']
        },
        'users': {
            columns: ['id', 'first_name', 'last_name', 'second_last_name', 'email', 'password_hash', 
                     'role', 'phone', 'specialty', 'client_number', 'curp', 'institution_id', 
                     'photo_url', 'is_active', 'created_at', 'updated_at'],
            alternativeColumns: ['id', 'name', 'email', 'password_hash', 'role', 'phone', 'specialty', 
                                'client_number', 'curp', 'company_id', 'photo_url', 'is_active', 
                                'created_at', 'updated_at']
        },
        'roles': {
            columns: ['id_role', 'code', 'name', 'description', 'is_active', 'created_at', 'updated_at']
        },
        'question_bank': {
            columns: ['id_question', 'institution_id', 'professor_id', 'category', 'type', 'content', 
                     'correct_answer', 'is_active', 'created_at', 'updated_at']
        },
        'exams': {
            columns: ['id_exam', 'institution_id', 'professor_id', 'title', 'description', 'settings', 
                     'tolerance_minutes', 'is_active', 'created_at', 'updated_at']
        },
        'exam_assignments': {
            columns: ['id_assignment', 'exam_id', 'student_id', 'attempt_number', 'status', 'score', 
                     'scheduled_at', 'scheduled_end_at', 'started_at', 'finished_at', 'completed_at', 
                     'created_at', 'updated_at']
        },
        'exam_questions': {
            columns: ['exam_id', 'question_id', 'order_index']
        },
        'exam_responses': {
            columns: ['id_response', 'assignment_id', 'question_id', 'student_answer', 'is_correct', 
                     'points_earned', 'feedback', 'created_at', 'updated_at']
        },
        'exam_logs': {
            columns: ['id_log', 'assignment_id', 'event_type', 'event_data', 'timestamp']
        },
        'notifications': {
            columns: ['id', 'user_id', 'type', 'title', 'message', 'is_read', 'related_service_id', 'created_at']
        },
        'system_config': {
            columns: ['config_key', 'config_value', 'description', 'updated_at']
        },
        'modules': {
            columns: ['id_module', 'code', 'name', 'description', 'is_active']
        },
        'institution_modules': {
            columns: ['institution_id', 'module_id', 'is_enabled', 'settings']
        },
        'specialties': {
            columns: ['id_specialty', 'institution_id', 'name', 'description', 'is_active', 'created_at', 'updated_at']
        }
    },
    indexes: {
        'users': ['idx_users_email', 'idx_users_institution_id', 'idx_users_first_name', 'idx_users_last_name', 
                 'users_phone_key', 'users_curp_key'],
        'question_bank': ['idx_question_bank_institution', 'idx_question_bank_professor', 'idx_question_bank_category'],
        'exams': ['idx_exams_institution'],
        'exam_assignments': ['idx_exam_assignments_student', 'idx_exam_assignments_exam', 'idx_exam_assignments_status',
                            'idx_exam_assignments_exam_student_attempt', 'idx_exam_assignments_scheduled'],
        'specialties': ['idx_specialties_institution', 'idx_specialties_active']
    },
    constraints: {
        'users': {
            unique: ['email'],
            uniquePartial: ['phone', 'curp'] // Índices únicos parciales
        }
    }
};

class DatabaseVerifier {
    constructor() {
        this.issues = [];
        this.warnings = [];
        this.info = [];
    }

    log(level, message, details = null) {
        const entry = {
            timestamp: new Date().toISOString(),
            level,
            message,
            details
        };
        
        if (level === 'ERROR') {
            this.issues.push(entry);
        } else if (level === 'WARNING') {
            this.warnings.push(entry);
        } else {
            this.info.push(entry);
        }
        
        const prefix = level === 'ERROR' ? '❌' : level === 'WARNING' ? '⚠️' : 'ℹ️';
        console.log(`${prefix} [${level}] ${message}`);
        if (details) {
            console.log(`   Detalles:`, details);
        }
    }

    async verifyTables() {
        this.log('INFO', 'Verificando tablas...');
        
        try {
            const res = await query(`
                SELECT table_name 
                FROM information_schema.tables 
                WHERE table_schema = 'public' 
                AND table_type = 'BASE TABLE'
                ORDER BY table_name
            `);
            
            const existingTables = res.rows.map(row => row.table_name);
            this.log('INFO', `Tablas encontradas: ${existingTables.length}`, existingTables);
            
            // Verificar tablas esperadas
            for (const [expectedTable, config] of Object.entries(EXPECTED_STRUCTURE.tables)) {
                if (!existingTables.includes(expectedTable)) {
                    // Verificar si existe tabla alternativa
                    if (config.alternative && existingTables.includes(config.alternative)) {
                        this.log('WARNING', `Tabla '${expectedTable}' no existe, pero se encontró '${config.alternative}' (tabla alternativa)`, {
                            expected: expectedTable,
                            found: config.alternative
                        });
                    } else {
                        this.log('ERROR', `Tabla esperada '${expectedTable}' no existe`, {
                            expected: expectedTable,
                            alternative: config.alternative || 'N/A'
                        });
                    }
                } else {
                    this.log('INFO', `✓ Tabla '${expectedTable}' existe`);
                }
            }
            
            // Verificar tablas inesperadas
            const expectedTableNames = Object.keys(EXPECTED_STRUCTURE.tables);
            const alternativeTables = Object.values(EXPECTED_STRUCTURE.tables)
                .map(t => t.alternative)
                .filter(Boolean);
            const allExpected = [...expectedTableNames, ...alternativeTables];
            
            for (const table of existingTables) {
                if (!allExpected.includes(table)) {
                    this.log('WARNING', `Tabla '${table}' existe pero no está en la estructura esperada`, {
                        table
                    });
                }
            }
            
            return existingTables;
        } catch (error) {
            this.log('ERROR', `Error al verificar tablas: ${error.message}`, error);
            return [];
        }
    }

    async verifyColumns(tableName, expectedColumns, alternativeColumns = null) {
        try {
            const res = await query(`
                SELECT column_name, data_type, is_nullable, column_default
                FROM information_schema.columns 
                WHERE table_schema = 'public' 
                AND table_name = $1
                ORDER BY ordinal_position
            `, [tableName]);
            
            if (res.rows.length === 0) {
                // Verificar tabla alternativa
                if (alternativeColumns) {
                    const altTable = tableName === 'institutions' ? 'companies' : null;
                    if (altTable) {
                        return await this.verifyColumns(altTable, alternativeColumns);
                    }
                }
                this.log('ERROR', `No se encontraron columnas para la tabla '${tableName}'`);
                return false;
            }
            
            const existingColumns = res.rows.map(row => row.column_name);
            const missingColumns = expectedColumns.filter(col => !existingColumns.includes(col));
            const extraColumns = existingColumns.filter(col => !expectedColumns.includes(col) && 
                !(alternativeColumns && alternativeColumns.includes(col)));
            
            if (missingColumns.length > 0) {
                // Si hay columnas alternativas, verificar si existen
                if (alternativeColumns) {
                    const foundAlternatives = missingColumns.filter(col => {
                        const altCol = this.getAlternativeColumn(col);
                        return altCol && existingColumns.includes(altCol);
                    });
                    
                    if (foundAlternatives.length > 0) {
                        this.log('WARNING', `Tabla '${tableName}': Columnas esperadas no encontradas, pero existen alternativas`, {
                            missing: missingColumns,
                            foundAlternatives: foundAlternatives.map(c => this.getAlternativeColumn(c))
                        });
                    } else {
                        this.log('ERROR', `Tabla '${tableName}': Columnas faltantes`, {
                            missing: missingColumns,
                            existing: existingColumns
                        });
                    }
                } else {
                    this.log('ERROR', `Tabla '${tableName}': Columnas faltantes`, {
                        missing: missingColumns,
                        existing: existingColumns
                    });
                }
            }
            
            if (extraColumns.length > 0) {
                this.log('WARNING', `Tabla '${tableName}': Columnas adicionales encontradas`, {
                    extra: extraColumns
                });
            }
            
            if (missingColumns.length === 0 && extraColumns.length === 0) {
                this.log('INFO', `✓ Tabla '${tableName}': Todas las columnas esperadas están presentes`);
            }
            
            return true;
        } catch (error) {
            this.log('ERROR', `Error al verificar columnas de '${tableName}': ${error.message}`, error);
            return false;
        }
    }

    getAlternativeColumn(columnName) {
        const mapping = {
            'id_institution': 'id',
            'institution_id': 'company_id',
            'max_students': 'max_clients',
            'max_teachers': 'max_employees',
            'first_name': 'name',
            'last_name': 'name',
            'second_last_name': 'name'
        };
        return mapping[columnName] || null;
    }

    async verifyIndexes(tableName, expectedIndexes) {
        try {
            const res = await query(`
                SELECT indexname, indexdef
                FROM pg_indexes 
                WHERE schemaname = 'public' 
                AND tablename = $1
            `, [tableName]);
            
            const existingIndexes = res.rows.map(row => row.indexname);
            const missingIndexes = expectedIndexes.filter(idx => !existingIndexes.includes(idx));
            
            if (missingIndexes.length > 0) {
                this.log('WARNING', `Tabla '${tableName}': Índices faltantes`, {
                    missing: missingIndexes,
                    existing: existingIndexes
                });
            } else {
                this.log('INFO', `✓ Tabla '${tableName}': Todos los índices esperados están presentes`);
            }
            
            return existingIndexes;
        } catch (error) {
            this.log('ERROR', `Error al verificar índices de '${tableName}': ${error.message}`, error);
            return [];
        }
    }

    async verifyConstraints(tableName, expectedConstraints) {
        try {
            // Verificar constraints únicos
            if (expectedConstraints.unique) {
                for (const column of expectedConstraints.unique) {
                    const res = await query(`
                        SELECT conname, contype
                        FROM pg_constraint 
                        WHERE conrelid = (
                            SELECT oid FROM pg_class WHERE relname = $1
                        )
                        AND contype = 'u'
                    `, [tableName]);
                    
                    const uniqueConstraints = res.rows.map(row => row.conname);
                    const columnConstraint = uniqueConstraints.find(c => c.includes(column));
                    
                    if (!columnConstraint) {
                        this.log('WARNING', `Tabla '${tableName}': Constraint único faltante para columna '${column}'`);
                    } else {
                        this.log('INFO', `✓ Tabla '${tableName}': Constraint único presente para '${column}'`);
                    }
                }
            }
            
            // Verificar índices únicos parciales
            if (expectedConstraints.uniquePartial) {
                for (const column of expectedConstraints.uniquePartial) {
                    const res = await query(`
                        SELECT indexname
                        FROM pg_indexes 
                        WHERE schemaname = 'public' 
                        AND tablename = $1
                        AND indexname LIKE $2
                    `, [tableName, `%${column}%`]);
                    
                    if (res.rows.length === 0) {
                        this.log('WARNING', `Tabla '${tableName}': Índice único parcial faltante para columna '${column}'`);
                    } else {
                        this.log('INFO', `✓ Tabla '${tableName}': Índice único parcial presente para '${column}'`);
                    }
                }
            }
        } catch (error) {
            this.log('ERROR', `Error al verificar constraints de '${tableName}': ${error.message}`, error);
        }
    }

    async verifyForeignKeys() {
        this.log('INFO', 'Verificando foreign keys...');
        
        try {
            const res = await query(`
                SELECT
                    tc.table_name, 
                    kcu.column_name, 
                    ccu.table_name AS foreign_table_name,
                    ccu.column_name AS foreign_column_name,
                    tc.constraint_name
                FROM information_schema.table_constraints AS tc 
                JOIN information_schema.key_column_usage AS kcu
                  ON tc.constraint_name = kcu.constraint_name
                JOIN information_schema.constraint_column_usage AS ccu
                  ON ccu.constraint_name = tc.constraint_name
                WHERE tc.constraint_type = 'FOREIGN KEY'
                AND tc.table_schema = 'public'
                ORDER BY tc.table_name, kcu.column_name
            `);
            
            this.log('INFO', `Foreign keys encontradas: ${res.rows.length}`);
            
            // Verificar foreign keys esperadas
            const expectedFKs = [
                { table: 'users', column: 'institution_id', refTable: 'institutions', refColumn: 'id_institution' },
                { table: 'question_bank', column: 'institution_id', refTable: 'institutions', refColumn: 'id_institution' },
                { table: 'question_bank', column: 'professor_id', refTable: 'users', refColumn: 'id' },
                { table: 'exams', column: 'institution_id', refTable: 'institutions', refColumn: 'id_institution' },
                { table: 'exams', column: 'professor_id', refTable: 'users', refColumn: 'id' }
            ];
            
            for (const expectedFK of expectedFKs) {
                const found = res.rows.find(fk => 
                    fk.table_name === expectedFK.table && 
                    fk.column_name === expectedFK.column &&
                    (fk.foreign_table_name === expectedFK.refTable || 
                     (expectedFK.refTable === 'institutions' && fk.foreign_table_name === 'companies'))
                );
                
                if (!found) {
                    this.log('WARNING', `Foreign key faltante: ${expectedFK.table}.${expectedFK.column} -> ${expectedFK.refTable}.${expectedFK.refColumn}`);
                } else {
                    this.log('INFO', `✓ Foreign key presente: ${expectedFK.table}.${expectedFK.column} -> ${found.foreign_table_name}.${found.foreign_column_name}`);
                }
            }
        } catch (error) {
            this.log('ERROR', `Error al verificar foreign keys: ${error.message}`, error);
        }
    }

    async verifyDataIntegrity() {
        this.log('INFO', 'Verificando integridad de datos...');
        
        try {
            // Verificar usuarios sin institución válida
            const usersCheck = await query(`
                SELECT COUNT(*) as count
                FROM users u
                WHERE u.institution_id IS NOT NULL 
                AND NOT EXISTS (
                    SELECT 1 FROM institutions i WHERE i.id_institution = u.institution_id
                )
                AND NOT EXISTS (
                    SELECT 1 FROM companies c WHERE c.id = u.institution_id
                )
            `);
            
            if (parseInt(usersCheck.rows[0].count) > 0) {
                this.log('WARNING', `Usuarios con institution_id inválido: ${usersCheck.rows[0].count}`);
            } else {
                this.log('INFO', '✓ Todos los usuarios tienen institution_id válido');
            }
            
            // Verificar emails duplicados
            const duplicateEmails = await query(`
                SELECT email, COUNT(*) as count
                FROM users
                WHERE email IS NOT NULL
                GROUP BY email
                HAVING COUNT(*) > 1
            `);
            
            if (duplicateEmails.rows.length > 0) {
                this.log('ERROR', `Emails duplicados encontrados: ${duplicateEmails.rows.length}`, {
                    duplicates: duplicateEmails.rows
                });
            } else {
                this.log('INFO', '✓ No hay emails duplicados');
            }
            
        } catch (error) {
            this.log('ERROR', `Error al verificar integridad de datos: ${error.message}`, error);
        }
    }

    async generateReport() {
        // Crear directorio de logs si no existe
        if (!fs.existsSync(REPORT_DIR)) {
            fs.mkdirSync(REPORT_DIR, { recursive: true });
        }
        
        const report = [];
        report.push('='.repeat(80));
        report.push('REPORTE DE VERIFICACIÓN DE BASE DE DATOS - KeQuiz');
        report.push('='.repeat(80));
        report.push(`Fecha: ${new Date().toLocaleString('es-MX')}`);
        report.push('');
        
        report.push('RESUMEN:');
        report.push(`  ❌ Errores: ${this.issues.length}`);
        report.push(`  ⚠️  Advertencias: ${this.warnings.length}`);
        report.push(`  ℹ️  Información: ${this.info.length}`);
        report.push('');
        
        if (this.issues.length > 0) {
            report.push('='.repeat(80));
            report.push('ERRORES ENCONTRADOS:');
            report.push('='.repeat(80));
            this.issues.forEach((issue, index) => {
                report.push(`${index + 1}. [${issue.timestamp}] ${issue.message}`);
                if (issue.details) {
                    report.push(`   Detalles: ${JSON.stringify(issue.details, null, 2)}`);
                }
                report.push('');
            });
        }
        
        if (this.warnings.length > 0) {
            report.push('='.repeat(80));
            report.push('ADVERTENCIAS:');
            report.push('='.repeat(80));
            this.warnings.forEach((warning, index) => {
                report.push(`${index + 1}. [${warning.timestamp}] ${warning.message}`);
                if (warning.details) {
                    report.push(`   Detalles: ${JSON.stringify(warning.details, null, 2)}`);
                }
                report.push('');
            });
        }
        
        report.push('='.repeat(80));
        report.push('FIN DEL REPORTE');
        report.push('='.repeat(80));
        
        const reportContent = report.join('\n');
        fs.writeFileSync(REPORT_FILE, reportContent, 'utf8');
        
        console.log('\n' + '='.repeat(80));
        console.log(`Reporte guardado en: ${REPORT_FILE}`);
        console.log('='.repeat(80));
        
        return REPORT_FILE;
    }

    async run() {
        console.log('Iniciando verificación de base de datos...\n');
        
        try {
            // Verificar tablas
            const existingTables = await this.verifyTables();
            
            // Verificar columnas de cada tabla
            for (const [tableName, config] of Object.entries(EXPECTED_STRUCTURE.tables)) {
                if (existingTables.includes(tableName) || 
                    (config.alternative && existingTables.includes(config.alternative))) {
                    const actualTable = existingTables.includes(tableName) ? tableName : config.alternative;
                    await this.verifyColumns(actualTable, config.columns, config.alternativeColumns);
                }
            }
            
            // Verificar índices
            for (const [tableName, indexes] of Object.entries(EXPECTED_STRUCTURE.indexes)) {
                if (existingTables.includes(tableName)) {
                    await this.verifyIndexes(tableName, indexes);
                }
            }
            
            // Verificar constraints
            for (const [tableName, constraints] of Object.entries(EXPECTED_STRUCTURE.constraints)) {
                if (existingTables.includes(tableName)) {
                    await this.verifyConstraints(tableName, constraints);
                }
            }
            
            // Verificar foreign keys
            await this.verifyForeignKeys();
            
            // Verificar integridad de datos
            await this.verifyDataIntegrity();
            
            // Generar reporte
            const reportFile = await this.generateReport();
            
            console.log('\n' + '='.repeat(80));
            console.log('VERIFICACIÓN COMPLETADA');
            console.log('='.repeat(80));
            console.log(`Errores: ${this.issues.length}`);
            console.log(`Advertencias: ${this.warnings.length}`);
            console.log(`Información: ${this.info.length}`);
            console.log(`\nReporte completo guardado en: ${reportFile}`);
            
            // Código de salida
            process.exit(this.issues.length > 0 ? 1 : 0);
            
        } catch (error) {
            console.error('Error fatal durante la verificación:', error);
            process.exit(1);
        }
    }
}

// Ejecutar verificación
const verifier = new DatabaseVerifier();
verifier.run();

