import 'dotenv/config';
import express from 'express';
import cors from 'cors';
import { createServer } from 'http';
import { Server } from 'socket.io';
import jwt from 'jsonwebtoken';
import fs from 'fs';
import path from 'path';

// Routes
import authRoutes from './routes/auth.js';
import userRoutes from './routes/users.js';
import institutionRoutes from './routes/institutions.js';
import examRoutes from './routes/exams.js';
import questionRoutes from './routes/questions.js';
import notificationRoutes from './routes/notifications.js';
import configRoutes from './routes/config.js';
import rootRoutes from './routes/root.js';
import moduleRoutes from './routes/modules.js';
import specialtyRoutes from './routes/specialties.js';

const app = express();
const httpServer = createServer(app);
const io = new Server(httpServer, {
    cors: {
        origin: ["https://kequiz.kemarketing.mx", "http://kequiz.kemarketing.mx", "http://localhost:8080", "http://localhost:5173"],
        methods: ["GET", "POST"]
    }
});

const PORT = process.env.PORT || 8081;
const JWT_SECRET = process.env.JWT_SECRET || 'supersecretkey';

app.use(cors());
app.use(express.json());
app.use('/uploads', express.static('uploads'));

// Middleware for auth
const verifyToken = (req, res, next) => {
    const authHeader = req.headers['authorization'];
    const token = authHeader && authHeader.split(' ')[1];

    if (!token) return res.status(401).json({ error: 'Token missing' });

    jwt.verify(token, JWT_SECRET, (err, user) => {
        if (err) return res.status(403).json({ error: 'Invalid token' });
        req.user = user;
        next();
    });
};

// Public routes
app.use('/api/auth', authRoutes);

// Protected routes
app.use('/api/users', verifyToken, userRoutes);
app.use('/api/institutions', verifyToken, institutionRoutes);
app.use('/api/exams', verifyToken, examRoutes);
app.use('/api/questions', verifyToken, questionRoutes);
app.use('/api/notifications', verifyToken, notificationRoutes);
app.use('/api/config', verifyToken, configRoutes);
app.use('/api/root', verifyToken, rootRoutes);
app.use('/api/modules', verifyToken, moduleRoutes);
app.use('/api/specialties', verifyToken, specialtyRoutes);

// Socket.io logic (Live Monitor)
io.on('connection', (socket) => {
    console.log('User connected:', socket.id);

    socket.on('join-exam', (examId) => {
        socket.join(`exam-${examId}`);
        console.log(`User joined exam room: exam-${examId}`);
    });

    socket.on('exam-event', (data) => {
        // data: { examId, studentId, type: 'FOCUS_LOST' | 'TAB_CHANGE' | 'HEARTBEAT' }
        const { examId } = data;
        io.to(`exam-${examId}`).emit('monitor-alert', data);
        console.log('Exam event received:', data);
    });

    socket.on('disconnect', () => {
        console.log('User disconnected:', socket.id);
    });
});

httpServer.listen(PORT, () => {
    console.log(`KeQuiz server running on http://localhost:${PORT}`);
});
