import express from 'express';
import { ExamModel } from '../../src/models/exam.js';
import pool from '../../src/models/database.js';
import { notifyExamAssigned } from '../utils/notifications.js';

const router = express.Router();

router.get('/', async (req, res) => {
    try {
        const institutionId = req.user.role !== 'ROOT' ? req.user.institution_id : null;
        const exams = await ExamModel.getAll(institutionId);
        res.json(exams);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

router.post('/', async (req, res) => {
    try {
        const data = {
            ...req.body,
            professor_id: req.user.id,
            institution_id: req.user.institution_id
        };
        const exam = await ExamModel.create(data);
        res.status(201).json(exam);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

router.get('/:id', async (req, res) => {
    try {
        const studentId = req.user.role === 'ESTUDIANTE' || req.user.role === 'CLIENT' ? req.user.id : null;

        // El modelo ExamModel.getById ya fue actualizado para traer el intento más reciente
        const exam = await ExamModel.getById(req.params.id, studentId);

        if (!exam) return res.status(404).json({ error: 'Exam not found' });

        // Si es un estudiante, validar reglas de negocio (tiempos, re-aplicación)
        // NOTA: exam.assignment ya contiene el último intento gracias al cambio en el modelo
        if (studentId && exam.assignment) {
            const assignment = exam.assignment;
            const now = new Date();

            // 1. Ya completado (solo validamos este intento específico)
            if (assignment.status === 'COMPLETED') {
                return res.status(403).json({ error: 'Ya has completado este examen.' });
            }

            // 2. Tiempo de aplicación y tolerancia
            const rawScheduledAt = assignment.scheduled_at ? new Date(assignment.scheduled_at) : null;

            const scheduledAt = rawScheduledAt;

            const toleranceMinutes = exam.tolerance_minutes || 15;

            if (scheduledAt && assignment.status === 'PENDING') {
                const limitWithTolerance = new Date(scheduledAt.getTime() + toleranceMinutes * 60000);

                if (now < scheduledAt) {
                    return res.status(403).json({
                        error: `El examen aún no está disponible. Podrás iniciar a las ${scheduledAt.toLocaleString()}.`
                    });
                }

                if (now > limitWithTolerance) {
                    console.log(`[Exam Logic] Examen ${exam.id} marcando 0. Now: ${now}, Limit: ${limitWithTolerance.toISOString()}, Scheduled: ${scheduledAt.toISOString()}`);
                    // Marcar como No Presentado (score 0) automáticamente
                    await ExamModel.updateAssignment(exam.id, studentId, {
                        status: 'COMPLETED',
                        completed_at: now,
                        score: 0.00
                    });
                    return res.status(403).json({
                        error: 'Tu tiempo de tolerancia ha expirado. No puedes realizar el examen y se ha registrado una calificación de 0.'
                    });
                }
            }

            // 3. Temporizador persistente si ya inició
            if (assignment.status === 'IN_PROGRESS' && assignment.started_at) {
                const startedAt = new Date(assignment.started_at);
                const timerMinutes = parseInt(exam.settings?.timer_minutes) || 60;
                const expirationTime = new Date(startedAt.getTime() + timerMinutes * 60000);

                if (now > expirationTime) {
                    // Cerrar examen si el tiempo ya pasó mientras estaba fuera
                    await ExamModel.updateAssignment(exam.id, studentId, {
                        status: 'COMPLETED',
                        completed_at: expirationTime,
                        score: 0.00 // Debería ser proporcional a lo que contestó, pero por ahora 0 si no terminó
                    });
                    return res.status(403).json({ error: 'Tu tiempo para realizar el examen ha expirado.' });
                }

                // Enviar tiempo restante al frontend
                exam.timeLeftSeconds = Math.floor((expirationTime - now) / 1000);
            }
        }

        res.json(exam);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

router.put('/:id', async (req, res) => {
    try {
        const exam = await ExamModel.update(req.params.id, req.body);
        res.json(exam);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

router.delete('/:id', async (req, res) => {
    try {
        await ExamModel.delete(req.params.id);
        res.json({ message: 'Exam deleted' });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

// Assignments
router.get('/:id/assignments', async (req, res) => {
    try {
        const assignments = await ExamModel.getAssignments(req.params.id);
        res.json(assignments);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

router.post('/:id/assign', async (req, res) => {
    try {
        const { student_id, scheduled_at, scheduled_end_at } = req.body;
        const assignment = await ExamModel.assignToStudent(req.params.id, student_id, scheduled_at, scheduled_end_at);

        // Notificar al estudiante con las fechas de esta asignación específica
        await notifyExamAssigned(student_id, req.params.id, scheduled_at, scheduled_end_at);

        res.status(201).json(assignment);
    } catch (err) {
        // Errores de validación de negocio (conflictos de horario) deben ser 400
        // Errores del sistema (BD, etc.) deben ser 500
        const statusCode = err.message.includes('No se puede asignar') || err.message.includes('no encontrado') ? 400 : 500;
        res.status(statusCode).json({ error: err.message });
    }
});

router.delete('/:id/assign/:studentId', async (req, res) => {
    try {
        await ExamModel.unassignStudent(req.params.id, req.params.studentId);
        res.json({ message: 'Assignment removed' });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

// Taking Exams
router.post('/:id/start', async (req, res) => {
    try {
        const studentId = req.user.id;
        const examId = req.params.id;

        // Obtener el intento más reciente que NO esté completado
        const assignmentRes = await pool.query(
            `SELECT * FROM exam_assignments 
             WHERE exam_id = $1 AND student_id = $2 
             ORDER BY attempt_number DESC LIMIT 1`,
            [examId, studentId]
        );
        const assignment = assignmentRes.rows[0];

        if (!assignment) return res.status(404).json({ error: 'No assignment found for this student' });

        if (assignment.status === 'COMPLETED') {
            return res.status(403).json({ error: 'This exam has already been completed' });
        }

        // Validate time constraints before starting
        if (assignment.scheduled_at && assignment.status === 'PENDING') {
            const now = new Date();
            const rawScheduledAt = new Date(assignment.scheduled_at);

            const scheduledAt = rawScheduledAt;

            const examData = await ExamModel.getById(examId);
            const toleranceMinutes = examData.tolerance_minutes || 15;

            // Check if too early
            if (now < scheduledAt) {
                return res.status(403).json({
                    error: `El examen aún no está disponible. Podrás iniciar a las ${scheduledAt.toLocaleString()}.`
                });
            }

            // Check if too late (tolerance expired)
            const limitWithTolerance = new Date(scheduledAt.getTime() + toleranceMinutes * 60000);
            if (now > limitWithTolerance) {
                return res.status(403).json({
                    error: 'Tu tiempo de tolerancia ha expirado. No puedes realizar el examen.'
                });
            }
        }

        const updated = await ExamModel.updateAssignment(examId, studentId, {
            status: 'IN_PROGRESS',
            started_at: new Date()
        });

        res.json(updated);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

router.post('/:id/submit', async (req, res) => {
    try {
        const studentId = req.user.id;
        const examId = req.params.id;
        const { answers } = req.body; // { questionId: selectedOptionId/text }

        const exam = await ExamModel.getById(examId);
        if (!exam) return res.status(404).json({ error: 'Exam not found' });

        // Obtener el intento más reciente
        const assignmentRes = await pool.query(
            `SELECT * FROM exam_assignments 
             WHERE exam_id = $1 AND student_id = $2 
             ORDER BY attempt_number DESC LIMIT 1`,
            [examId, studentId]
        );
        const assignment = assignmentRes.rows[0];

        if (!assignment) return res.status(404).json({ error: 'No assignment found for this student' });

        if (assignment.status === 'COMPLETED') {
            return res.status(403).json({ error: 'This exam has already been completed' });
        }

        // Calculate score and save responses (UPSERT to avoid conflict with progressive save)
        let correctCount = 0;
        const totalQuestions = exam.questions.length;
        const responses = [];

        for (const q of exam.questions) {
            const studentAnswer = answers[q.id_question];
            const content = typeof q.content === 'string' ? JSON.parse(q.content) : q.content;

            let isCorrect = null;
            let points = 0;

            if (q.type === 'MULTIPLE_CHOICE' || q.type === 'TRUE_FALSE') {
                // El estudiante envía el INDEX de la opción seleccionada
                // q.correct_answer contiene el TEXTO de la respuesta correcta
                const selectedOptionText = content.options[parseInt(studentAnswer)];
                isCorrect = selectedOptionText === q.correct_answer;

                if (isCorrect) {
                    correctCount++;
                    points = 1;
                }
            }
            // Para OPEN, isCorrect es null hasta que el profesor califique

            responses.push({
                question_id: q.id_question,
                student_answer: studentAnswer || '',
                is_correct: isCorrect,
                points_earned: points
            });
        }

        // Save responses to DB (UPSERT to avoid conflict with progressive save)
        console.log(`Finalizando: Guardando ${responses.length} respuestas para asignación ${assignment.id_assignment}`);
        for (const resp of responses) {
            await pool.query(
                `INSERT INTO exam_responses (assignment_id, question_id, student_answer, is_correct, points_earned)
                 VALUES ($1, $2, $3, $4, $5)
                 ON CONFLICT (assignment_id, question_id) 
                 DO UPDATE SET student_answer = EXCLUDED.student_answer, 
                               is_correct = EXCLUDED.is_correct, 
                               points_earned = EXCLUDED.points_earned`,
                [assignment.id_assignment, resp.question_id, resp.student_answer, resp.is_correct, resp.points_earned]
            );
        }

        const score = (correctCount / totalQuestions) * 10; // Escala 0-10

        const updated = await ExamModel.updateAssignment(examId, studentId, {
            status: 'COMPLETED',
            completed_at: new Date(),
            score: Number(parseFloat(score).toFixed(2)) // Formato X.XX
        });

        res.json({
            message: 'Exam submitted successfully',
            score: updated.score,
            total_questions: totalQuestions,
            correct_answers: correctCount
        });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

// Grading and Review
router.post('/assignments/:assignmentId/responses', async (req, res) => {
    try {
        const { question_id, student_answer } = req.body;
        const { assignmentId } = req.params;

        console.log(`[Progress Save] Assignment ${assignmentId}, Question ${question_id}, Answer: ${student_answer}`);

        // 1. Obtener tipo de pregunta para auto-calificar si es posible
        const qRes = await pool.query('SELECT type, content, correct_answer FROM question_bank WHERE id_question = $1', [question_id]);
        if (!qRes.rows[0]) return res.status(404).json({ error: 'Question not found' });

        const q = qRes.rows[0];
        const content = typeof q.content === 'string' ? JSON.parse(q.content) : q.content;

        let isCorrect = null;
        let points = 0;

        if (q.type === 'MULTIPLE_CHOICE' || q.type === 'TRUE_FALSE') {
            const selectedOptionText = content.options[parseInt(student_answer)];
            isCorrect = selectedOptionText === q.correct_answer;
            if (isCorrect) points = 1;
        }

        // 2. Upsert de la respuesta
        const respRes = await pool.query(
            `INSERT INTO exam_responses (assignment_id, question_id, student_answer, is_correct, points_earned)
             VALUES ($1, $2, $3, $4, $5)
             ON CONFLICT (assignment_id, question_id) 
             DO UPDATE SET student_answer = EXCLUDED.student_answer, 
                           is_correct = EXCLUDED.is_correct, 
                           points_earned = EXCLUDED.points_earned
             RETURNING *`,
            [assignmentId, question_id, student_answer, isCorrect, points]
        );

        res.json(respRes.rows[0]);
    } catch (err) {
        console.error('[Progress Save ERROR]:', err.message);
        res.status(500).json({ error: err.message });
    }
});

router.get('/assignments/:assignmentId/responses', async (req, res) => {
    try {
        const result = await pool.query(
            `SELECT er.*, q.type as question_type, q.content as question_content, q.correct_answer
             FROM exam_responses er
             JOIN question_bank q ON er.question_id = q.id_question
             WHERE er.assignment_id = $1
             ORDER BY er.id_response ASC`,
            [req.params.assignmentId]
        );
        res.json(result.rows);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

router.put('/assignments/:assignmentId/responses/:responseId', async (req, res) => {
    try {
        const { is_correct, points_earned, feedback } = req.body;
        const { assignmentId, responseId } = req.params;

        // 1. Update individual response
        await pool.query(
            `UPDATE exam_responses 
             SET is_correct = $1, points_earned = $2, feedback = $3 
             WHERE id_response = $4 AND assignment_id = $5`,
            [is_correct, points_earned, feedback, responseId, assignmentId]
        );

        // 2. Recalculate total score for assignment
        const scoreResult = await pool.query(
            `SELECT SUM(points_earned) as earned, COUNT(*) as total
             FROM exam_responses 
             WHERE assignment_id = $1`,
            [assignmentId]
        );

        const { earned, total } = scoreResult.rows[0];
        const newScore = (parseFloat(earned) / parseFloat(total)) * 10; // Escala 0-10

        await pool.query(
            `UPDATE exam_assignments SET score = $1 WHERE id_assignment = $2`,
            [Number(parseFloat(newScore).toFixed(2)), assignmentId]
        );

        res.json({ message: 'Response updated', new_score: Number(parseFloat(newScore).toFixed(2)) });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

// Exam Logs / Audit
router.get('/:id/logs', async (req, res) => {
    try {
        const result = await pool.query(
            `SELECT el.*, ea.student_id 
             FROM exam_logs el
             JOIN exam_assignments ea ON el.assignment_id = ea.id_assignment
             WHERE ea.exam_id = $1
             ORDER BY el.timestamp DESC`,
            [req.params.id]
        );
        res.json(result.rows);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

export default router;
