import express from 'express';
import { InstitutionModel } from '../../src/models/institution.js';
import { ModuleModel } from '../../src/models/module.js';
import { upload } from '../utils/upload.js';

const router = express.Router();

router.get('/', async (req, res) => {
    try {
        const institutions = await InstitutionModel.getAll();
        console.log('API /institutions - Returning:', institutions.length, 'items');
        res.json(institutions);
    } catch (err) {
        console.error('API /institutions ERROR:', err.message);
        res.status(500).json({ error: err.message });
    }
});

router.get('/:id', async (req, res) => {
    try {
        const institution = await InstitutionModel.getById(req.params.id);
        if (!institution) return res.status(404).json({ error: 'Institution not found' });
        res.json(institution);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

router.post('/', async (req, res) => {
    try {
        const institution = await InstitutionModel.create(req.body);
        res.status(201).json(institution);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

router.put('/:id', async (req, res) => {
    try {
        const institution = await InstitutionModel.update(req.params.id, req.body);
        res.json(institution);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

router.delete('/:id', async (req, res) => {
    try {
        await InstitutionModel.delete(req.params.id);
        res.json({ message: 'Institution deleted' });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

router.get('/:id/modules', async (req, res) => {
    try {
        const modules = await ModuleModel.getByCompanyId(req.params.id);
        res.json(modules);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

router.put('/:id/modules/:moduleId', async (req, res) => {
    try {
        const { is_enabled } = req.body;
        const result = await ModuleModel.toggleModule(req.params.id, req.params.moduleId, is_enabled);
        res.json(result);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

router.post('/:id/logo', upload.single('logo'), async (req, res) => {
    try {
        if (!req.file) {
            return res.status(400).json({ error: 'No file uploaded' });
        }
        const logoUrl = `/uploads/${req.query.type || 'common'}/${req.file.filename}`;
        await InstitutionModel.update(req.params.id, { logo_url: logoUrl });
        res.json({ logo_url: logoUrl });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

export default router;
