import express from 'express';
import { QuestionModel } from '../../src/models/question.js';

const router = express.Router();

router.get('/', async (req, res) => {
    try {
        const filters = {};
        if (req.user.role === 'PROFESOR') {
            filters.professor_id = req.user.id;
        } else if (req.user.role !== 'ROOT') {
            filters.institution_id = req.user.institution_id;
        }

        const questions = await QuestionModel.getAll(filters);
        res.json(questions);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

router.post('/', async (req, res) => {
    try {
        const { content } = req.body;

        // Validar duplicado para este profesor
        const existing = await QuestionModel.checkDuplicate(req.user.id, content.question);
        if (existing) {
            return res.status(400).json({ error: 'Ya tienes una pregunta idéntica en tu banco de reactivos.' });
        }

        const data = {
            ...req.body,
            professor_id: req.user.id,
            institution_id: req.user.institution_id
        };
        const question = await QuestionModel.create(data);
        res.status(201).json(question);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

router.put('/:id', async (req, res) => {
    try {
        const question = await QuestionModel.update(req.params.id, req.body);
        res.json(question);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

router.delete('/:id', async (req, res) => {
    try {
        const result = await QuestionModel.delete(req.params.id);
        res.json(result);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

router.post('/bulk-delete', async (req, res) => {
    try {
        const { ids } = req.body;
        if (!Array.isArray(ids)) {
            return res.status(400).json({ error: 'ids must be an array' });
        }
        const result = await QuestionModel.deleteBulk(ids);
        res.json(result);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

export default router;

