import express from 'express';
import pool from '../../src/models/database.js';

const router = express.Router();

router.get('/institution-stats/:id', async (req, res) => {
    // Log para depuración extrema
    console.log('--- DEBUG STATS ACCESS ---');
    console.log('Full req.user:', JSON.stringify(req.user, null, 2));

    const { id } = req.params;
    const role = (req.user?.role || '').toString().trim().toUpperCase();

    // Fallback: Si el email es el de root, permitir aunque falle el role (temporal para debug)
    const isRootEmail = req.user?.email === 'hola@kemarketing.mx';

    if (role !== 'ROOT' && !isRootEmail) {
        console.warn(`[403] Bloqueado: ${req.user?.email}, Rol detectado: "${role}"`);
        return res.status(403).json({
            error: 'Access denied: Root only',
            your_role: role,
            your_email: req.user?.email
        });
    }

    console.log(`[200] Acceso concedido a ${req.user?.email} para estadísticas de institución ${id}`);

    try {
        // 1. Estudiantes (Mapeado a "clients" para el frontend)
        const studentsResult = await pool.query(
            'SELECT is_active, COUNT(*) FROM users WHERE institution_id = $1 AND role IN (\'ESTUDIANTE\', \'CLIENT\') GROUP BY is_active',
            [id]
        );
        const clients = { active: 0, inactive: 0 };
        studentsResult.rows.forEach(row => {
            if (row.is_active) clients.active = parseInt(row.count);
            else clients.inactive = parseInt(row.count);
        });

        // 2. Profesores (Mapeado a "employees" para el frontend)
        const professorsResult = await pool.query(
            'SELECT is_active, COUNT(*) FROM users WHERE institution_id = $1 AND role IN (\'PROFESOR\', \'EMPLOYEE\', \'ADMIN\') GROUP BY is_active',
            [id]
        );
        const employees = { active: 0, inactive: 0 };
        professorsResult.rows.forEach(row => {
            if (row.is_active) employees.active = parseInt(row.count);
            else employees.inactive = parseInt(row.count);
        });

        // 3. Exámenes (Mapeado a "usage.totalServices" y "ordersByStatus")
        const examsResult = await pool.query(
            'SELECT COUNT(*) FROM exams WHERE institution_id = $1',
            [id]
        );

        // Simular órdenes por estado basadas en asignaciones si existen, o simplemente por el examen
        const assignmentsResult = await pool.query(
            'SELECT status, COUNT(*) FROM exam_assignments ea JOIN exams e ON ea.exam_id = e.id_exam WHERE e.institution_id = $1 GROUP BY status',
            [id]
        );
        const ordersByStatus = {};
        assignmentsResult.rows.forEach(row => {
            ordersByStatus[row.status] = parseInt(row.count);
        });

        const stats = {
            clients,
            employees,
            usage: {
                totalServices: parseInt(examsResult.rows[0].count)
            },
            ordersByStatus,
            storage: {
                mb: 12.5, // Simulado por ahora
                gb: 0.012
            }
        };

        res.json(stats);
    } catch (err) {
        console.error('Error calculating institution stats:', err);
        res.status(500).json({ error: err.message });
    }
});

router.get('/time', (req, res) => {
    res.json({ time: new Date().toISOString() });
});

export default router;

