import express from 'express';
import { SpecialtyModel } from '../../src/models/specialty.js';

const router = express.Router();

// Middleware de autenticación (asumiendo que existe)
// router.use(authenticate);

router.get('/', async (req, res) => {
    try {
        const institutionId = req.user?.institution_id || req.user?.company_id || null;
        const specialties = await SpecialtyModel.getAll(institutionId);
        res.json(specialties);
    } catch (err) {
        console.error('API /specialties ERROR:', err.message);
        res.status(500).json({ error: err.message });
    }
});

router.get('/:id', async (req, res) => {
    try {
        const specialty = await SpecialtyModel.getById(req.params.id);
        if (!specialty) {
            return res.status(404).json({ error: 'Especialidad no encontrada' });
        }
        res.json(specialty);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

router.post('/', async (req, res) => {
    try {
        const { name, description, is_active } = req.body;
        const institutionId = req.user?.institution_id || req.user?.company_id;
        
        if (!institutionId) {
            return res.status(400).json({ error: 'ID de institución requerido' });
        }
        
        if (!name || !name.trim()) {
            return res.status(400).json({ error: 'El nombre de la especialidad es obligatorio' });
        }
        
        // Verificar duplicado
        const duplicate = await SpecialtyModel.checkDuplicate(institutionId, name);
        if (duplicate) {
            return res.status(400).json({ error: 'Ya existe una especialidad con ese nombre en tu institución' });
        }
        
        const specialty = await SpecialtyModel.create({
            institution_id: institutionId,
            name: name.trim(),
            description: description?.trim() || null,
            is_active: is_active !== undefined ? is_active : true
        });
        
        res.status(201).json(specialty);
    } catch (err) {
        console.error('Error creating specialty:', err);
        if (err.code === '23505') { // Unique violation
            res.status(400).json({ error: 'Ya existe una especialidad con ese nombre en tu institución' });
        } else {
            res.status(500).json({ error: err.message });
        }
    }
});

router.put('/:id', async (req, res) => {
    try {
        const { name, description, is_active } = req.body;
        const institutionId = req.user?.institution_id || req.user?.company_id;
        
        // Verificar que la especialidad existe y pertenece a la institución
        const existing = await SpecialtyModel.getById(req.params.id);
        if (!existing) {
            return res.status(404).json({ error: 'Especialidad no encontrada' });
        }
        
        if (existing.institution_id !== institutionId) {
            return res.status(403).json({ error: 'No tienes permiso para modificar esta especialidad' });
        }
        
        if (name && name.trim()) {
            // Verificar duplicado excluyendo el registro actual
            const duplicate = await SpecialtyModel.checkDuplicate(institutionId, name, req.params.id);
            if (duplicate) {
                return res.status(400).json({ error: 'Ya existe una especialidad con ese nombre en tu institución' });
            }
        }
        
        const updateData = {};
        if (name !== undefined) updateData.name = name.trim();
        if (description !== undefined) updateData.description = description?.trim() || null;
        if (is_active !== undefined) updateData.is_active = is_active;
        
        const specialty = await SpecialtyModel.update(req.params.id, updateData);
        res.json(specialty);
    } catch (err) {
        console.error('Error updating specialty:', err);
        if (err.code === '23505') { // Unique violation
            res.status(400).json({ error: 'Ya existe una especialidad con ese nombre en tu institución' });
        } else {
            res.status(500).json({ error: err.message });
        }
    }
});

router.delete('/:id', async (req, res) => {
    try {
        const institutionId = req.user?.institution_id || req.user?.company_id;
        
        // Verificar que la especialidad existe y pertenece a la institución
        const existing = await SpecialtyModel.getById(req.params.id);
        if (!existing) {
            return res.status(404).json({ error: 'Especialidad no encontrada' });
        }
        
        if (existing.institution_id !== institutionId) {
            return res.status(403).json({ error: 'No tienes permiso para eliminar esta especialidad' });
        }
        
        await SpecialtyModel.delete(req.params.id);
        res.json({ message: 'Especialidad eliminada exitosamente' });
    } catch (err) {
        console.error('Error deleting specialty:', err);
        if (err.code === '23503') { // Foreign key violation
            res.status(400).json({ error: 'No se puede eliminar la especialidad porque está siendo utilizada por uno o más profesores' });
        } else {
            res.status(500).json({ error: err.message });
        }
    }
});

export default router;


