import express from 'express';
import bcrypt from 'bcryptjs';
import { UserModel } from '../../src/models/user.js';
import { upload } from '../utils/upload.js';

import pool from '../../src/models/database.js';

const router = express.Router();

// Get student assigned exams
router.get('/:id/exams', async (req, res) => {
    try {
        const studentId = req.params.id;
        const result = await pool.query(`
            SELECT e.*, ea.status as assignment_status, ea.score, ea.started_at, ea.completed_at,
                   ea.scheduled_at as assignment_scheduled_at, ea.scheduled_end_at as assignment_scheduled_end_at,
                   ea.attempt_number, ea.id_assignment, e.tolerance_minutes
            FROM exams e
            JOIN exam_assignments ea ON e.id_exam = ea.exam_id
            WHERE ea.student_id = $1
            ORDER BY ea.attempt_number DESC, e.created_at DESC
        `, [studentId]);
        res.json(result.rows);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

router.get('/', async (req, res) => {
    try {
        const users = await UserModel.getAll();
        console.log('API /users - Returning:', users.length, 'items');
        res.json(users);
    } catch (err) {
        console.error('API /users ERROR:', err.message);
        res.status(500).json({ error: err.message });
    }
});

router.post('/', async (req, res) => {
    try {
        const { email, password, role, is_active, institution_id, name, phone, specialty, client_number, first_name, last_name, second_last_name, curp } = req.body;

        if (!email) {
            return res.status(400).json({ error: 'El email es obligatorio.' });
        }

        const cleanEmailPrefix = email.split('@')[0].replace(/[^a-zA-Z0-9]/g, '') || 'user';
        const rolePrefix = (role || 'root').toLowerCase();
        const composedId = `usr-${rolePrefix}-${cleanEmailPrefix}-${Math.floor(Date.now() / 1000).toString().slice(-4)}`;

        const data = {
            id: req.body.id || composedId,
            email,
            role: role || 'ROOT',
            is_active: is_active !== undefined ? is_active : true,
            institution_id: institution_id || null,
            name: name || null,
            first_name: first_name || null,
            last_name: last_name || null,
            second_last_name: second_last_name || null,
            phone: phone || null,
            specialty: specialty || null,
            client_number: client_number || null,
            curp: curp || null,
            photo_url: req.body.photo_url || null
        };

        if (password) {
            const salt = await bcrypt.genSalt(10);
            data.password_hash = await bcrypt.hash(password, salt);
        } else if (!req.body.id) {
            return res.status(400).json({ error: 'La contraseña es obligatoria para nuevos usuarios.' });
        }

        const user = await UserModel.create(data);
        res.status(201).json(user);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

router.put('/:id', async (req, res) => {
    try {
        const data = { ...req.body };
        if (data.password) {
            const salt = await bcrypt.genSalt(10);
            data.password_hash = await bcrypt.hash(data.password, salt);
        }
        delete data.password;
        const user = await UserModel.update(req.params.id, data);
        res.json(user);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

router.delete('/:id', async (req, res) => {
    try {
        await UserModel.delete(req.params.id);
        res.json({ message: 'User deleted' });
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
});

router.post('/:id/photo', upload.single('photo'), async (req, res) => {
    try {
        if (!req.file) {
            return res.status(400).json({ error: 'No file uploaded' });
        }

        // Obtener el usuario actual para ver si ya tiene una foto
        const oldUser = await UserModel.getById(req.params.id);
        if (oldUser && oldUser.photo_url) {
            try {
                // El photo_url es algo como /uploads/users/file.jpg
                // Necesitamos la ruta física: server/uploads/users/file.jpg o similar
                // Como estamos en server/routes/users.js, la base es la raíz del servidor
                const oldPath = path.join(process.cwd(), oldUser.photo_url.replace(/^\//, ''));
                if (fs.existsSync(oldPath)) {
                    fs.unlinkSync(oldPath);
                    console.log('Antigua foto eliminada:', oldPath);
                }
            } catch (unlinkErr) {
                console.error('Error al eliminar foto antigua:', unlinkErr);
            }
        }

        // Determinar el directorio basado en el query o por defecto a common
        const type = req.query.type || 'common';
        const photoUrl = `/uploads/${type}/${req.file.filename}`;

        console.log(`Guardando foto para usuario ${req.params.id}: ${photoUrl}`);

        await UserModel.update(req.params.id, { photo_url: photoUrl });
        res.json({ photo_url: photoUrl });
    } catch (err) {
        console.error('Error en upload photo:', err);
        res.status(500).json({ error: err.message });
    }
});

export default router;

