-- =====================================================
-- Script de Inicialización de Base de Datos
-- Sistema Multi-Tenant de Registro y Seguimiento de Servicios
-- Versión: 1.0.0-beta
-- Fecha: 11 de diciembre de 2025
-- =====================================================

-- Eliminar tablas existentes (en orden inverso por dependencias)
DROP TABLE IF EXISTS evidences CASCADE;
DROP TABLE IF EXISTS contracted_services CASCADE;
DROP TABLE IF EXISTS company_modules CASCADE;
DROP TABLE IF EXISTS services_catalog CASCADE;
DROP TABLE IF EXISTS employees CASCADE;
DROP TABLE IF EXISTS clients CASCADE;
DROP TABLE IF EXISTS users CASCADE;
DROP TABLE IF EXISTS modules CASCADE;
DROP TABLE IF EXISTS system_config CASCADE;
DROP TABLE IF EXISTS companies CASCADE;

-- =====================================================
-- TABLA: companies (Empresas Inquilinas)
-- =====================================================
CREATE TABLE companies (
    id SERIAL PRIMARY KEY,
    name VARCHAR(255) NOT NULL,
    logo_url TEXT,
    slogan VARCHAR(500),
    street VARCHAR(255),
    exterior_number VARCHAR(50),
    interior_number VARCHAR(50),
    neighborhood VARCHAR(255),
    municipality VARCHAR(255),
    state VARCHAR(100),
    postal_code VARCHAR(10),
    phone VARCHAR(20),
    email VARCHAR(255),
    service_completion_message TEXT,
    max_clients INTEGER DEFAULT 0,
    max_employees INTEGER DEFAULT 0,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);


-- =====================================================
-- TABLA: system_config (Configuración Global)
-- =====================================================
CREATE TABLE system_config (
    id SERIAL PRIMARY KEY,
    config_key VARCHAR(100) UNIQUE NOT NULL,
    config_value TEXT,
    description VARCHAR(500),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- =====================================================
-- TABLA: modules (Módulos del Sistema)
-- =====================================================
CREATE TABLE modules (
    id SERIAL PRIMARY KEY,
    name VARCHAR(255) NOT NULL,
    description TEXT,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- =====================================================
-- TABLA: users (Usuarios del Sistema)
-- =====================================================
CREATE TABLE users (
    id VARCHAR(50) PRIMARY KEY,
    company_id INTEGER REFERENCES companies(id) ON DELETE CASCADE,
    email VARCHAR(255) UNIQUE NOT NULL,
    password_hash VARCHAR(255) NOT NULL,
    role VARCHAR(20) NOT NULL CHECK (role IN ('ROOT', 'ADMIN', 'EMPLOYEE', 'CLIENT')),
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- =====================================================
-- TABLA: clients (Perfiles de Clientes)
-- =====================================================
CREATE TABLE clients (
    id SERIAL PRIMARY KEY,
    user_id VARCHAR(50) UNIQUE REFERENCES users(id) ON DELETE CASCADE,
    company_id INTEGER REFERENCES companies(id) ON DELETE CASCADE,
    client_number VARCHAR(50) UNIQUE NOT NULL,
    name VARCHAR(255) NOT NULL,
    phone VARCHAR(20),
    fiscal_name VARCHAR(255),
    rfc VARCHAR(13),
    fiscal_address TEXT,
    address TEXT,
    alternate_address TEXT,
    email VARCHAR(255),
    is_active BOOLEAN DEFAULT TRUE,
    allow_access_without_service BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- =====================================================
-- TABLA: employees (Perfiles de Empleados)
-- =====================================================
CREATE TABLE employees (
    id SERIAL PRIMARY KEY,
    user_id VARCHAR(50) UNIQUE REFERENCES users(id) ON DELETE CASCADE,
    company_id INTEGER REFERENCES companies(id) ON DELETE CASCADE,
    name VARCHAR(255) NOT NULL,
    phone VARCHAR(20),
    specialty VARCHAR(255),
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- =====================================================
-- TABLA: services_catalog (Catálogo de Servicios)
-- =====================================================
CREATE TABLE services_catalog (
    id SERIAL PRIMARY KEY,
    company_id INTEGER REFERENCES companies(id) ON DELETE CASCADE,
    name VARCHAR(255) NOT NULL,
    description TEXT,
    category VARCHAR(100),
    max_time_hours INTEGER,
    has_warranty BOOLEAN DEFAULT FALSE,
    warranty_months INTEGER,
    warranty_exceptions TEXT,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- =====================================================
-- TABLA: contracted_services (Órdenes de Servicio)
-- =====================================================
CREATE TABLE contracted_services (
    id SERIAL PRIMARY KEY,
    company_id INTEGER REFERENCES companies(id) ON DELETE CASCADE,
    client_id VARCHAR(50) REFERENCES users(id) ON DELETE CASCADE,
    service_catalogo_id INTEGER REFERENCES services_catalog(id) ON DELETE CASCADE,
    employee_id INTEGER REFERENCES employees(id) ON DELETE SET NULL,
    service_number VARCHAR(50) UNIQUE NOT NULL,
    request_date TIMESTAMP NOT NULL,
    deadline_date TIMESTAMP,
    start_date TIMESTAMP,
    end_date TIMESTAMP,
    closed_at TIMESTAMP,
    warranty_end_date TIMESTAMP,
    payment_reference VARCHAR(100),
    payment_status VARCHAR(20) DEFAULT 'PENDING',
    payment_proof_url TEXT,
    status VARCHAR(20) NOT NULL CHECK (status IN ('PENDING', 'IN_PROGRESS', 'FINISHED', 'CANCELLED', 'CERRADO_ADMIN', 'IN_REVIEW')),
    employee_report TEXT,
    client_rating INTEGER CHECK (client_rating >= 1 AND client_rating <= 5),
    client_comment TEXT,
    finalized_by_client BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- =====================================================
-- TABLA: evidences (Evidencias Multimedia)
-- =====================================================
CREATE TABLE evidences (
    id SERIAL PRIMARY KEY,
    contracted_service_id INTEGER REFERENCES contracted_services(id) ON DELETE CASCADE,
    evidence_type VARCHAR(20) CHECK (evidence_type IN ('PHOTO', 'VIDEO', 'DOCUMENT')),
    file_url TEXT NOT NULL,
    description TEXT,
    uploaded_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- =====================================================
-- TABLA: company_modules (Módulos Habilitados por Empresa)
-- =====================================================
    UNIQUE(company_id, module_id)
);

-- =====================================================
-- TABLA: notifications (Notificaciones de Usuarios)
-- =====================================================
CREATE TABLE notifications (
    id SERIAL PRIMARY KEY,
    user_id VARCHAR(50) REFERENCES users(id) ON DELETE CASCADE,
    type VARCHAR(50) NOT NULL,
    title VARCHAR(255) NOT NULL,
    message TEXT,
    related_service_id INTEGER REFERENCES contracted_services(id) ON DELETE CASCADE,
    is_read BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- =====================================================
-- ÍNDICES PARA OPTIMIZACIÓN
-- =====================================================
CREATE INDEX idx_users_company ON users(company_id);
CREATE INDEX idx_users_email ON users(email);
CREATE INDEX idx_users_role ON users(role);
CREATE INDEX idx_clients_company ON clients(company_id);
CREATE INDEX idx_clients_number ON clients(client_number);
CREATE INDEX idx_employees_company ON employees(company_id);
CREATE INDEX idx_services_catalog_company ON services_catalog(company_id);
CREATE INDEX idx_contracted_services_company ON contracted_services(company_id);
CREATE INDEX idx_contracted_services_client ON contracted_services(client_id);
CREATE INDEX idx_contracted_services_employee ON contracted_services(employee_id);
CREATE INDEX idx_contracted_services_status ON contracted_services(status);
CREATE INDEX idx_evidences_service ON evidences(contracted_service_id);
CREATE INDEX idx_notifications_user ON notifications(user_id);
CREATE INDEX idx_notifications_read ON notifications(is_read);

-- =====================================================
-- DATOS INICIALES: Empresas
-- =====================================================
INSERT INTO companies (id, name, logo_url, slogan, street, exterior_number, neighborhood, municipality, state, postal_code, phone, email, service_completion_message, is_active) VALUES
(1, 'Servicios Técnicos del Norte', 'https://ui-avatars.com/api/?name=STN&background=0EA5E9&color=fff&size=200', 'Soluciones integrales para tu hogar y negocio', 'Av. Revolución', '1234', 'Centro', 'Monterrey', 'Nuevo León', '64000', '8112345678', 'contacto@serviciosnorte.com', 'Gracias por confiar en nosotros. Su servicio ha sido completado satisfactoriamente.', true),
(2, 'Mantenimiento Integral SA', 'https://ui-avatars.com/api/?name=MI&background=10B981&color=fff&size=200', 'Mantenimiento profesional a tu alcance', 'Calle Juárez', '567', 'Zona Industrial', 'Guadalajara', 'Jalisco', '44100', '3398765432', 'info@mantenimientointegral.com', 'Servicio finalizado. Estamos a sus órdenes para cualquier consulta.', true),
(3, 'Construcciones y Servicios del Sur', 'https://ui-avatars.com/api/?name=CSS&background=EF4444&color=fff&size=200', 'Construyendo el futuro juntos', 'Blvd. Kukulcán', '890', 'Zona Hotelera', 'Cancún', 'Quintana Roo', '77500', '9981234567', 'ventas@construccionessur.com', 'Agradecemos su preferencia.', false);

-- =====================================================
-- DATOS INICIALES: Configuración del Sistema
-- =====================================================
INSERT INTO system_config (config_key, config_value, description) VALUES
('footer_text', '© 2025 KeSoftware - Sistema de Gestión de Servicios', 'Texto del pie de página global'),
('footer_link_text', 'Soporte Técnico', 'Texto del enlace en el pie de página'),
('footer_link_url', 'https://kesoftware.com/soporte', 'URL del enlace en el pie de página');

-- =====================================================
-- DATOS INICIALES: Módulos del Sistema
-- =====================================================
INSERT INTO modules (id, name, description) VALUES
(1, 'Gestión de Clientes', 'Administración de clientes y sus datos'),
(2, 'Catálogo de Servicios', 'Gestión del catálogo de servicios ofrecidos'),
(3, 'Órdenes de Servicio', 'Gestión de órdenes de trabajo'),
(4, 'Gestión de Empleados', 'Administración de personal técnico'),
(5, 'Reportes', 'Generación de reportes y estadísticas');

-- =====================================================
-- DATOS INICIALES: Usuarios
-- =====================================================
INSERT INTO users (id, company_id, email, password_hash, role, is_active) VALUES
-- ROOT
('usr-root', NULL, 'root@kesoftware.com', '$2a$10$demo123hash', 'ROOT', true),

-- ADMIN
('usr-001', 1, 'admin@serviciosnorte.com', '$2a$10$demo123hash', 'ADMIN', true),
('usr-002', 2, 'admin@mantenimientointegral.com', '$2a$10$demo123hash', 'ADMIN', true),
('usr-003', 3, 'admin@construccionessur.com', '$2a$10$demo123hash', 'ADMIN', true),

-- EMPLOYEE
('usr-010', 1, 'juan.perez@serviciosnorte.com', '$2a$10$demo123hash', 'EMPLOYEE', true),
('usr-011', 1, 'maria.garcia@serviciosnorte.com', '$2a$10$demo123hash', 'EMPLOYEE', true),
('usr-012', 2, 'carlos.lopez@mantenimientointegral.com', '$2a$10$demo123hash', 'EMPLOYEE', true),

-- CLIENT
('usr-100', 1, 'roberto.martinez@email.com', '$2a$10$demo123hash', 'CLIENT', true),
('usr-101', 1, 'ana.rodriguez@email.com', '$2a$10$demo123hash', 'CLIENT', true),
('usr-102', 2, 'luis.hernandez@email.com', '$2a$10$demo123hash', 'CLIENT', true);

-- =====================================================
-- DATOS INICIALES: Clientes
-- =====================================================
INSERT INTO clients (user_id, company_id, client_number, name, phone, fiscal_name, rfc, fiscal_address, address, alternate_address, allow_access_without_service) VALUES
('usr-100', 1, 'CLI-2024-001', 'Roberto Martínez', '8187654321', 'Roberto Martínez García', 'MAGR850315AB1', 'Av. Constitución 123, Col. Centro, Monterrey, NL', 'Calle Morelos 456, Col. Obrera, Monterrey, NL', 'Av. Lincoln 789, Col. Residencial, San Pedro, NL', false),
('usr-101', 1, 'CLI-2024-002', 'Ana Rodríguez', '8176543210', 'Ana Rodríguez López', 'ROLA920520CD2', 'Blvd. Díaz Ordaz 234, Col. Santa María, Monterrey, NL', 'Calle Hidalgo 567, Col. Del Valle, Monterrey, NL', NULL, true),
('usr-102', 2, 'CLI-2024-003', 'Luis Hernández', '3312345678', 'Luis Hernández Pérez', 'HEPL880710EF3', 'Av. Chapultepec 345, Col. Americana, Guadalajara, JAL', 'Calle Independencia 678, Col. Centro, Guadalajara, JAL', NULL, false);

-- =====================================================
-- DATOS INICIALES: Empleados
-- =====================================================
INSERT INTO employees (user_id, company_id, name, phone, specialty, is_active) VALUES
('usr-010', 1, 'Juan Pérez', '8198765432', 'Mantenimiento General', true),
('usr-011', 1, 'María García', '8187654321', 'Electricidad', true),
('usr-012', 2, 'Carlos López', '3387654321', 'Plomería', true);

-- =====================================================
-- DATOS INICIALES: Catálogo de Servicios
-- =====================================================
INSERT INTO services_catalog (id, company_id, name, description, category, max_time_hours, has_warranty, warranty_months, warranty_exceptions) VALUES
(1, 1, 'Reparación de Instalación Eléctrica', 'Diagnóstico y reparación de fallas eléctricas en instalaciones residenciales y comerciales', 'Electricidad', 4, true, 3, 'No cubre daños por sobrecarga, cortocircuitos externos o manipulación posterior'),
(2, 1, 'Mantenimiento de Plomería', 'Reparación de fugas, destapado de drenajes y mantenimiento preventivo', 'Plomería', 3, true, 6, 'No cubre daños por objetos extraños o uso inadecuado'),
(3, 1, 'Pintura Interior', 'Aplicación de pintura en interiores con preparación de superficies', 'Pintura', 8, true, 12, 'No cubre daños por humedad, filtración de agua o golpes'),
(4, 1, 'Instalación de Aire Acondicionado', 'Instalación profesional de equipos de climatización', 'Climatización', 6, true, 12, 'Garantía solo sobre instalación, no cubre falta de mantenimiento'),
(5, 1, 'Limpieza Profunda', 'Limpieza integral de espacios residenciales y comerciales', 'Limpieza', 5, false, NULL, NULL),
(6, 2, 'Reparación de Electrodomésticos', 'Diagnóstico y reparación de electrodomésticos de línea blanca', 'Electrodomésticos', 3, true, 3, 'No cubre piezas de desgaste natural'),
(7, 2, 'Mantenimiento Preventivo', 'Revisión y mantenimiento de instalaciones generales', 'Mantenimiento', 4, false, NULL, NULL);

-- =====================================================
-- DATOS INICIALES: Órdenes de Servicio
-- =====================================================
INSERT INTO contracted_services (id, company_id, client_id, service_catalogo_id, employee_id, service_number, request_date, deadline_date, start_date, end_date, closed_at, warranty_end_date, status, employee_report, client_rating, client_comment, finalized_by_client) VALUES
(1, 1, 'usr-100', 1, 2, 'SRV-2024-001', '2024-10-15 09:00:00', '2024-10-16 18:00:00', '2024-10-15 10:00:00', '2024-10-15 14:00:00', '2024-11-01 10:00:00', '2025-02-01 10:00:00', 'FINISHED', 'Se reparó cortocircuito en instalación principal. Se reemplazaron 3 contactos dañados.', 5, 'Excelente servicio, muy profesional', true),
(2, 1, 'usr-100', 2, 1, 'SRV-2024-002', '2024-11-20 08:00:00', '2024-11-21 17:00:00', '2024-11-20 09:00:00', NULL, NULL, NULL, 'IN_PROGRESS', NULL, NULL, NULL, false),
(3, 1, 'usr-101', 3, 1, 'SRV-2024-003', '2024-12-01 10:00:00', '2024-12-05 18:00:00', NULL, NULL, NULL, NULL, 'PENDING', NULL, NULL, NULL, false),
(4, 1, 'usr-101', 4, 2, 'SRV-2024-004', '2024-09-10 11:00:00', '2024-09-12 17:00:00', '2024-09-10 13:00:00', '2024-09-11 16:00:00', '2024-10-10 09:00:00', '2025-10-10 09:00:00', 'FINISHED', 'Instalación completa de minisplit 1.5 toneladas. Incluye tubería de cobre y cableado.', 5, 'Muy satisfecho con el trabajo', true),
(5, 2, 'usr-102', 6, 3, 'SRV-2024-005', '2024-12-05 14:00:00', '2024-12-07 18:00:00', NULL, NULL, NULL, NULL, 'PENDING', NULL, NULL, NULL, false),
(6, 1, 'usr-100', 5, 1, 'SRV-2024-006', '2024-11-10 08:00:00', '2024-11-10 18:00:00', '2024-11-10 09:00:00', '2024-11-10 14:00:00', '2024-11-15 10:00:00', NULL, 'FINISHED', 'Limpieza completa de departamento de 80m2', 4, 'Buen servicio', true);

-- =====================================================
-- DATOS INICIALES: Evidencias
-- =====================================================
INSERT INTO evidences (contracted_service_id, evidence_type, file_url, description, uploaded_at) VALUES
(1, 'PHOTO', 'https://via.placeholder.com/400x300/0EA5E9/FFFFFF?text=Instalacion+Electrica+Antes', 'Estado inicial de la instalación eléctrica', '2024-10-15 10:15:00'),
(1, 'PHOTO', 'https://via.placeholder.com/400x300/10B981/FFFFFF?text=Instalacion+Electrica+Despues', 'Instalación reparada y funcionando', '2024-10-15 14:00:00'),
(2, 'PHOTO', 'https://via.placeholder.com/400x300/F59E0B/FFFFFF?text=Fuga+de+Agua', 'Fuga detectada en tubería principal', '2024-11-20 09:30:00'),
(4, 'PHOTO', 'https://via.placeholder.com/400x300/8B5CF6/FFFFFF?text=Instalacion+AC+1', 'Unidad exterior instalada', '2024-09-11 12:00:00'),
(4, 'PHOTO', 'https://via.placeholder.com/400x300/EC4899/FFFFFF?text=Instalacion+AC+2', 'Unidad interior instalada', '2024-09-11 14:00:00'),
(4, 'PHOTO', 'https://via.placeholder.com/400x300/06B6D4/FFFFFF?text=Instalacion+AC+3', 'Sistema completo funcionando', '2024-09-11 16:00:00'),
(6, 'PHOTO', 'https://via.placeholder.com/400x300/84CC16/FFFFFF?text=Limpieza+Antes', 'Estado inicial del departamento', '2024-11-10 09:00:00'),
(6, 'PHOTO', 'https://via.placeholder.com/400x300/14B8A6/FFFFFF?text=Limpieza+Despues', 'Departamento limpio', '2024-11-10 14:00:00');

-- =====================================================
-- DATOS INICIALES: Módulos Habilitados por Empresa
-- =====================================================
INSERT INTO company_modules (company_id, module_id, is_enabled) VALUES
-- Empresa 1: Todos los módulos habilitados
(1, 1, true),
(1, 2, true),
(1, 3, true),
(1, 4, true),
(1, 5, true),

-- Empresa 2: Solo algunos módulos
(2, 1, true),
(2, 2, true),
(2, 3, true),
(2, 4, false),
(2, 5, false),

-- Empresa 3: Todos deshabilitados (empresa inactiva)
(3, 1, false),
(3, 2, false),
(3, 3, false),
(3, 4, false),
(3, 5, false);

-- =====================================================
-- RESETEAR SECUENCIAS
-- =====================================================
SELECT setval('companies_id_seq', (SELECT MAX(id) FROM companies));
SELECT setval('modules_id_seq', (SELECT MAX(id) FROM modules));
SELECT setval('clients_id_seq', (SELECT MAX(id) FROM clients));
SELECT setval('employees_id_seq', (SELECT MAX(id) FROM employees));
SELECT setval('services_catalog_id_seq', (SELECT MAX(id) FROM services_catalog));
SELECT setval('contracted_services_id_seq', (SELECT MAX(id) FROM contracted_services));
SELECT setval('evidences_id_seq', (SELECT MAX(id) FROM evidences));
SELECT setval('company_modules_id_seq', (SELECT MAX(id) FROM company_modules));
SELECT setval('notifications_id_seq', (SELECT MAX(id) FROM notifications));

-- =====================================================
-- VERIFICACIÓN DE DATOS
-- =====================================================
SELECT 'Empresas creadas: ' || COUNT(*) FROM companies;
SELECT 'Usuarios creados: ' || COUNT(*) FROM users;
SELECT 'Clientes creados: ' || COUNT(*) FROM clients;
SELECT 'Empleados creados: ' || COUNT(*) FROM employees;
SELECT 'Servicios en catálogo: ' || COUNT(*) FROM services_catalog;
SELECT 'Órdenes de servicio: ' || COUNT(*) FROM contracted_services;
SELECT 'Evidencias registradas: ' || COUNT(*) FROM evidences;
SELECT 'Módulos del sistema: ' || COUNT(*) FROM modules;

-- =====================================================
-- FIN DEL SCRIPT
-- =====================================================
