/*
  Migración: Agregar campos first_name, last_name, second_last_name a la tabla users
  Fecha: 2025-01-XX
  Descripción: 
    - Verifica si los campos existen antes de agregarlos
    - Agrega campos first_name, last_name, second_last_name si no existen
    - Migra datos existentes del campo 'name' a los nuevos campos (si existe)
    - Hace first_name y last_name NOT NULL
    - Elimina el campo 'name' antiguo (si existe)
*/

DO $$
BEGIN
    -- 1. Verificar y agregar first_name si no existe
    IF NOT EXISTS (
        SELECT 1 
        FROM information_schema.columns 
        WHERE table_schema = 'public' 
        AND table_name = 'users' 
        AND column_name = 'first_name'
    ) THEN
        ALTER TABLE "public"."users" 
        ADD COLUMN "first_name" varchar(255);
        RAISE NOTICE 'Campo first_name agregado';
    ELSE
        RAISE NOTICE 'Campo first_name ya existe';
    END IF;

    -- 2. Verificar y agregar last_name si no existe
    IF NOT EXISTS (
        SELECT 1 
        FROM information_schema.columns 
        WHERE table_schema = 'public' 
        AND table_name = 'users' 
        AND column_name = 'last_name'
    ) THEN
        ALTER TABLE "public"."users" 
        ADD COLUMN "last_name" varchar(255);
        RAISE NOTICE 'Campo last_name agregado';
    ELSE
        RAISE NOTICE 'Campo last_name ya existe';
    END IF;

    -- 3. Verificar y agregar second_last_name si no existe
    IF NOT EXISTS (
        SELECT 1 
        FROM information_schema.columns 
        WHERE table_schema = 'public' 
        AND table_name = 'users' 
        AND column_name = 'second_last_name'
    ) THEN
        ALTER TABLE "public"."users" 
        ADD COLUMN "second_last_name" varchar(255);
        RAISE NOTICE 'Campo second_last_name agregado';
    ELSE
        RAISE NOTICE 'Campo second_last_name ya existe';
    END IF;

    -- 4. Migrar datos del campo 'name' si existe
    IF EXISTS (
        SELECT 1 
        FROM information_schema.columns 
        WHERE table_schema = 'public' 
        AND table_name = 'users' 
        AND column_name = 'name'
    ) THEN
        -- Migrar datos existentes del campo 'name' a los nuevos campos
        UPDATE "public"."users"
        SET 
            "first_name" = CASE 
                WHEN "name" IS NOT NULL AND TRIM("name") != '' THEN
                    SPLIT_PART(TRIM("name"), ' ', 1)
                ELSE COALESCE("first_name", 'Usuario')
            END,
            "last_name" = CASE 
                WHEN "name" IS NOT NULL AND TRIM("name") != '' AND array_length(string_to_array(TRIM("name"), ' '), 1) >= 2 THEN
                    SPLIT_PART(TRIM("name"), ' ', 2)
                ELSE COALESCE("last_name", 'Sin Apellido')
            END,
            "second_last_name" = CASE 
                WHEN "name" IS NOT NULL AND TRIM("name") != '' AND array_length(string_to_array(TRIM("name"), ' '), 1) > 2 THEN
                    array_to_string((string_to_array(TRIM("name"), ' '))[3:], ' ')
                ELSE "second_last_name"
            END
        WHERE ("first_name" IS NULL OR "last_name" IS NULL) 
           OR ("name" IS NOT NULL AND TRIM("name") != '');
        
        RAISE NOTICE 'Datos migrados del campo name';
    ELSE
        RAISE NOTICE 'Campo name no existe, no se migran datos';
    END IF;

    -- 5. Establecer valores por defecto para registros sin nombre
    UPDATE "public"."users"
    SET 
        "first_name" = COALESCE("first_name", 'Usuario'),
        "last_name" = COALESCE("last_name", 'Sin Apellido')
    WHERE "first_name" IS NULL OR "last_name" IS NULL;

    -- 6. Hacer los campos obligatorios (solo si no tienen constraint NOT NULL)
    IF EXISTS (
        SELECT 1 
        FROM information_schema.columns 
        WHERE table_schema = 'public' 
        AND table_name = 'users' 
        AND column_name = 'first_name'
        AND is_nullable = 'YES'
    ) THEN
        ALTER TABLE "public"."users" 
        ALTER COLUMN "first_name" SET NOT NULL;
        RAISE NOTICE 'Campo first_name ahora es NOT NULL';
    END IF;

    IF EXISTS (
        SELECT 1 
        FROM information_schema.columns 
        WHERE table_schema = 'public' 
        AND table_name = 'users' 
        AND column_name = 'last_name'
        AND is_nullable = 'YES'
    ) THEN
        ALTER TABLE "public"."users" 
        ALTER COLUMN "last_name" SET NOT NULL;
        RAISE NOTICE 'Campo last_name ahora es NOT NULL';
    END IF;

    -- 7. Eliminar el campo 'name' antiguo si existe
    IF EXISTS (
        SELECT 1 
        FROM information_schema.columns 
        WHERE table_schema = 'public' 
        AND table_name = 'users' 
        AND column_name = 'name'
    ) THEN
        ALTER TABLE "public"."users" DROP COLUMN "name";
        RAISE NOTICE 'Campo name eliminado';
    ELSE
        RAISE NOTICE 'Campo name no existe, no se elimina';
    END IF;

    -- 8. Crear índices si no existen
    IF NOT EXISTS (
        SELECT 1 
        FROM pg_indexes 
        WHERE schemaname = 'public' 
        AND tablename = 'users' 
        AND indexname = 'idx_users_first_name'
    ) THEN
        CREATE INDEX "idx_users_first_name" ON "public"."users" ("first_name");
        RAISE NOTICE 'Índice idx_users_first_name creado';
    END IF;

    IF NOT EXISTS (
        SELECT 1 
        FROM pg_indexes 
        WHERE schemaname = 'public' 
        AND tablename = 'users' 
        AND indexname = 'idx_users_last_name'
    ) THEN
        CREATE INDEX "idx_users_last_name" ON "public"."users" ("last_name");
        RAISE NOTICE 'Índice idx_users_last_name creado';
    END IF;

END $$;

