-- =====================================================
-- Script de Configuración Inicial de la Base de Datos
-- Versión: 1.1.0
-- Fecha: 19 de diciembre de 2025
-- Descripción: Crea la estructura y los datos mínimos
--              para que el sistema sea funcional.
-- =====================================================

-- Eliminar tablas existentes (en orden inverso por dependencias)
DROP TABLE IF EXISTS evidences CASCADE;
DROP TABLE IF EXISTS contracted_services CASCADE;
DROP TABLE IF EXISTS company_modules CASCADE;
DROP TABLE IF EXISTS services_catalog CASCADE;
DROP TABLE IF EXISTS employees CASCADE;
DROP TABLE IF EXISTS clients CASCADE;
DROP TABLE IF EXISTS users CASCADE;
DROP TABLE IF EXISTS modules CASCADE;
DROP TABLE IF EXISTS system_config CASCADE;
DROP TABLE IF EXISTS companies CASCADE;

-- =====================================================
-- ESTRUCTURA DE TABLAS
-- =====================================================

CREATE TABLE companies (
    id SERIAL PRIMARY KEY,
    name VARCHAR(255) NOT NULL,
    logo_url TEXT,
    slogan VARCHAR(500),
    street VARCHAR(255),
    exterior_number VARCHAR(50),
    interior_number VARCHAR(50),
    neighborhood VARCHAR(255),
    municipality VARCHAR(255),
    state VARCHAR(100),
    postal_code VARCHAR(10),
    phone VARCHAR(20),
    email VARCHAR(255),
    service_completion_message TEXT,
    max_clients INTEGER DEFAULT 0,
    max_employees INTEGER DEFAULT 0,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);


CREATE TABLE system_config (
    id SERIAL PRIMARY KEY,
    config_key VARCHAR(100) UNIQUE NOT NULL,
    config_value TEXT,
    description VARCHAR(500),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE TABLE modules (
    id SERIAL PRIMARY KEY,
    name VARCHAR(255) NOT NULL,
    description TEXT,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE TABLE users (
    id VARCHAR(50) PRIMARY KEY,
    company_id INTEGER REFERENCES companies(id) ON DELETE CASCADE,
    email VARCHAR(255) UNIQUE NOT NULL,
    password_hash VARCHAR(255) NOT NULL,
    role VARCHAR(20) NOT NULL CHECK (role IN ('ROOT', 'ADMIN', 'EMPLOYEE', 'CLIENT')),
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE TABLE clients (
    id SERIAL PRIMARY KEY,
    user_id VARCHAR(50) UNIQUE REFERENCES users(id) ON DELETE CASCADE,
    company_id INTEGER REFERENCES companies(id) ON DELETE CASCADE,
    client_number VARCHAR(50) UNIQUE NOT NULL,
    name VARCHAR(255) NOT NULL,
    phone VARCHAR(20),
    fiscal_name VARCHAR(255),
    rfc VARCHAR(13),
    fiscal_address TEXT,
    address TEXT,
    alternate_address TEXT,
    email VARCHAR(255),
    is_active BOOLEAN DEFAULT TRUE,
    allow_access_without_service BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE TABLE employees (
    id SERIAL PRIMARY KEY,
    user_id VARCHAR(50) UNIQUE REFERENCES users(id) ON DELETE CASCADE,
    company_id INTEGER REFERENCES companies(id) ON DELETE CASCADE,
    name VARCHAR(255) NOT NULL,
    phone VARCHAR(20),
    specialty VARCHAR(255),
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE TABLE services_catalog (
    id SERIAL PRIMARY KEY,
    company_id INTEGER REFERENCES companies(id) ON DELETE CASCADE,
    name VARCHAR(255) NOT NULL,
    description TEXT,
    category VARCHAR(100),
    max_time_hours INTEGER,
    has_warranty BOOLEAN DEFAULT FALSE,
    warranty_months INTEGER,
    warranty_exceptions TEXT,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE TABLE contracted_services (
    id SERIAL PRIMARY KEY,
    company_id INTEGER REFERENCES companies(id) ON DELETE CASCADE,
    client_id VARCHAR(50) REFERENCES users(id) ON DELETE CASCADE,
    service_catalogo_id INTEGER REFERENCES services_catalog(id) ON DELETE CASCADE,
    employee_id INTEGER REFERENCES employees(id) ON DELETE SET NULL,
    service_number VARCHAR(50) UNIQUE NOT NULL,
    service_address TEXT,
    request_date TIMESTAMP NOT NULL,
    deadline_date TIMESTAMP,
    start_date TIMESTAMP,
    end_date TIMESTAMP,
    closed_at TIMESTAMP,
    warranty_end_date TIMESTAMP,
    payment_reference VARCHAR(100),
    payment_status VARCHAR(20) DEFAULT 'PENDING',
    payment_proof_url TEXT,
    status VARCHAR(20) NOT NULL CHECK (status IN ('PENDING', 'IN_PROGRESS', 'FINISHED', 'CANCELLED', 'CERRADO_ADMIN', 'IN_REVIEW')),
    employee_report TEXT,
    client_rating INTEGER CHECK (client_rating >= 1 AND client_rating <= 5),
    client_comment TEXT,
    finalized_by_client BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE TABLE evidences (
    id SERIAL PRIMARY KEY,
    contracted_service_id INTEGER REFERENCES contracted_services(id) ON DELETE CASCADE,
    evidence_type VARCHAR(20) CHECK (evidence_type IN ('PHOTO', 'VIDEO', 'DOCUMENT')),
    file_url TEXT NOT NULL,
    description TEXT,
    uploaded_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE TABLE company_modules (
    id SERIAL PRIMARY KEY,
    company_id INTEGER REFERENCES companies(id) ON DELETE CASCADE,
    module_id INTEGER REFERENCES modules(id) ON DELETE CASCADE,
    is_enabled BOOLEAN DEFAULT TRUE,
    enabled_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE(company_id, module_id)
);

CREATE TABLE notifications (
    id SERIAL PRIMARY KEY,
    user_id VARCHAR(50) REFERENCES users(id) ON DELETE CASCADE,
    type VARCHAR(50) NOT NULL,
    title VARCHAR(255) NOT NULL,
    message TEXT,
    related_service_id INTEGER REFERENCES contracted_services(id) ON DELETE CASCADE,
    is_read BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- =====================================================
-- ÍNDICES PARA OPTIMIZACIÓN
-- =====================================================
CREATE INDEX idx_users_company ON users(company_id);
CREATE INDEX idx_users_email ON users(email);
CREATE INDEX idx_users_role ON users(role);
CREATE INDEX idx_clients_company ON clients(company_id);
CREATE INDEX idx_clients_number ON clients(client_number);
CREATE INDEX idx_employees_company ON employees(company_id);
CREATE INDEX idx_services_catalog_company ON services_catalog(company_id);
CREATE INDEX idx_contracted_services_company ON contracted_services(company_id);
CREATE INDEX idx_contracted_services_client ON contracted_services(client_id);
CREATE INDEX idx_contracted_services_employee ON contracted_services(employee_id);
CREATE INDEX idx_contracted_services_status ON contracted_services(status);
CREATE INDEX idx_evidences_service ON evidences(contracted_service_id);


-- =====================================================
-- DATOS DE CATÁLOGOS
-- =====================================================

INSERT INTO system_config (config_key, config_value, description) VALUES
('footer_text', '© 2025 KeMarketing - Sistema de Gestión de Servicios', 'Texto del pie de página global'),
('footer_link_text', 'Soporte Técnico', 'Texto del enlace en el pie de página'),
('footer_link_url', 'https://kemarketing.mx/soporte', 'URL del enlace en el pie de página');

INSERT INTO modules (id, name, description) VALUES
(1, 'Gestión de Clientes', 'Administración de clientes y sus datos'),
(2, 'Catálogo de Servicios', 'Gestión del catálogo de servicios ofrecidos'),
(3, 'Órdenes de Servicio', 'Gestión de órdenes de trabajo'),
(4, 'Gestión de Empleados', 'Administración de personal técnico'),
(5, 'Reportes', 'Generación de reportes y estadísticas');

-- =====================================================
-- DATOS MÍNIMOS FUNCIONALES
-- =====================================================

-- 1. Usuario ROOT
INSERT INTO users (id, company_id, email, password_hash, role, is_active) VALUES
('usr-root', NULL, 'hola@kemarketing.mx', '$2a$10$YjpIDhLLzz05H4OpxptIeuFAy46CKps6fWDcNn7oUkTd.zz8/tud.', 'ROOT', true);

-- 2. Empresa de Ejemplo
INSERT INTO companies (id, name, email, is_active) VALUES
(1, 'Mi Empresa de Servicios', 'contacto@miempresa.com', true);

-- 3. Módulos para la Empresa de Ejemplo (todos habilitados)
INSERT INTO company_modules (company_id, module_id, is_enabled) VALUES
(1, 1, true),
(1, 2, true),
(1, 3, true),
(1, 4, true),
(1, 5, true);

-- 4. Usuario Administrador de la Empresa
INSERT INTO users (id, company_id, email, password_hash, role, is_active) VALUES
('usr-admin-01', 1, 'admin@miempresa.com', '$2a$10$YjpIDhLLzz05H4OpxptIeuFAy46CKps6fWDcNn7oUkTd.zz8/tud.', 'ADMIN', true);

-- 5. Empleado de la Empresa
INSERT INTO users (id, company_id, email, password_hash, role, is_active) VALUES
('usr-emp-01', 1, 'empleado@miempresa.com', '$2a$10$YjpIDhLLzz05H4OpxptIeuFAy46CKps6fWDcNn7oUkTd.zz8/tud.', 'EMPLOYEE', true);
INSERT INTO employees (user_id, company_id, name, phone, specialty, is_active) VALUES
('usr-emp-01', 1, 'Juan Empleado', '5512345678', 'Técnico General', true);

-- 6. Cliente de la Empresa
INSERT INTO users (id, company_id, email, password_hash, role, is_active) VALUES
('usr-cli-01', 1, 'cliente@email.com', '$2a$10$YjpIDhLLzz05H4OpxptIeuFAy46CKps6fWDcNn7oUkTd.zz8/tud.', 'CLIENT', true);
INSERT INTO clients (user_id, company_id, client_number, name, phone) VALUES
('usr-cli-01', 1, 'CLI-2025-001', 'Ana Cliente', '5587654321');

-- =====================================================
-- REINICIO DE SECUENCIAS
-- =====================================================
SELECT setval('companies_id_seq', (SELECT MAX(id) FROM companies));
SELECT setval('modules_id_seq', (SELECT MAX(id) FROM modules));
SELECT setval('clients_id_seq', (SELECT MAX(id) FROM clients));
SELECT setval('employees_id_seq', (SELECT MAX(id) FROM employees));

-- =====================================================
-- FIN DEL SCRIPT
-- =====================================================
