import { useState } from 'react';
import {
  Box,
  Flex,
  Text,
  Button,
  SimpleGrid,
  Badge,
  Modal,
  ModalOverlay,
  ModalContent,
  ModalHeader,
  ModalBody,
  ModalFooter,
  ModalCloseButton,
  useDisclosure,
  VStack,
  HStack,
  Divider,
  Card,
  CardBody,
  Heading,
  ButtonGroup,
  Icon,
  Grid,
  GridItem,
  Tooltip
} from '@chakra-ui/react';
import { BiCalendar, BiTime, BiGridAlt } from 'react-icons/bi';
import { useNavigate } from 'react-router-dom';

const Calendar = ({ clases = [], bitacoras = [], alumnos = [], onDayClick, userType = 'admin' }) => {
  const [currentDate, setCurrentDate] = useState(new Date());
  const [selectedDay, setSelectedDay] = useState(null);
  const [selectedClase, setSelectedClase] = useState(null);
  const [viewMode, setViewMode] = useState('month'); // 'month' | 'week'
  const { isOpen, onOpen, onClose } = useDisclosure();
  const { isOpen: isDetailOpen, onOpen: onDetailOpen, onClose: onDetailClose } = useDisclosure();
  const navigate = useNavigate();

  const year = currentDate.getFullYear();
  const month = currentDate.getMonth();

  // Obtener primer día del mes y cantidad de días
  const firstDay = new Date(year, month, 1);
  const lastDay = new Date(year, month + 1, 0);
  const daysInMonth = lastDay.getDate();
  const startingDayOfWeek = firstDay.getDay();

  // Días de la semana
  const weekDays = ['Dom', 'Lun', 'Mar', 'Mié', 'Jue', 'Vie', 'Sáb'];
  const weekDaysFull = ['Domingo', 'Lunes', 'Martes', 'Miércoles', 'Jueves', 'Viernes', 'Sábado'];
  const monthNames = [
    'Enero', 'Febrero', 'Marzo', 'Abril', 'Mayo', 'Junio',
    'Julio', 'Agosto', 'Septiembre', 'Octubre', 'Noviembre', 'Diciembre'
  ];

  // Mapeo de días de la semana en español completo a abreviado
  const dayNameMap = {
    'Lunes': 'Lun',
    'Martes': 'Mar',
    'Miércoles': 'Mié',
    'Jueves': 'Jue',
    'Viernes': 'Vie',
    'Sábado': 'Sáb',
    'Domingo': 'Dom'
  };

  // Colores para diferentes materias
  const materiaColors = [
    { bg: 'blue.500', border: 'blue.600' },
    { bg: 'purple.500', border: 'purple.600' },
    { bg: 'green.500', border: 'green.600' },
    { bg: 'orange.500', border: 'orange.600' },
    { bg: 'pink.500', border: 'pink.600' },
    { bg: 'teal.500', border: 'teal.600' },
  ];

  const getMateriaColor = (materiaId) => {
    const index = materiaId ? (materiaId - 1) % materiaColors.length : 0;
    return materiaColors[index];
  };

  // Obtener clases para un día específico
  const getClasesForDay = (day) => {
    const dateStr = `${year}-${String(month + 1).padStart(2, '0')}-${String(day).padStart(2, '0')}`;
    const dayOfWeek = new Date(year, month, day).getDay();
    const dayNameShort = weekDays[dayOfWeek];

    return clases.filter(clase => {
      const claseDay = clase.dia_semana;
      return claseDay === dayNameShort ||
        claseDay === weekDays[dayOfWeek] ||
        dayNameMap[claseDay] === dayNameShort ||
        clase.fecha === dateStr;
    });
  };

  // Obtener clases para un día de la semana específico (para vista semanal)
  const getClasesForWeekday = (dayName) => {
    return clases.filter(clase => {
      const claseDay = clase.dia_semana;
      return claseDay === dayName || dayNameMap[claseDay] === dayName;
    });
  };

  // Verificar si es hoy
  const isToday = (day) => {
    const today = new Date();
    return (
      day === today.getDate() &&
      month === today.getMonth() &&
      year === today.getFullYear()
    );
  };

  // Manejar clic en día
  const handleDayClick = (day) => {
    const dayClases = getClasesForDay(day);
    setSelectedDay({ day, clases: dayClases });
    if (onDayClick) {
      onDayClick({ day, clases: dayClases });
    }
    onOpen();
  };

  // Navegar al mes anterior
  const goToPreviousMonth = () => {
    setCurrentDate(new Date(year, month - 1, 1));
  };

  // Navegar al mes siguiente
  const goToNextMonth = () => {
    setCurrentDate(new Date(year, month + 1, 1));
  };

  // Ir a hoy
  const goToToday = () => {
    setCurrentDate(new Date());
  };

  // Generar días del calendario
  const calendarDays = [];

  // Días vacíos al inicio
  for (let i = 0; i < startingDayOfWeek; i++) {
    calendarDays.push(null);
  }

  // Días del mes
  for (let day = 1; day <= daysInMonth; day++) {
    calendarDays.push(day);
  }

  // Horas para la vista semanal (de 8 AM a 8 PM)
  const hours = Array.from({ length: 13 }, (_, i) => i + 8);

  // Vista Semanal
  const WeekView = () => {
    return (
      <Box overflowX="auto" bg="white" borderRadius="xl" border="1px solid" borderColor="gray.200">
        <Grid templateColumns="80px repeat(7, 1fr)" gap={0}>
          {/* Header con días */}
          <GridItem bg="gray.50" p={3} borderRight="1px" borderBottom="2px" borderColor="gray.200">
            <Text fontSize="xs" fontWeight="bold" color="gray.500">HORA</Text>
          </GridItem>
          {weekDaysFull.map((day, idx) => {
            const isCurrentDay = new Date().getDay() === idx;
            return (
              <GridItem
                key={day}
                bg={isCurrentDay ? 'blue.50' : 'gray.50'}
                p={3}
                borderRight="1px"
                borderBottom="2px"
                borderColor="gray.200"
                textAlign="center"
              >
                <Text
                  fontSize="xs"
                  fontWeight="bold"
                  color={isCurrentDay ? 'blue.700' : 'gray.700'}
                  textTransform="uppercase"
                  letterSpacing="wider"
                >
                  {day}
                </Text>
              </GridItem>
            );
          })}

          {/* Grid de horas */}
          {hours.map(hour => (
            <>
              <GridItem
                key={`time-${hour}`}
                bg="gray.50"
                p={2}
                borderRight="1px"
                borderBottom="1px"
                borderColor="gray.200"
                textAlign="center"
              >
                <Text fontSize="xs" fontWeight="medium" color="gray.600">
                  {hour}:00
                </Text>
              </GridItem>
              {weekDaysFull.map(day => {
                const dayShort = weekDays[weekDaysFull.indexOf(day)];
                const dayClases = getClasesForWeekday(dayShort).filter(clase => {
                  const claseHour = parseInt(clase.hora_inicio);
                  return claseHour === hour;
                });

                return (
                  <GridItem
                    key={`${day}-${hour}`}
                    borderRight="1px"
                    borderBottom="1px"
                    borderColor="gray.200"
                    p={1}
                    minH="60px"
                    bg="white"
                    position="relative"
                  >
                    {dayClases.map((clase, idx) => {
                      const colors = getMateriaColor(clase.id_materia);
                      return (
                        <Tooltip
                          key={idx}
                          label={`${clase.materia?.nombre || 'Clase'} - ${clase.alumno?.nombre || 'Alumno'} - $${clase.materia?.precio || '0'}`}
                          hasArrow
                        >
                          <Box
                            bg={colors.bg}
                            borderLeft="3px solid"
                            borderColor={colors.border}
                            borderRadius="md"
                            p={2}
                            mb={1}
                            cursor="pointer"
                            onClick={() => {
                              setSelectedClase(clase);
                              onDetailOpen();
                            }}
                            _hover={{
                              transform: 'scale(1.02)',
                              boxShadow: 'md'
                            }}
                            transition="all 0.2s"
                          >
                            <Text fontSize="xs" fontWeight="bold" color="white" noOfLines={1}>
                              {clase.hora_inicio}
                            </Text>
                            <Text fontSize="xs" color="whiteAlpha.900" noOfLines={1}>
                              {clase.materia?.nombre || 'Clase'}
                            </Text>
                            <Text fontSize="xs" color="whiteAlpha.800" noOfLines={1}>
                              {clase.alumno?.nombre || 'Alumno'}
                            </Text>
                          </Box>
                        </Tooltip>
                      );
                    })}
                  </GridItem>
                );
              })}
            </>
          ))}
        </Grid>
      </Box>
    );
  };

  // Vista Mensual Mejorada
  const MonthView = () => {
    return (
      <>
        {/* Días de la semana */}
        <SimpleGrid columns={7} spacing={2} mb={3}>
          {weekDays.map((day) => (
            <Box
              key={day}
              textAlign="center"
              py={3}
              fontWeight="700"
              color="gray.600"
              fontSize="13px"
              textTransform="uppercase"
              letterSpacing="1px"
            >
              {day}
            </Box>
          ))}
        </SimpleGrid>

        {/* Grid del calendario */}
        <SimpleGrid columns={7} spacing={3}>
          {calendarDays.map((day, index) => {
            if (day === null) {
              return <Box key={index} h="130px" />;
            }

            const dayClases = getClasesForDay(day);
            const today = isToday(day);

            return (
              <Box
                key={day}
                h="130px"
                border="1px solid"
                borderColor={today ? 'blue.400' : 'gray.200'}
                borderRadius="lg"
                p={2}
                cursor="pointer"
                bg={today ? 'blue.50' : 'white'}
                _hover={{
                  bg: today ? 'blue.100' : 'gray.50',
                  borderColor: 'blue.500',
                  transform: 'translateY(-2px)',
                  boxShadow: 'lg'
                }}
                position="relative"
                onClick={() => handleDayClick(day)}
                transition="all 0.2s"
              >
                {/* Número del día */}
                <Flex justify="space-between" align="center" mb={2}>
                  <Box
                    minW="26px"
                    h="26px"
                    display="flex"
                    alignItems="center"
                    justifyContent="center"
                    borderRadius="full"
                    bg={today ? 'blue.500' : 'transparent'}
                    color={today ? 'white' : 'gray.700'}
                    fontWeight="700"
                    fontSize="13px"
                  >
                    {day}
                  </Box>
                  {dayClases.length > 0 && (
                    <Badge
                      colorScheme="blue"
                      borderRadius="full"
                      fontSize="xs"
                      px={2}
                    >
                      {dayClases.length}
                    </Badge>
                  )}
                </Flex>

                {/* Indicadores de clases */}
                <VStack spacing={1} align="stretch">
                  {dayClases.slice(0, 2).map((clase, idx) => {
                    const colors = getMateriaColor(clase.id_materia);
                    return (
                      <Box
                        key={idx}
                        bg={colors.bg}
                        borderLeft="3px solid"
                        borderColor={colors.border}
                        color="white"
                        px={2}
                        py={1}
                        borderRadius="md"
                        fontSize="11px"
                        fontWeight="600"
                        overflow="hidden"
                        textOverflow="ellipsis"
                        whiteSpace="nowrap"
                        transition="all 0.2s"
                        _hover={{
                          transform: 'scale(1.05)'
                        }}
                      >
                        {clase.hora_inicio} {clase.materia?.nombre?.substring(0, 10)}
                      </Box>
                    );
                  })}
                  {dayClases.length > 2 && (
                    <Text fontSize="xs" color="gray.500" fontWeight="600" textAlign="center">
                      +{dayClases.length - 2}
                    </Text>
                  )}
                </VStack>
              </Box>
            );
          })}
        </SimpleGrid>
      </>
    );
  };

  return (
    <>
      <Box
        bg="white"
        borderRadius="xl"
        border="1px solid"
        borderColor="gray.200"
        p={6}
        boxShadow="sm"
      >
        {/* Header del Calendario */}
        <Flex justify="space-between" align="center" mb={6} flexWrap="wrap" gap={4}>
          <HStack spacing={3}>
            <Button
              size="sm"
              variant="ghost"
              onClick={goToPreviousMonth}
              borderRadius="lg"
              w="36px"
              h="36px"
              p={0}
              _hover={{
                bg: 'gray.100',
              }}
            >
              ‹
            </Button>
            <Heading size="lg" color="gray.800" fontWeight="700">
              {monthNames[month]} {year}
            </Heading>
            <Button
              size="sm"
              variant="ghost"
              onClick={goToNextMonth}
              borderRadius="lg"
              w="36px"
              h="36px"
              p={0}
              _hover={{
                bg: 'gray.100',
              }}
            >
              ›
            </Button>
            <Button
              size="sm"
              variant="outline"
              onClick={goToToday}
              borderRadius="lg"
              ml={2}
              colorScheme="blue"
            >
              Hoy
            </Button>
          </HStack>

          <ButtonGroup size="sm" isAttached variant="outline">
            <Button
              leftIcon={<Icon as={BiCalendar} />}
              colorScheme={viewMode === 'month' ? 'blue' : 'gray'}
              variant={viewMode === 'month' ? 'solid' : 'outline'}
              onClick={() => setViewMode('month')}
            >
              Mes
            </Button>
            <Button
              leftIcon={<Icon as={BiGridAlt} />}
              colorScheme={viewMode === 'week' ? 'blue' : 'gray'}
              variant={viewMode === 'week' ? 'solid' : 'outline'}
              onClick={() => setViewMode('week')}
            >
              Semana
            </Button>
          </ButtonGroup>
        </Flex>

        {/* Renderizar vista según modo */}
        {viewMode === 'month' ? <MonthView /> : <WeekView />}
      </Box>

      {/* Modal con resumen del día */}
      <Modal isOpen={isOpen} onClose={onClose} size="lg">
        <ModalOverlay backdropFilter="blur(4px)" />
        <ModalContent borderRadius="xl" boxShadow="xl">
          <ModalHeader
            pb={3}
            borderBottom="1px solid"
            borderColor="gray.200"
            fontWeight="700"
          >
            Clases del {selectedDay?.day} de {monthNames[month]}
          </ModalHeader>
          <ModalCloseButton />
          <ModalBody py={6}>
            {selectedDay?.clases && selectedDay.clases.length > 0 ? (
              <VStack align="stretch" spacing={3}>
                {selectedDay.clases.map((clase, idx) => {
                  const colors = getMateriaColor(clase.id_materia);
                  return (
                    <Card key={idx} borderLeft="4px solid" borderColor={colors.bg} bg="gray.50">
                      <CardBody p={4}>
                        <VStack align="stretch" spacing={3}>
                          <HStack justify="space-between">
                            <HStack>
                              <Icon as={BiTime} color={colors.bg} />
                              <Text fontWeight="700" fontSize="16px" color="gray.900">
                                {clase.hora_inicio} - {clase.hora_fin}
                              </Text>
                            </HStack>
                            <Badge colorScheme="blue" borderRadius="md" px={2} py={1}>
                              {clase.dia_semana}
                            </Badge>
                          </HStack>
                          <Divider />
                          <Box>
                            <Text fontSize="14px" color="gray.700" mb={1}>
                              <Text as="span" fontWeight="600">Materia:</Text> {clase.materia?.nombre || 'N/A'}
                            </Text>
                            <Text fontSize="14px" color="gray.700" mb={1}>
                              <Text as="span" fontWeight="600">Alumno:</Text> {clase.alumno?.nombre || 'N/A'}
                            </Text>
                            {clase.profesor && (
                              <Text fontSize="14px" color="gray.700">
                                <Text as="span" fontWeight="600">Profesor:</Text> {clase.profesor.nombre}
                              </Text>
                            )}
                          </Box>
                          <HStack spacing={2}>
                            <Button
                              size="sm"
                              colorScheme="blue"
                              onClick={() => {
                                setSelectedClase(clase);
                                onClose();
                                onDetailOpen();
                              }}
                            >
                              Ver Detalle
                            </Button>
                            {userType === 'admin' && (
                              <Button
                                size="sm"
                                variant="outline"
                                onClick={() => {
                                  onClose();
                                  if (clase.alumno?.id) {
                                    navigate(`/admin/alumnos/${clase.alumno.id}`);
                                  }
                                }}
                              >
                                Ver Ficha
                              </Button>
                            )}
                          </HStack>
                        </VStack>
                      </CardBody>
                    </Card>
                  );
                })}
              </VStack>
            ) : (
              <Text color="gray.500" textAlign="center" py={8}>
                No hay clases programadas para este día
              </Text>
            )}
          </ModalBody>
          <ModalFooter borderTop="1px solid" borderColor="gray.200">
            <Button onClick={onClose} variant="ghost">Cerrar</Button>
          </ModalFooter>
        </ModalContent>
      </Modal>

      {/* Modal con Detalle del Alumno y Bitácora */}
      <Modal isOpen={isDetailOpen} onClose={onDetailClose} size="xl">
        <ModalOverlay backdropFilter="blur(4px)" />
        <ModalContent borderRadius="xl" boxShadow="xl">
          <ModalHeader
            pb={3}
            borderBottom="1px solid"
            borderColor="gray.200"
            fontWeight="700"
          >
            Detalle de {selectedClase?.alumno?.nombre || 'Alumno'}
          </ModalHeader>
          <ModalCloseButton />
          <ModalBody py={6}>
            {selectedClase && (
              <VStack align="stretch" spacing={4}>
                {/* Información de la Clase */}
                <Card border="1px solid" borderColor="gray.200">
                  <CardBody p={5}>
                    <Heading size="sm" mb={4} color="gray.900" fontWeight="700">
                      Información de la Clase
                    </Heading>
                    <VStack align="stretch" spacing={2}>
                      <Text fontSize="14px" color="gray.700">
                        <Text as="span" fontWeight="600">Fecha:</Text> {selectedDay?.day} de {monthNames[month]} de {year}
                      </Text>
                      <Text fontSize="14px" color="gray.700">
                        <Text as="span" fontWeight="600">Hora:</Text> {selectedClase.hora_inicio} - {selectedClase.hora_fin}
                      </Text>
                      <Text fontSize="14px" color="gray.700">
                        <Text as="span" fontWeight="600">Materia:</Text> {selectedClase.materia?.nombre || 'N/A'}
                      </Text>
                      <Text fontSize="14px" color="gray.700">
                        <Text as="span" fontWeight="600">Costo Mensual:</Text> ${selectedClase.materia?.precio || '0.00'}
                      </Text>
                      <Text fontSize="14px" color="gray.700">
                        <Text as="span" fontWeight="600">Periodo de Pago:</Text> {selectedClase.materia?.periodo_pago || 'N/A'}
                      </Text>
                      {selectedClase.profesor && (
                        <Text fontSize="14px" color="gray.700">
                          <Text as="span" fontWeight="600">Profesor:</Text> {selectedClase.profesor.nombre}
                        </Text>
                      )}
                    </VStack>
                  </CardBody>
                </Card>

                {/* Bitácora del Día */}
                {(() => {
                  const dateStr = `${year}-${String(month + 1).padStart(2, '0')}-${String(selectedDay?.day).padStart(2, '0')}`;
                  const bitacoraDelDia = bitacoras.filter(b =>
                    b.id_alumno === selectedClase.alumno?.id &&
                    b.fecha === dateStr
                  );

                  return bitacoraDelDia.length > 0 ? (
                    <Card border="1px solid" borderColor="gray.200">
                      <CardBody p={5}>
                        <Heading size="sm" mb={4} color="gray.900" fontWeight="700">
                          Bitácora del Día
                        </Heading>
                        {bitacoraDelDia.map((bitacora, idx) => (
                          <VStack key={idx} align="stretch" spacing={3}>
                            <Box>
                              <Text fontWeight="600" mb={2} fontSize="14px" color="gray.900">Objetivo:</Text>
                              <Text fontSize="14px" color="gray.700" lineHeight="1.6">{bitacora.objetivo}</Text>
                            </Box>
                            <Divider />
                            <Box>
                              <Text fontWeight="600" mb={2} fontSize="14px" color="gray.900">Actividad:</Text>
                              <Text fontSize="14px" color="gray.700" lineHeight="1.6">{bitacora.actividad}</Text>
                            </Box>
                            <Divider />
                            <Box>
                              <Text fontWeight="600" mb={2} fontSize="14px" color="gray.900">Evaluación:</Text>
                              <Text fontSize="14px" color="gray.700" lineHeight="1.6">{bitacora.evaluacion}</Text>
                            </Box>
                            {bitacora.tarea && (
                              <>
                                <Divider />
                                <Box>
                                  <Text fontWeight="600" mb={2} fontSize="14px" color="gray.900">Tarea:</Text>
                                  <Text fontSize="14px" color="gray.700" lineHeight="1.6">{bitacora.tarea}</Text>
                                </Box>
                              </>
                            )}
                          </VStack>
                        ))}
                      </CardBody>
                    </Card>
                  ) : (
                    <Card border="1px solid" borderColor="gray.200">
                      <CardBody p={5}>
                        <Text color="gray.500" textAlign="center" py={4}>
                          No hay bitácora registrada para este día
                        </Text>
                      </CardBody>
                    </Card>
                  );
                })()}
              </VStack>
            )}
          </ModalBody>
          <ModalFooter borderTop="1px solid" borderColor="gray.200">
            {selectedClase?.alumno?.id && userType === 'admin' && (
              <Button
                colorScheme="blue"
                onClick={() => {
                  onDetailClose();
                  navigate(`/admin/alumnos/${selectedClase.alumno.id}`);
                }}
                mr={2}
              >
                Ver Ficha Completa
              </Button>
            )}
            <Button onClick={onDetailClose} variant="ghost">Cerrar</Button>
          </ModalFooter>
        </ModalContent>
      </Modal>
    </>
  );
};

export default Calendar;
