import {
  Box,
  Flex,
  HStack,
  VStack,
  Heading,
  Text,
  Button,
  Icon,
  Popover,
  PopoverTrigger,
  PopoverContent,
  PopoverHeader,
  PopoverBody,
  PopoverFooter,
  PopoverArrow,
  PopoverCloseButton,
  Portal,
  Badge,
  Divider,
  Spinner,
  IconButton,
  useBreakpointValue
} from '@chakra-ui/react';
import { useState, useEffect } from 'react';
import { useLocation } from 'react-router-dom';
import { FiMenu, FiBell, FiCheck, FiInfo, FiAlertTriangle, FiAlertCircle } from 'react-icons/fi';
import Sidebar from './Sidebar';
import { getNotifications, markNotificationAsRead, markAllNotificationsAsRead } from '../services/api';

const Layout = ({ links, userType, userName, userPhoto, children }) => {
  const isMobile = useBreakpointValue({ base: true, md: false });
  const location = useLocation();
  const [isSidebarOpen, setIsSidebarOpen] = useState(true);
  const [notifications, setNotifications] = useState([]);
  const [loadingNotifications, setLoadingNotifications] = useState(false);
  const unreadCount = notifications.filter(n => !n.is_read).length;

  const isExamView = location.pathname.includes('/student/examen/');

  const fetchNotifications = async () => {
    try {
      const data = await getNotifications();
      if (Array.isArray(data)) {
        setNotifications(data);
      }
    } catch (error) {
      // No mostrar error si es de autenticación (401/403) - se redirige automáticamente al login
      if (error.message && (error.message.includes('Invalid token') || error.message.includes('403') || error.message.includes('401'))) {
        return; // La redirección al login se maneja en api.js
      }
      console.error('Error fetching notifications:', error);
    }
  };

  useEffect(() => {
    fetchNotifications();
    const interval = setInterval(fetchNotifications, 60000); // Polling cada 60s
    return () => clearInterval(interval);
  }, []);

  const handleMarkAsRead = async (id) => {
    try {
      await markNotificationAsRead(id);
      setNotifications(notifications.map(n => n.id === id ? { ...n, is_read: true } : n));
    } catch (error) {
      console.error('Error marking as read:', error);
    }
  };

  const handleMarkAllAsRead = async () => {
    try {
      await markAllNotificationsAsRead();
      setNotifications(notifications.map(n => ({ ...n, is_read: true })));
    } catch (error) {
      console.error('Error marking all as read:', error);
    }
  };

  const getIcon = (type) => {
    switch (type) {
      case 'ASSIGNED': return FiInfo;
      case 'REOPEN': return FiAlertCircle;
      case 'EXPIRING': return FiAlertTriangle;
      default: return FiBell;
    }
  };

  const getColor = (type) => {
    switch (type) {
      case 'ASSIGNED': return 'blue.500';
      case 'REOPEN': return 'orange.500';
      case 'EXPIRING': return 'red.500';
      default: return 'gray.500';
    }
  };

  useEffect(() => {
    if (isMobile) {
      setIsSidebarOpen(false);
    } else {
      setIsSidebarOpen(true);
    }
  }, [isMobile]);

  const toggleSidebar = () => {
    setIsSidebarOpen(!isSidebarOpen);
  };

  const handleSidebarClose = () => {
    if (isMobile) {
      setIsSidebarOpen(false);
    }
  };

  // Determinar título de la página basado en la ruta
  const getPageTitle = () => {
    const currentLink = links.find(link => link.path === location.pathname);
    return currentLink?.nombre || 'Dashboard';
  };

  return (
    <Flex>
      {!isExamView && (
        <Sidebar
          links={links}
          userType={userType}
          userName={userName}
          userPhoto={userPhoto}
          isOpen={isSidebarOpen}
          onClose={handleSidebarClose}
        />
      )}
      <Box
        ml={!isExamView && isSidebarOpen && !isMobile ? "260px" : "0"}
        w={!isExamView && isSidebarOpen && !isMobile ? "calc(100% - 260px)" : "100%"}
        minH="100vh"
        bg="#F8FAFC"
        transition="all 0.3s ease-in-out"
      >
        {/* Topbar */}
        {!isExamView && (
          <Box
            borderBottom="1px solid"
            borderColor="gray.200"
            px={8}
            py={6}
            position="sticky"
            top="0"
            zIndex={100}
            backdropFilter="blur(10px)"
            bg="rgba(255, 255, 255, 0.95)"
            boxShadow="0 1px 3px 0 rgba(0, 0, 0, 0.1), 0 1px 2px 0 rgba(0, 0, 0, 0.06)"
            transition="all 0.2s ease-in-out"
          >
            <Flex justify="space-between" align="center">
              <HStack spacing={4}>
                <Button
                  onClick={toggleSidebar}
                  variant="ghost"
                  size="md"
                  fontSize="24px"
                  aria-label="Toggle Sidebar"
                >
                  <Icon as={FiMenu} />
                </Button>
                <Heading
                  size="xl"
                  color="gray.900"
                  fontWeight="700"
                  letterSpacing="-0.5px"
                  fontSize="24px"
                >
                  {getPageTitle()}
                </Heading>
              </HStack>
              <HStack spacing={3}>
                <Popover placement="bottom-end">
                  <PopoverTrigger>
                    <Button
                      variant="ghost"
                      size="md"
                      w="44px"
                      h="44px"
                      borderRadius="10px"
                      fontSize="20px"
                      _hover={{
                        bg: 'gray.100',
                        transform: 'scale(1.05)',
                      }}
                      position="relative"
                      transition="all 0.2s ease-in-out"
                    >
                      <Icon as={FiBell} />
                      {unreadCount > 0 && (
                        <Box
                          position="absolute"
                          top="10px"
                          right="10px"
                          w="8px"
                          h="8px"
                          bg="red.500"
                          borderRadius="full"
                          border="2px solid white"
                          boxShadow="0 0 0 2px rgba(239, 68, 68, 0.2)"
                          sx={{
                            "@keyframes pulse": {
                              "0%": { transform: "scale(0.95)", boxShadow: "0 0 0 0 rgba(239, 68, 68, 0.7)" },
                              "70%": { transform: "scale(1)", boxShadow: "0 0 0 10px rgba(239, 68, 68, 0)" },
                              "100%": { transform: "scale(0.95)", boxShadow: "0 0 0 0 rgba(239, 68, 68, 0)" }
                            }
                          }}
                          animation="pulse 2s cubic-bezier(0.4, 0, 0.6, 1) infinite"
                        />
                      )}
                    </Button>
                  </PopoverTrigger>
                  <Portal>
                    <PopoverContent w="350px" boxShadow="xl" borderRadius="lg">
                      <PopoverArrow />
                      <PopoverCloseButton />
                      <PopoverHeader fontWeight="bold" border="0" pt={4} display="flex" justifyContent="space-between" alignItems="center">
                        <Text>Notificaciones</Text>
                        {unreadCount > 0 && (
                          <Button size="xs" colorScheme="brand" variant="ghost" onClick={handleMarkAllAsRead}>
                            Marcar todas leídas
                          </Button>
                        )}
                      </PopoverHeader>
                      <PopoverBody maxH="400px" overflowY="auto" px={2}>
                        {notifications.length === 0 ? (
                          <Box py={10} textAlign="center">
                            <Icon as={FiBell} fontSize="3xl" color="gray.300" mb={2} />
                            <Text color="gray.500">No tienes notificaciones</Text>
                          </Box>
                        ) : (
                          <VStack align="stretch" spacing={0}>
                            {notifications.map((n) => (
                              <Box
                                key={n.id}
                                p={3}
                                borderRadius="md"
                                bg={n.is_read ? "transparent" : "blue.50"}
                                _hover={{ bg: "gray.50" }}
                                position="relative"
                                mb={1}
                              >
                                <HStack spacing={3} align="start">
                                  <Flex
                                    w="32px" h="32px"
                                    bg={getColor(n.type)}
                                    borderRadius="full"
                                    align="center" justify="center"
                                    color="white" flexShrink={0}
                                  >
                                    <Icon as={getIcon(n.type)} />
                                  </Flex>
                                  <Box flex="1">
                                    <Text fontSize="sm" fontWeight="bold">{n.title}</Text>
                                    <Text fontSize="xs" color="gray.600">{n.message}</Text>
                                    <Text fontSize="10px" color="gray.400" mt={1}>
                                      {new Date(n.created_at).toLocaleString()}
                                    </Text>
                                  </Box>
                                  {!n.is_read && (
                                    <IconButton
                                      icon={<FiCheck />}
                                      size="xs"
                                      variant="ghost"
                                      onClick={() => handleMarkAsRead(n.id)}
                                      aria-label="Marcar como leída"
                                    />
                                  )}
                                </HStack>
                              </Box>
                            ))}
                          </VStack>
                        )}
                      </PopoverBody>
                    </PopoverContent>
                  </Portal>
                </Popover>
              </HStack>
            </Flex>
          </Box>
        )}

        {/* Content */}
        <Box p={isExamView ? 0 : 8} pb={0} minH={isExamView ? "100vh" : "calc(100vh - 80px)"} display="flex" flexDirection="column">
          <Box flex="1">
            {children}
          </Box>

          {/* Footer */}
          <Box
            mt={8}
            py={4}
            borderTop="1px solid"
            borderColor="gray.200"
            textAlign="center"
          >
            <Text fontSize="xs" color="gray.500">
              © {new Date().getFullYear()} Desarrollado por{' '}
              <Text as="span" fontWeight="600" color="brand.600">
                KeSoftware
              </Text>
              {' '}• Todos los derechos reservados
            </Text>
          </Box>
        </Box>
      </Box>
    </Flex>
  );
};

export default Layout;

