import React, { useState, useEffect } from 'react';
import {
    Box,
    Input,
    VStack,
    HStack,
    Text,
    Tag,
    TagLabel,
    TagCloseButton,
    List,
    ListItem,
    Avatar,
    InputGroup,
    InputLeftElement,
    FormControl,
    FormLabel,
    Radio,
    RadioGroup,
    Stack,
    useOutsideClick,
    useColorModeValue,
    Select,
    SimpleGrid,
    Checkbox
} from '@chakra-ui/react';
import { FaSearch, FaUserPlus, FaFilter } from 'react-icons/fa';

// Mock Data for search
const mockUsers = {
    profesores: [
        { id: 'p1', nombre: 'Juan Pérez', email: 'juan.perez@artschool.com', avatar: '' },
        { id: 'p2', nombre: 'María González', email: 'maria.gonzalez@artschool.com', avatar: '' },
        { id: 'p3', nombre: 'Carlos Rodríguez', email: 'carlos.rodriguez@artschool.com', avatar: '' },
        { id: 'p4', nombre: 'Ana Martínez', email: 'ana.martinez@artschool.com', avatar: '' },
        { id: 'p5', nombre: 'Luis Hernández', email: 'luis.hernandez@artschool.com', avatar: '' },
    ],
    padres: [
        { id: 'pa1', nombre: 'Roberto Sánchez', email: 'roberto.s@gmail.com', avatar: '' },
        { id: 'pa2', nombre: 'Laura Torres', email: 'laura.torres@hotmail.com', avatar: '' },
        { id: 'pa3', nombre: 'Miguel Ángel', email: 'miguel.angel@yahoo.com', avatar: '' },
        { id: 'pa4', nombre: 'Patricia Lima', email: 'paty.lima@gmail.com', avatar: '' },
    ],
    alumnos: [
        { id: 'a1', nombre: 'Sofía Sánchez', email: 'sofia.s@artschool.com', avatar: '' },
        { id: 'a2', nombre: 'Diego Torres', email: 'diego.t@artschool.com', avatar: '' },
        { id: 'a3', nombre: 'Valentina Ángel', email: 'valentina.a@artschool.com', avatar: '' },
        { id: 'a4', nombre: 'Camila Lima', email: 'camila.l@artschool.com', avatar: '' },
        { id: 'a5', nombre: 'Mateo Ruiz', email: 'mateo.ruiz@artschool.com', avatar: '' },
    ]
};

const mockMaterias = [
    { id: 'm1', nombre: 'Piano' },
    { id: 'm2', nombre: 'Guitarra' },
    { id: 'm3', nombre: 'Violín' },
    { id: 'm4', nombre: 'Canto' },
    { id: 'm5', nombre: 'Pintura' },
    { id: 'm6', nombre: 'Danza' },
];

const RecipientSelector = ({
    tipoUsuario,
    selectedRecipients,
    onSelectionChange,
    mode,
    onModeChange,
    filters,
    onFiltersChange
}) => {
    const [searchTerm, setSearchTerm] = useState('');
    const [searchResults, setSearchResults] = useState([]);
    const [isResultsOpen, setIsResultsOpen] = useState(false);
    const searchRef = React.useRef();

    const bgList = useColorModeValue('white', 'gray.700');
    const borderColor = useColorModeValue('gray.200', 'gray.600');

    useOutsideClick({
        ref: searchRef,
        handler: () => setIsResultsOpen(false),
    });

    useEffect(() => {
        if (searchTerm.trim().length > 0) {
            const users = mockUsers[tipoUsuario] || [];
            const filtered = users.filter(user =>
                !selectedRecipients.find(r => r.id === user.id) && // Exclude already selected
                (user.nombre.toLowerCase().includes(searchTerm.toLowerCase()) ||
                    user.email.toLowerCase().includes(searchTerm.toLowerCase()))
            );
            setSearchResults(filtered);
            setIsResultsOpen(true);
        } else {
            setSearchResults([]);
            setIsResultsOpen(false);
        }
    }, [searchTerm, tipoUsuario, selectedRecipients]);

    const handleSelectUser = (user) => {
        onSelectionChange([...selectedRecipients, user]);
        setSearchTerm('');
        setIsResultsOpen(false);
    };

    const handleRemoveUser = (userId) => {
        onSelectionChange(selectedRecipients.filter(u => u.id !== userId));
    };

    const handleFilterChange = (field, value) => {
        onFiltersChange({ ...filters, [field]: value });
    };

    return (
        <VStack align="stretch" spacing={4}>
            <FormControl>
                <FormLabel fontWeight="600" color="gray.600">Alcance del Envío</FormLabel>
                <RadioGroup onChange={onModeChange} value={mode}>
                    <Stack direction={{ base: 'column', md: 'row' }} spacing={5}>
                        <Radio value="masivo" colorScheme="brand">
                            <Text fontWeight={mode === 'masivo' ? 'bold' : 'normal'}>
                                Envío Masivo (Todos)
                            </Text>
                        </Radio>
                        <Radio value="filtro" colorScheme="brand">
                            <Text fontWeight={mode === 'filtro' ? 'bold' : 'normal'}>
                                Por Grupo/Filtro
                            </Text>
                        </Radio>
                        <Radio value="particular" colorScheme="brand">
                            <Text fontWeight={mode === 'particular' ? 'bold' : 'normal'}>
                                Particular (Selección manual)
                            </Text>
                        </Radio>
                    </Stack>
                </RadioGroup>
            </FormControl>

            {/* MODO FILTRO */}
            {mode === 'filtro' && (
                <Box
                    p={4}
                    bg="brand.50"
                    borderRadius="md"
                    border="1px dashed"
                    borderColor="brand.200"
                >
                    <VStack align="stretch" spacing={3}>
                        <HStack color="brand.600" fontWeight="bold" fontSize="sm">
                            <FaFilter />
                            <Text>Configurar Filtros</Text>
                        </HStack>

                        <SimpleGrid columns={{ base: 1, md: 2 }} spacing={4}>
                            <FormControl>
                                <FormLabel fontSize="sm">Materia / Actividad</FormLabel>
                                <Select
                                    bg="white"
                                    size="sm"
                                    placeholder="Todas las materias"
                                    value={filters?.materia || ''}
                                    onChange={(e) => handleFilterChange('materia', e.target.value)}
                                >
                                    {mockMaterias.map(m => (
                                        <option key={m.id} value={m.id}>{m.nombre}</option>
                                    ))}
                                </Select>
                            </FormControl>

                            {(tipoUsuario === 'alumnos' || tipoUsuario === 'padres') && (
                                <FormControl>
                                    <FormLabel fontSize="sm">Profesor Asignado</FormLabel>
                                    <Select
                                        bg="white"
                                        size="sm"
                                        placeholder="Todos los profesores"
                                        value={filters?.profesor || ''}
                                        onChange={(e) => handleFilterChange('profesor', e.target.value)}
                                    >
                                        {mockUsers.profesores.map(p => (
                                            <option key={p.id} value={p.id}>{p.nombre}</option>
                                        ))}
                                    </Select>
                                </FormControl>
                            )}

                            <FormControl>
                                <FormLabel fontSize="sm">Semana / Fecha de Clase</FormLabel>
                                <Input
                                    type="week"
                                    bg="white"
                                    size="sm"
                                    value={filters?.semana || ''}
                                    onChange={(e) => handleFilterChange('semana', e.target.value)}
                                />
                            </FormControl>

                            <FormControl display="flex" alignItems="center" mt={6}>
                                <Checkbox
                                    colorScheme="brand"
                                    isChecked={filters?.soloActivos}
                                    onChange={(e) => handleFilterChange('soloActivos', e.target.checked)}
                                >
                                    <Text fontSize="sm">Solo usuarios activos</Text>
                                </Checkbox>
                            </FormControl>
                        </SimpleGrid>

                        <Text fontSize="xs" color="gray.500" mt={2}>
                            * Se enviará a todos los {tipoUsuario} que coincidan con <strong>todos</strong> los filtros seleccionados.
                        </Text>
                    </VStack>
                </Box>
            )}

            {/* MODO PARTICULAR */}
            {mode === 'particular' && (
                <Box position="relative" ref={searchRef}>
                    <InputGroup size="md" mb={2}>
                        <InputLeftElement pointerEvents="none">
                            <FaSearch color="gray.300" />
                        </InputLeftElement>
                        <Input
                            placeholder={`Buscar ${tipoUsuario} por nombre o correo...`}
                            value={searchTerm}
                            onChange={(e) => setSearchTerm(e.target.value)}
                            onFocus={() => searchTerm.length > 0 && setIsResultsOpen(true)}
                            bg="white"
                        />
                    </InputGroup>

                    {/* Search Results Dropdown */}
                    {isResultsOpen && searchResults.length > 0 && (
                        <List
                            position="absolute"
                            top="100%"
                            left={0}
                            right={0}
                            zIndex={10}
                            bg={bgList}
                            boxShadow="lg"
                            borderRadius="md"
                            border="1px solid"
                            borderColor={borderColor}
                            maxH="200px"
                            overflowY="auto"
                        >
                            {searchResults.map((user) => (
                                <ListItem
                                    key={user.id}
                                    p={3}
                                    cursor="pointer"
                                    _hover={{ bg: 'brand.50' }}
                                    onClick={() => handleSelectUser(user)}
                                    borderBottom="1px solid"
                                    borderColor="gray.100"
                                >
                                    <HStack>
                                        <Avatar size="xs" name={user.nombre} src={user.avatar} />
                                        <VStack align="start" spacing={0}>
                                            <Text fontSize="sm" fontWeight="bold">{user.nombre}</Text>
                                            <Text fontSize="xs" color="gray.500">{user.email}</Text>
                                        </VStack>
                                        <Box flex={1} />
                                        <FaUserPlus color="gray.400" />
                                    </HStack>
                                </ListItem>
                            ))}
                        </List>
                    )}

                    {/* Selected Recipients Tags */}
                    <Box
                        minH="50px"
                        p={2}
                        border="1px dashed"
                        borderColor="gray.300"
                        borderRadius="md"
                        bg="gray.50"
                    >
                        {selectedRecipients.length === 0 ? (
                            <Text fontSize="sm" color="gray.400" textAlign="center" py={2}>
                                No hay destinatarios seleccionados
                            </Text>
                        ) : (
                            <HStack spacing={2} wrap="wrap">
                                {selectedRecipients.map((user) => (
                                    <Tag
                                        key={user.id}
                                        size="md"
                                        borderRadius="full"
                                        variant="solid"
                                        colorScheme="brand"
                                        mb={1}
                                    >
                                        <Avatar
                                            src={user.avatar}
                                            size="xs"
                                            name={user.nombre}
                                            ml={-1}
                                            mr={2}
                                        />
                                        <TagLabel>{user.nombre}</TagLabel>
                                        <TagCloseButton onClick={() => handleRemoveUser(user.id)} />
                                    </Tag>
                                ))}
                            </HStack>
                        )}
                    </Box>
                    <Text fontSize="xs" color="gray.500" mt={1} textAlign="right">
                        {selectedRecipients.length} destinatarios seleccionados
                    </Text>
                </Box>
            )}
        </VStack>
    );
};

export default RecipientSelector;
