import React from 'react';
import { Box, Grid, GridItem, Text, Tooltip, VStack, HStack, Avatar } from '@chakra-ui/react';
import { mockProfesores, mockMaterias } from '../../data/mockData';

const DAYS = ['Lunes', 'Martes', 'Miércoles', 'Jueves', 'Viernes', 'Sábado', 'Domingo'];
const HOURS = Array.from({ length: 13 }, (_, i) => i + 8); // 8:00 to 20:00

const WeeklyGrid = ({ existingClasses, onSlotClick, selectedDay, selectedHour }) => {

    const getSlotClasses = (day, hour) => {
        return existingClasses.filter(clase => {
            if (clase.dia_semana !== day) return false;
            const start = parseInt(clase.hora_inicio.split(':')[0]);
            const end = parseInt(clase.hora_fin.split(':')[0]);
            // Check if this hour is within the class duration
            // Note: This logic assumes classes start/end on the hour. 
            // If a class is 10:30 - 12:00, it covers 10 and 11.
            // For simplicity in this grid (which is hourly), we check if the hour is covered.
            return hour >= start && hour < end;
        });
    };

    const getProfesorName = (id) => {
        const prof = mockProfesores.find(p => p.id === id);
        return prof ? prof.nombre.split(' ')[0] : 'Prof.';
    };

    const getMateriaName = (id) => {
        const mat = mockMaterias.find(m => m.id === id);
        return mat ? mat.nombre : 'Clase';
    };

    return (
        <Box overflowX="auto" bg="white" p={4} borderRadius="xl" boxShadow="sm">
            <Grid templateColumns="80px repeat(7, 1fr)" gap={1} minW="1000px">
                {/* Header Row */}
                <GridItem>
                    <Box h="40px" display="flex" alignItems="center" justifyContent="center">
                        <Text fontSize="xs" fontWeight="bold" color="gray.500">HORA</Text>
                    </Box>
                </GridItem>
                {DAYS.map(day => (
                    <GridItem key={day}>
                        <Box h="40px" display="flex" alignItems="center" justifyContent="center" bg="gray.50" borderRadius="md">
                            <Text fontSize="sm" fontWeight="bold" color="gray.700">{day}</Text>
                        </Box>
                    </GridItem>
                ))}

                {/* Time Slots */}
                {HOURS.map(hour => (
                    <React.Fragment key={hour}>
                        {/* Time Label */}
                        <GridItem>
                            <Box h="80px" display="flex" alignItems="start" justifyContent="center" pt={2}>
                                <Text fontSize="xs" color="gray.400">{`${hour}:00`}</Text>
                            </Box>
                        </GridItem>

                        {/* Day Slots for this Hour */}
                        {DAYS.map(day => {
                            const classesInSlot = getSlotClasses(day, hour);
                            const isSelected = selectedDay === day && selectedHour === hour;

                            return (
                                <GridItem key={`${day}-${hour}`}>
                                    <Box
                                        h="80px"
                                        bg={isSelected ? 'blue.50' : 'gray.50'}
                                        border="1px solid"
                                        borderColor={isSelected ? 'blue.200' : 'gray.100'}
                                        borderRadius="md"
                                        _hover={{ bg: 'blue.50', borderColor: 'blue.300' }}
                                        onClick={() => onSlotClick(day, hour, null)} // Click empty space to add
                                        transition="all 0.2s"
                                        position="relative"
                                        p={1}
                                        cursor="pointer"
                                        overflowY="auto"
                                        className="custom-scrollbar"
                                    >
                                        <VStack spacing={1} align="stretch" h="100%">
                                            {classesInSlot.map((clase, idx) => (
                                                <Tooltip key={`${clase.id}-${idx}`} label={`${getMateriaName(clase.id_materia)} con ${getProfesorName(clase.id_profesor)}`} hasArrow>
                                                    <Box
                                                        bg="white"
                                                        p={1}
                                                        borderRadius="sm"
                                                        boxShadow="sm"
                                                        borderLeft="3px solid"
                                                        borderLeftColor="blue.400"
                                                        fontSize="xs"
                                                        cursor="pointer"
                                                        onClick={(e) => {
                                                            e.stopPropagation(); // Prevent triggering "Add"
                                                            onSlotClick(day, hour, clase);
                                                        }}
                                                        _hover={{ bg: 'blue.50' }}
                                                    >
                                                        <HStack spacing={1}>
                                                            <Avatar size="xs" src={mockProfesores.find(p => p.id === clase.id_profesor)?.foto_url} name={getProfesorName(clase.id_profesor)} />
                                                            <Text fontWeight="bold" noOfLines={1} fontSize="10px">
                                                                {getProfesorName(clase.id_profesor)}
                                                            </Text>
                                                        </HStack>
                                                        <Text fontSize="9px" color="gray.500" noOfLines={1} ml={5}>
                                                            {getMateriaName(clase.id_materia)}
                                                        </Text>
                                                    </Box>
                                                </Tooltip>
                                            ))}
                                        </VStack>

                                        {/* Show "Add" hint on hover if empty? Or just rely on cursor */}
                                    </Box>
                                </GridItem>
                            );
                        })}
                    </React.Fragment>
                ))}
            </Grid>
        </Box>
    );
};

export default WeeklyGrid;
