import { query, getClient } from './database.js';

export const ExamModel = {
    async getAll(institutionId = null) {
        try {
            // Verificar qué campos existen en la tabla users
            let checkColumnsSql = `
                SELECT column_name 
                FROM information_schema.columns 
                WHERE table_schema = 'public' 
                AND table_name = 'users'
            `;
            
            let columnCheck;
            let availableColumns = [];
            try {
                columnCheck = await query(checkColumnsSql);
                availableColumns = columnCheck.rows.map(row => row.column_name);
            } catch (error) {
                console.error('Error checking columns:', error);
                availableColumns = [];
            }
            
            const hasNewFields = availableColumns.includes('first_name') && availableColumns.includes('last_name');
            const hasOldField = availableColumns.includes('name');
            
            let professorNameSql;
            if (hasNewFields) {
                professorNameSql = `TRIM(CONCAT(
                    COALESCE(u.first_name, ''), 
                    ' ', 
                    COALESCE(u.last_name, ''), 
                    CASE 
                        WHEN u.second_last_name IS NOT NULL AND TRIM(u.second_last_name) != '' 
                        THEN CONCAT(' ', u.second_last_name) 
                        ELSE '' 
                    END
                ))`;
            } else if (hasOldField) {
                professorNameSql = `COALESCE(TRIM(u.name), 'Sin nombre')`;
            } else {
                professorNameSql = `COALESCE(u.email, 'Sin nombre')`;
            }
            
            let sql = `
                SELECT e.*, 
                       e.id_exam as id, 
                       ${professorNameSql} as professor_name 
                FROM exams e
                LEFT JOIN users u ON e.professor_id = u.id
            `;
            const params = [];
            if (institutionId) {
                sql += ' WHERE e.institution_id = $1';
                params.push(institutionId);
            }
            sql += ' ORDER BY e.created_at DESC';
            const res = await query(sql, params);
            return res.rows;
        } catch (error) {
            console.error('ExamModel.getAll Error:', error);
            throw error;
        }
    },

    async getById(id, studentId = null) {
        const res = await query('SELECT *, id_exam as id FROM exams WHERE id_exam = $1', [id]);
        if (!res.rows[0]) return null;
        
        const exam = res.rows[0];
        
        // Si se provee studentId, adjuntar info de su asignación específica (la más reciente)
        if (studentId) {
            const assignRes = await query(
                `SELECT * FROM exam_assignments 
                 WHERE exam_id = $1 AND student_id = $2 
                 ORDER BY attempt_number DESC LIMIT 1`,
                [id, studentId]
            );
            exam.assignment = assignRes.rows[0] || null;
        }

        // Fetch questions for this exam
        const questionsRes = await query(
            `SELECT q.* FROM question_bank q
             JOIN exam_questions eq ON q.id_question = eq.question_id
             WHERE eq.exam_id = $1
             ORDER BY eq.order_index ASC`,
            [id]
        );
        exam.questions = questionsRes.rows;
        return exam;
    },

    async create(exam) {
        const { institution_id, professor_id, title, description, settings, questions: questionIds } = exam;
        // scheduled_at y scheduled_end_at NO van en exams, van en exam_assignments
        
        const client = await getClient();
        try {
            await client.query('BEGIN');
            const res = await client.query(
                `INSERT INTO exams (institution_id, professor_id, title, description, settings)
                 VALUES ($1, $2, $3, $4, $5) RETURNING *`,
                [institution_id, professor_id, title, description, settings]
            );
            const newExam = res.rows[0];
            const examId = newExam.id_exam;

            // Insert questions if any
            if (questionIds && Array.isArray(questionIds) && questionIds.length > 0) {
                for (let i = 0; i < questionIds.length; i++) {
                    await client.query(
                        `INSERT INTO exam_questions (exam_id, question_id, order_index)
                         VALUES ($1, $2, $3)`,
                        [examId, questionIds[i], i]
                    );
                }
            }

            await client.query('COMMIT');
            return newExam;
        } catch (error) {
            await client.query('ROLLBACK');
            throw error;
        } finally {
            client.release();
        }
    },

    async update(id, exam) {
        const { questions: questionIds, ...examData } = exam;
        const fields = [];
        const values = [];
        let idx = 1;

        const client = await getClient();
        try {
            await client.query('BEGIN');

            // Sync questions if provided
            if (questionIds && Array.isArray(questionIds)) {
                // Delete existing ones
                await client.query('DELETE FROM exam_questions WHERE exam_id = $1', [id]);
                // Insert new ones
                for (let i = 0; i < questionIds.length; i++) {
                    await client.query(
                        `INSERT INTO exam_questions (exam_id, question_id, order_index)
                         VALUES ($1, $2, $3)`,
                        [id, questionIds[i], i]
                    );
                }
            }

            // Update main exam data
            // Campos que NO deben actualizarse (son calculados o no existen en la tabla)
            const excludedFields = ['updated_at', 'created_at', 'id_exam', 'id', 'professor_name', 'scheduled_at', 'scheduled_end_at'];
            
            for (const [key, value] of Object.entries(examData)) {
                if (!excludedFields.includes(key)) {
                    fields.push(`${key} = $${idx}`);
                    values.push(value);
                    idx++;
                }
            }

            if (fields.length > 0) {
                values.push(id);
                await client.query(
                    `UPDATE exams SET ${fields.join(', ')}, updated_at = CURRENT_TIMESTAMP WHERE id_exam = $${idx}`,
                    values
                );
            }

            await client.query('COMMIT');
            return this.getById(id);
        } catch (error) {
            await client.query('ROLLBACK');
            throw error;
        } finally {
            client.release();
        }
    },

    async delete(id) {
        await query('DELETE FROM exams WHERE id_exam = $1', [id]);
        return true;
    },

    // Assignments
    async assignToStudent(examId, studentId, scheduledAt = null, scheduledEndAt = null) {
        // Solo validar conflictos si es una asignación de fecha específica (no rango)
        if (scheduledAt && !scheduledEndAt) {
            // 1. Obtener información del examen (duración y tolerancia)
            const examRes = await query('SELECT settings, tolerance_minutes FROM exams WHERE id_exam = $1', [examId]);
            if (examRes.rows.length === 0) {
                throw new Error('Examen no encontrado');
            }
            
            const exam = examRes.rows[0];
            const settings = typeof exam.settings === 'string' ? JSON.parse(exam.settings) : exam.settings;
            const timerMinutes = parseInt(settings?.timer_minutes) || 60;
            const toleranceMinutes = parseInt(exam.tolerance_minutes) || 15;
            const totalMinutes = timerMinutes + toleranceMinutes;
            
            // 2. Calcular el rango de tiempo del examen (inicio + duración + tolerancia)
            const startTime = new Date(scheduledAt);
            const endTime = new Date(startTime.getTime() + totalMinutes * 60000);
            
            // 3. Buscar conflictos con otras asignaciones del mismo estudiante
            // Solo considerar asignaciones con fecha específica (scheduled_end_at IS NULL)
            // y que estén en estado PENDING o IN_PROGRESS
            const conflictRes = await query(
                `SELECT ea.*, e.title as exam_title, e.settings, e.tolerance_minutes
                 FROM exam_assignments ea
                 JOIN exams e ON ea.exam_id = e.id_exam
                 WHERE ea.student_id = $1
                   AND ea.scheduled_at IS NOT NULL
                   AND ea.scheduled_end_at IS NULL
                   AND ea.status IN ('PENDING', 'IN_PROGRESS')
                   AND ea.exam_id != $2`,
                [studentId, examId]
            );
            
            // Verificar conflictos calculando los rangos de tiempo
            for (const conflict of conflictRes.rows) {
                const conflictStart = new Date(conflict.scheduled_at);
                const conflictSettings = typeof conflict.settings === 'string' ? JSON.parse(conflict.settings) : conflict.settings;
                const conflictTimer = parseInt(conflictSettings?.timer_minutes) || 60;
                const conflictTolerance = parseInt(conflict.tolerance_minutes) || 15;
                const conflictEnd = new Date(conflictStart.getTime() + (conflictTimer + conflictTolerance) * 60000);
                
                // Verificar si hay solapamiento de horarios
                // Dos rangos se solapan si: start1 < end2 && start2 < end1
                if (startTime < conflictEnd && conflictStart < endTime) {
                    throw new Error(
                        `No se puede asignar este examen. El estudiante ya tiene el examen "${conflict.exam_title}" ` +
                        `programado de ${conflictStart.toLocaleString()} a ${conflictEnd.toLocaleString()}, ` +
                        `lo cual se solapa con el horario propuesto (${startTime.toLocaleString()} a ${endTime.toLocaleString()}).`
                    );
                }
            }
        }
        
        // 4. Verificar si ya existe una asignación PENDING o IN_PROGRESS para este examen
        const existingRes = await query(
            `SELECT id_assignment, attempt_number FROM exam_assignments 
             WHERE exam_id = $1 AND student_id = $2 AND status IN ('PENDING', 'IN_PROGRESS')
             ORDER BY attempt_number DESC LIMIT 1`,
            [examId, studentId]
        );

        if (existingRes.rows.length > 0) {
            // Actualizar la existente en lugar de crear un nuevo intento
            const assignmentId = existingRes.rows[0].id_assignment;
            const res = await query(
                `UPDATE exam_assignments 
                 SET scheduled_at = $1, scheduled_end_at = $2, status = 'PENDING', started_at = NULL, completed_at = NULL, score = NULL
                 WHERE id_assignment = $3 RETURNING *`,
                [scheduledAt, scheduledEndAt, assignmentId]
            );
            return res.rows[0];
        }

        // 5. Si no hay pendientes, crear un nuevo intento
        const attemptRes = await query(
            `SELECT COALESCE(MAX(attempt_number), 0) as last_attempt 
             FROM exam_assignments WHERE exam_id = $1 AND student_id = $2`,
            [examId, studentId]
        );
        const nextAttempt = attemptRes.rows[0].last_attempt + 1;

        // 6. Insert new assignment
        const res = await query(
            `INSERT INTO exam_assignments (exam_id, student_id, scheduled_at, scheduled_end_at, attempt_number) 
             VALUES ($1, $2, $3, $4, $5) RETURNING *`,
            [examId, studentId, scheduledAt, scheduledEndAt, nextAttempt]
        );
        return res.rows[0];
    },

    async getAssignments(examId) {
        try {
            // Verificar qué campos existen en la tabla users
            let checkColumnsSql = `
                SELECT column_name 
                FROM information_schema.columns 
                WHERE table_schema = 'public' 
                AND table_name = 'users'
            `;
            
            let columnCheck;
            let availableColumns = [];
            try {
                columnCheck = await query(checkColumnsSql);
                availableColumns = columnCheck.rows.map(row => row.column_name);
            } catch (error) {
                console.error('Error checking columns:', error);
                availableColumns = [];
            }
            
            const hasNewFields = availableColumns.includes('first_name') && availableColumns.includes('last_name');
            const hasOldField = availableColumns.includes('name');
            
            let studentNameSql;
            if (hasNewFields) {
                studentNameSql = `TRIM(CONCAT(
                    COALESCE(u.first_name, ''), 
                    ' ', 
                    COALESCE(u.last_name, ''), 
                    CASE 
                        WHEN u.second_last_name IS NOT NULL AND TRIM(u.second_last_name) != '' 
                        THEN CONCAT(' ', u.second_last_name) 
                        ELSE '' 
                    END
                ))`;
            } else if (hasOldField) {
                studentNameSql = `COALESCE(TRIM(u.name), 'Sin nombre')`;
            } else {
                studentNameSql = `COALESCE(u.email, 'Sin nombre')`;
            }
            
            const res = await query(
                `SELECT ea.*, 
                        u.email as student_email, 
                        ${studentNameSql} as student_name
                 FROM exam_assignments ea 
                 JOIN users u ON ea.student_id = u.id 
                 WHERE ea.exam_id = $1`,
                [examId]
            );
            return res.rows;
        } catch (error) {
            console.error('ExamModel.getAssignments Error:', error);
            throw error;
        }
    },

    async unassignStudent(examId, studentId) {
        await query(
            `DELETE FROM exam_assignments 
             WHERE exam_id = $1 AND student_id = $2`,
            [examId, studentId]
        );
        return true;
    },

    async getAssignment(examId, studentId) {
        const res = await query(
            `SELECT * FROM exam_assignments 
             WHERE exam_id = $1 AND student_id = $2
             ORDER BY attempt_number DESC LIMIT 1`,
            [examId, studentId]
        );
        return res.rows[0];
    },

    async updateAssignment(examId, studentId, data) {
        // Encontrar el intento más reciente que no esté completado, o el último en general
        const currentRes = await query(
            `SELECT id_assignment FROM exam_assignments 
             WHERE exam_id = $1 AND student_id = $2 
             ORDER BY attempt_number DESC LIMIT 1`,
            [examId, studentId]
        );
        
        if (!currentRes.rows[0]) return null;
        const id_assignment = currentRes.rows[0].id_assignment;

        const fields = [];
        const values = [];
        let idx = 1;

        for (const [key, value] of Object.entries(data)) {
            fields.push(`${key} = $${idx}`);
            values.push(value);
            idx++;
        }

        if (fields.length === 0) return null;

        values.push(id_assignment);
        const res = await query(
            `UPDATE exam_assignments 
             SET ${fields.join(', ')} 
             WHERE id_assignment = $${idx} 
             RETURNING *`,
            values
        );
        return res.rows[0];
    }
};

