import { query } from './database.js';

export const InstitutionModel = {
    async getAll() {
        try {
            // Verificar qué tabla y columnas existen
            let tableName = 'institutions';
            let idColumn = 'id_institution';
            let maxStudentsColumn = 'max_students';
            let maxTeachersColumn = 'max_teachers';
            
            try {
                // Verificar qué tabla existe
                const testRes = await query(`
                    SELECT table_name 
                    FROM information_schema.tables 
                    WHERE table_schema = 'public' 
                    AND table_name IN ('institutions', 'companies')
                    LIMIT 1
                `);
                
                if (testRes.rows.length > 0) {
                    const foundTable = testRes.rows[0].table_name;
                    if (foundTable === 'companies') {
                        tableName = 'companies';
                        idColumn = 'id';
                    }
                }
                
                // Verificar qué columnas de límites existen
                const columnsRes = await query(`
                    SELECT column_name 
                    FROM information_schema.columns 
                    WHERE table_schema = 'public' 
                    AND table_name = $1
                    AND column_name IN ('max_students', 'max_clients', 'max_teachers', 'max_employees')
                `, [tableName]);
                
                const availableColumns = columnsRes.rows.map(row => row.column_name);
                
                // Determinar qué columnas usar
                if (availableColumns.includes('max_students')) {
                    maxStudentsColumn = 'max_students';
                } else if (availableColumns.includes('max_clients')) {
                    maxStudentsColumn = 'max_clients';
                } else {
                    maxStudentsColumn = null;
                }
                
                if (availableColumns.includes('max_teachers')) {
                    maxTeachersColumn = 'max_teachers';
                } else if (availableColumns.includes('max_employees')) {
                    maxTeachersColumn = 'max_employees';
                } else {
                    maxTeachersColumn = null;
                }
            } catch (checkError) {
                // Si falla la verificación, intentar con companies
                try {
                    await query('SELECT 1 FROM companies LIMIT 1');
                    tableName = 'companies';
                    idColumn = 'id';
                    // Intentar detectar columnas en companies
                    try {
                        const columnsRes = await query(`
                            SELECT column_name 
                            FROM information_schema.columns 
                            WHERE table_schema = 'public' 
                            AND table_name = 'companies'
                            AND column_name IN ('max_students', 'max_clients', 'max_teachers', 'max_employees')
                        `);
                        const availableColumns = columnsRes.rows.map(row => row.column_name);
                        if (availableColumns.includes('max_clients')) {
                            maxStudentsColumn = 'max_clients';
                        } else if (availableColumns.includes('max_students')) {
                            maxStudentsColumn = 'max_students';
                        } else {
                            maxStudentsColumn = null;
                        }
                        if (availableColumns.includes('max_employees')) {
                            maxTeachersColumn = 'max_employees';
                        } else if (availableColumns.includes('max_teachers')) {
                            maxTeachersColumn = 'max_teachers';
                        } else {
                            maxTeachersColumn = null;
                        }
                    } catch (err) {
                        // Usar valores por defecto
                    }
                } catch (err) {
                    // Si también falla, usar institutions por defecto
                }
            }
            
            // Construir SELECT dinámicamente
            let selectColumns = `*, ${idColumn} as id, ${idColumn} as company_id, ${idColumn} as id_institution`;
            
            if (maxStudentsColumn) {
                selectColumns += `, ${maxStudentsColumn} as max_clients, ${maxStudentsColumn} as max_students`;
            } else {
                selectColumns += `, 0 as max_clients, 0 as max_students`;
            }
            
            if (maxTeachersColumn) {
                selectColumns += `, ${maxTeachersColumn} as max_employees, ${maxTeachersColumn} as max_teachers`;
            } else {
                selectColumns += `, 0 as max_employees, 0 as max_teachers`;
            }
            
            selectColumns += `, name as commercial_name`;
            
            const res = await query(`
                SELECT ${selectColumns}
                FROM ${tableName} 
                ORDER BY created_at DESC
            `);
            console.log('InstitutionModel.getAll: Found', res.rows.length, 'institutions/companies');
            return res.rows;
        } catch (error) {
            console.error('InstitutionModel.getAll Error:', error.message);
            // Intentar con companies si institutions falla
            if (error.message.includes('institutions') && !error.message.includes('companies')) {
                try {
                    // Intentar con columnas legacy
                    const res = await query(`
                        SELECT *, 
                        id as id_institution,
                        id as id, 
                        id as company_id,
                        COALESCE(max_clients, 0) as max_clients,
                        COALESCE(max_clients, 0) as max_students,
                        COALESCE(max_employees, 0) as max_employees,
                        COALESCE(max_employees, 0) as max_teachers,
                        name as commercial_name
                        FROM companies 
                        ORDER BY created_at DESC
                    `);
                    return res.rows;
                } catch (fallbackError) {
                    // Último intento: solo columnas básicas
                    try {
                        const res = await query(`
                            SELECT *, 
                            id as id_institution,
                            id as id, 
                            id as company_id,
                            0 as max_clients,
                            0 as max_students,
                            0 as max_employees,
                            0 as max_teachers,
                            name as commercial_name
                            FROM companies 
                            ORDER BY created_at DESC
                        `);
                        return res.rows;
                    } catch (finalError) {
                        throw error; // Lanzar el error original
                    }
                }
            }
            throw error;
        }
    },

    async getById(id) {
        try {
            const res = await query(`
                SELECT *, 
                id_institution as id, 
                id_institution as company_id,
                COALESCE(max_students, 0) as max_clients,
                COALESCE(max_students, 0) as max_students,
                COALESCE(max_teachers, 0) as max_employees,
                COALESCE(max_teachers, 0) as max_teachers,
                name as commercial_name
                FROM institutions 
                WHERE id_institution = $1
            `, [id]);
            return res.rows[0];
        } catch (error) {
            // Si falla con institutions, intentar con companies
            if (error.message.includes('institutions') || error.message.includes('max_students') || error.message.includes('max_teachers')) {
                try {
                    const res = await query(`
                        SELECT *, 
                        id as id_institution,
                        id as id, 
                        id as company_id,
                        COALESCE(max_clients, 0) as max_clients,
                        COALESCE(max_clients, 0) as max_students,
                        COALESCE(max_employees, 0) as max_employees,
                        COALESCE(max_employees, 0) as max_teachers,
                        name as commercial_name
                        FROM companies 
                        WHERE id = $1
                    `, [id]);
                    return res.rows[0];
                } catch (fallbackError) {
                    // Último intento: solo columnas básicas
                    const res = await query(`
                        SELECT *, 
                        id as id_institution,
                        id as id, 
                        id as company_id,
                        0 as max_clients,
                        0 as max_students,
                        0 as max_employees,
                        0 as max_teachers,
                        name as commercial_name
                        FROM companies 
                        WHERE id = $1
                    `, [id]);
                    return res.rows[0];
                }
            }
            throw error;
        }
    },

    async create(institution) {
        const { 
            name, rfc, phone, email, fiscal_address, 
            logo_url, tagline, exam_footer, executive_contact,
            street, exterior_number, neighborhood, municipality, state, postal_code,
            max_teachers, max_students, max_exams, is_active, settings,
            max_employees, max_clients // legacy fields
        } = institution;
        
        const finalMaxTeachers = max_teachers !== undefined ? max_teachers : (max_employees || 0);
        const finalMaxStudents = max_students !== undefined ? max_students : (max_clients || 0);

        try {
            const res = await query(
                `INSERT INTO institutions (
                    name, rfc, phone, email, fiscal_address, 
                    logo_url, tagline, exam_footer, executive_contact,
                    street, exterior_number, neighborhood, municipality, state, postal_code,
                    max_teachers, max_students, max_exams, is_active, settings
                ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, $15, $16, $17, $18, $19, $20)
                RETURNING *, id_institution as id`,
                [
                    name, rfc, phone, email, fiscal_address, 
                    logo_url, tagline, exam_footer, executive_contact,
                    street, exterior_number, neighborhood, municipality, state, postal_code,
                    finalMaxTeachers, finalMaxStudents, max_exams || 0, 
                    is_active !== undefined ? is_active : true, settings
                ]
            );
            return res.rows[0];
        } catch (error) {
            // Si falla con institutions, intentar con companies
            if (error.message.includes('institutions') || error.message.includes('max_students') || error.message.includes('max_teachers')) {
                try {
                    // Intentar con columnas legacy (max_clients, max_employees)
                    const res = await query(
                        `INSERT INTO companies (
                            name, rfc, phone, email, fiscal_address, 
                            logo_url, tagline, exam_footer, executive_contact,
                            street, exterior_number, neighborhood, municipality, state, postal_code,
                            max_employees, max_clients, max_exams, is_active, settings
                        ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, $15, $16, $17, $18, $19, $20)
                        RETURNING *, id as id_institution, id as id`,
                        [
                            name, rfc, phone, email, fiscal_address, 
                            logo_url, tagline, exam_footer, executive_contact,
                            street, exterior_number, neighborhood, municipality, state, postal_code,
                            finalMaxTeachers, finalMaxStudents, max_exams || 0, 
                            is_active !== undefined ? is_active : true, settings
                        ]
                    );
                    return res.rows[0];
                } catch (fallbackError) {
                    // Si también falla, intentar sin columnas de límites
                    try {
                        const res = await query(
                            `INSERT INTO companies (
                                name, rfc, phone, email, fiscal_address, 
                                logo_url, tagline, exam_footer, executive_contact,
                                street, exterior_number, neighborhood, municipality, state, postal_code,
                                is_active, settings
                            ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, $15, $16, $17)
                            RETURNING *, id as id_institution, id as id`,
                            [
                                name, rfc, phone, email, fiscal_address, 
                                logo_url, tagline, exam_footer, executive_contact,
                                street, exterior_number, neighborhood, municipality, state, postal_code,
                                is_active !== undefined ? is_active : true, settings
                            ]
                        );
                        return res.rows[0];
                    } catch (finalError) {
                        throw error; // Lanzar el error original
                    }
                }
            }
            throw error;
        }
    },

    async update(id, institution) {
        const fields = [];
        const values = [];
        let idx = 1;

        // Mapeo de campos legacy a nuevos campos
        const mappedInstitution = { ...institution };
        
        // Mapear max_employees/max_clients a max_teachers/max_students si vienen
        if (institution.max_employees !== undefined) {
            mappedInstitution.max_teachers = institution.max_employees;
        }
        if (institution.max_clients !== undefined) {
            mappedInstitution.max_students = institution.max_clients;
        }
        
        // Eliminar campos legacy y virtuales para que no interfieran con la actualización
        delete mappedInstitution.max_employees;
        delete mappedInstitution.max_clients;
        delete mappedInstitution.id;
        delete mappedInstitution.company_id;
        delete mappedInstitution.commercial_name;
        
        // YA NO eliminamos los campos de dirección ya que ahora existen en la base de datos

        console.log('InstitutionModel.update - Cleaned Data:', Object.keys(mappedInstitution));

        // Separar campos de límites para manejar según la tabla
        const limitFields = {};
        if (mappedInstitution.max_teachers !== undefined) {
            limitFields.max_teachers = mappedInstitution.max_teachers;
            delete mappedInstitution.max_teachers;
        }
        if (mappedInstitution.max_students !== undefined) {
            limitFields.max_students = mappedInstitution.max_students;
            delete mappedInstitution.max_students;
        }

        for (const [key, value] of Object.entries(mappedInstitution)) {
            if (key !== 'updated_at' && key !== 'created_at' && key !== 'id_institution') {
                fields.push(`${key} = $${idx}`);
                values.push(value);
                idx++;
            }
        }
        values.push(id);

        try {
            // Agregar campos de límites si existen
            let limitFieldsSql = '';
            if (limitFields.max_teachers !== undefined) {
                limitFieldsSql += `, max_teachers = $${idx++}`;
                values.splice(values.length - 1, 0, limitFields.max_teachers);
            }
            if (limitFields.max_students !== undefined) {
                limitFieldsSql += `, max_students = $${idx++}`;
                values.splice(values.length - 1, 0, limitFields.max_students);
            }
            
            const res = await query(
                `UPDATE institutions SET ${fields.join(', ')}${limitFieldsSql}, updated_at = CURRENT_TIMESTAMP WHERE id_institution = $${idx} RETURNING *, id_institution as id`,
                values
            );
            return res.rows[0];
        } catch (error) {
            // Si falla con institutions, intentar con companies
            if (error.message.includes('institutions') || error.message.includes('max_students') || error.message.includes('max_teachers')) {
                // Reconstruir fields para companies (usar max_employees y max_clients)
                const companiesFields = [];
                const companiesValues = [];
                let companiesIdx = 1;
                
                for (const [key, value] of Object.entries(mappedInstitution)) {
                    if (key !== 'updated_at' && key !== 'created_at' && key !== 'id_institution') {
                        companiesFields.push(`${key} = $${companiesIdx++}`);
                        companiesValues.push(value);
                    }
                }
                
                // Agregar campos de límites con nombres legacy
                if (limitFields.max_teachers !== undefined) {
                    companiesFields.push(`max_employees = $${companiesIdx++}`);
                    companiesValues.push(limitFields.max_teachers);
                }
                if (limitFields.max_students !== undefined) {
                    companiesFields.push(`max_clients = $${companiesIdx++}`);
                    companiesValues.push(limitFields.max_students);
                }
                
                companiesValues.push(id);
                
                try {
                    const res = await query(
                        `UPDATE companies SET ${companiesFields.join(', ')}, updated_at = CURRENT_TIMESTAMP WHERE id = $${companiesIdx} RETURNING *, id as id_institution, id as id`,
                        companiesValues
                    );
                    return res.rows[0];
                } catch (fallbackError) {
                    // Si falla, intentar sin campos de límites
                    const basicFields = companiesFields.filter(f => !f.includes('max_'));
                    const basicValues = companiesValues.slice(0, -1).filter((v, i) => !companiesFields[i].includes('max_'));
                    basicValues.push(id);
                    
                    try {
                        const res = await query(
                            `UPDATE companies SET ${basicFields.join(', ')}, updated_at = CURRENT_TIMESTAMP WHERE id = $${basicFields.length + 1} RETURNING *, id as id_institution, id as id`,
                            basicValues
                        );
                        return res.rows[0];
                    } catch (finalError) {
                        throw error; // Lanzar el error original
                    }
                }
            }
            throw error;
        }
    },

    async delete(id) {
        try {
            await query('DELETE FROM institutions WHERE id_institution = $1', [id]);
            return true;
        } catch (error) {
            // Si falla con institutions, intentar con companies
            if (error.message.includes('institutions')) {
                await query('DELETE FROM companies WHERE id = $1', [id]);
                return true;
            }
            throw error;
        }
    }
};

